#!/usr/bin/env node
/**
 * Monitor Script for telebiz-mcp
 *
 * Checks health and maintains state file for status tracking.
 * Outputs alerts when state changes (for integration with notifications).
 *
 * Usage:
 *   node monitor.js              # Check and output status
 *   node monitor.js --json       # Output JSON
 *   node monitor.js --quiet      # Only output on state change
 *
 * Exit codes:
 *   0 = healthy (relay up, executor connected)
 *   1 = degraded (relay up, executor disconnected)
 *   2 = down (relay not running)
 *   3 = state changed (for alerting)
 */

import * as fs from 'fs';
import * as path from 'path';
import { checkHealth, type HealthStatus } from './health.js';

const STATE_FILE = process.env.TELEBIZ_STATE_FILE ||
  path.join(process.env.HOME || '/tmp', '.telebiz-mcp-state.json');

interface MonitorState {
  lastStatus: HealthStatus | null;
  lastChange: number | null;
  consecutiveFailures: number;
  lastAlert: number | null;
}

function loadState(): MonitorState {
  try {
    if (fs.existsSync(STATE_FILE)) {
      return JSON.parse(fs.readFileSync(STATE_FILE, 'utf-8'));
    }
  } catch (e) {
    // Ignore
  }
  return {
    lastStatus: null,
    lastChange: null,
    consecutiveFailures: 0,
    lastAlert: null,
  };
}

function saveState(state: MonitorState): void {
  try {
    fs.writeFileSync(STATE_FILE, JSON.stringify(state, null, 2));
  } catch (e) {
    console.error('Failed to save state:', e);
  }
}

function statusChanged(a: HealthStatus | null, b: HealthStatus): boolean {
  if (!a) return true;
  return a.relay !== b.relay || a.executor !== b.executor;
}

function getStatusEmoji(status: HealthStatus): string {
  if (status.relay === 'down') return '🔴';
  if (status.executor !== 'connected') return '🟡';
  return '🟢';
}

function formatAlert(status: HealthStatus, state: MonitorState): string {
  const emoji = getStatusEmoji(status);
  const lines: string[] = [];

  if (status.relay === 'down') {
    lines.push(`${emoji} **Telebiz MCP Down**`);
    lines.push('Relay server is not running.');
    lines.push('');
    lines.push('To fix: `~/clawd/skills/telebiz-mcp/start-relay.sh`');
  } else if (status.executor !== 'connected') {
    lines.push(`${emoji} **Telebiz MCP Degraded**`);
    lines.push('Relay running but browser not connected.');
    lines.push('');
    lines.push('To fix: Open telebiz-tt in browser and enable MCP.');
  } else {
    lines.push(`${emoji} **Telebiz MCP Healthy**`);
    lines.push('All systems operational.');
  }

  if (state.consecutiveFailures > 1) {
    lines.push(`(${state.consecutiveFailures} consecutive failures)`);
  }

  if (status.error) {
    lines.push(`Error: ${status.error}`);
  }

  return lines.join('\n');
}

async function main(): Promise<void> {
  const args = process.argv.slice(2);
  const jsonOutput = args.includes('--json');
  const quietMode = args.includes('--quiet');

  const state = loadState();
  const status = await checkHealth();

  // Track state changes
  const changed = statusChanged(state.lastStatus, status);
  const wasHealthy = state.lastStatus?.relay === 'up' && state.lastStatus?.executor === 'connected';
  const isHealthy = status.relay === 'up' && status.executor === 'connected';

  // Update failure counter
  if (!isHealthy) {
    state.consecutiveFailures++;
  } else {
    state.consecutiveFailures = 0;
  }

  // Update state
  if (changed) {
    state.lastChange = Date.now();
  }
  state.lastStatus = status;
  saveState(state);

  // Determine if we should alert
  // Alert on: state change, or first run, or recovery
  const shouldAlert = changed || (wasHealthy === false && isHealthy);

  // Output
  if (jsonOutput) {
    console.log(JSON.stringify({
      ...status,
      changed,
      shouldAlert,
      consecutiveFailures: state.consecutiveFailures,
      lastChange: state.lastChange,
    }, null, 2));
  } else if (!quietMode || shouldAlert) {
    console.log(formatAlert(status, state));
  }

  // Exit codes
  if (shouldAlert && changed) {
    process.exit(3); // State changed
  } else if (status.relay === 'down') {
    process.exit(2); // Down
  } else if (status.executor !== 'connected') {
    process.exit(1); // Degraded
  } else {
    process.exit(0); // Healthy
  }
}

main().catch((error) => {
  console.error('Monitor error:', error);
  process.exit(2);
});
