#!/usr/bin/env node
/**
 * MCP Server for telebiz-tt
 *
 * Connects to the local WebSocket relay and exposes telebiz tools
 * via the MCP protocol (stdio transport).
 *
 * Usage: node mcp-server.js
 * Configure in mcporter: { "command": "node", "args": ["path/to/mcp-server.js"] }
 */
import { Server } from '@modelcontextprotocol/sdk/server/index.js';
import { StdioServerTransport } from '@modelcontextprotocol/sdk/server/stdio.js';
import { CallToolRequestSchema, ListToolsRequestSchema, } from '@modelcontextprotocol/sdk/types.js';
import WebSocket from 'ws';
const RELAY_URL = process.env.TELEBIZ_RELAY_URL || 'ws://localhost:9716';
const RECONNECT_INTERVAL = 5000;
// Tool definitions (matching telebiz-tt registry)
const TOOLS = [
    // Chat Tools
    {
        name: 'listChats',
        description: 'Get a list of chats with optional filters. Returns chat IDs, titles, types, and last message info.',
        inputSchema: {
            type: 'object',
            properties: {
                chatType: {
                    type: 'string',
                    enum: ['private', 'group', 'supergroup', 'channel', 'all'],
                    description: 'Filter by chat type. Default: all',
                },
                hasUnread: {
                    type: 'boolean',
                    description: 'true = only unread chats, false = only read chats',
                },
                isArchived: {
                    type: 'boolean',
                    description: 'true = archived only, false = non-archived only',
                },
                lastMessageOlderThanDays: {
                    type: 'number',
                    description: 'Chats where last message is OLDER than N days',
                },
                lastMessageNewerThanDays: {
                    type: 'number',
                    description: 'Chats where last message is NEWER than N days',
                },
                iAmLastSender: {
                    type: 'boolean',
                    description: 'true = I sent last message, false = they sent last message',
                },
                titleContains: {
                    type: 'string',
                    description: 'Filter by chat title containing this text (case-insensitive)',
                },
                folderId: {
                    type: 'number',
                    description: 'Filter to chats in a specific folder ID',
                },
                limit: {
                    type: 'number',
                    description: 'Max chats to return (default: 50, max: 200)',
                },
            },
        },
    },
    {
        name: 'getChatInfo',
        description: 'Get detailed information about a specific chat including members count, description, and settings.',
        inputSchema: {
            type: 'object',
            properties: {
                chatId: {
                    type: 'string',
                    description: 'The ID of the chat to get info for',
                },
            },
            required: ['chatId'],
        },
    },
    {
        name: 'getCurrentChat',
        description: 'Get info about the currently open chat in the browser.',
        inputSchema: {
            type: 'object',
            properties: {},
        },
    },
    {
        name: 'openChat',
        description: 'Navigate to and open a specific chat in the browser UI.',
        inputSchema: {
            type: 'object',
            properties: {
                chatId: {
                    type: 'string',
                    description: 'The ID of the chat to open',
                },
            },
            required: ['chatId'],
        },
    },
    {
        name: 'archiveChat',
        description: 'Archive a chat to move it to the archived folder.',
        inputSchema: {
            type: 'object',
            properties: {
                chatId: {
                    type: 'string',
                    description: 'The ID of the chat to archive',
                },
            },
            required: ['chatId'],
        },
    },
    {
        name: 'unarchiveChat',
        description: 'Unarchive a chat to move it back to the main chat list.',
        inputSchema: {
            type: 'object',
            properties: {
                chatId: {
                    type: 'string',
                    description: 'The ID of the chat to unarchive',
                },
            },
            required: ['chatId'],
        },
    },
    {
        name: 'pinChat',
        description: 'Pin a chat to keep it at the top of the chat list.',
        inputSchema: {
            type: 'object',
            properties: {
                chatId: {
                    type: 'string',
                    description: 'The ID of the chat to pin',
                },
            },
            required: ['chatId'],
        },
    },
    {
        name: 'unpinChat',
        description: 'Unpin a chat from the top of the chat list.',
        inputSchema: {
            type: 'object',
            properties: {
                chatId: {
                    type: 'string',
                    description: 'The ID of the chat to unpin',
                },
            },
            required: ['chatId'],
        },
    },
    {
        name: 'muteChat',
        description: 'Mute notifications for a chat.',
        inputSchema: {
            type: 'object',
            properties: {
                chatId: {
                    type: 'string',
                    description: 'The ID of the chat to mute',
                },
                muteUntil: {
                    type: 'number',
                    description: 'Unix timestamp until when to mute. Use 0 for forever.',
                },
            },
            required: ['chatId'],
        },
    },
    {
        name: 'unmuteChat',
        description: 'Unmute notifications for a chat.',
        inputSchema: {
            type: 'object',
            properties: {
                chatId: {
                    type: 'string',
                    description: 'The ID of the chat to unmute',
                },
            },
            required: ['chatId'],
        },
    },
    {
        name: 'deleteChat',
        description: 'Delete a chat or leave a group/channel. WARNING: Destructive action.',
        inputSchema: {
            type: 'object',
            properties: {
                chatId: {
                    type: 'string',
                    description: 'The ID of the chat to delete/leave',
                },
            },
            required: ['chatId'],
        },
    },
    // Message Tools
    {
        name: 'sendMessage',
        description: 'Send a text message to a chat. Supports Telegram markdown: **bold**, __italic__, ~~strike~~, `code`, ||spoiler||, [text](url).',
        inputSchema: {
            type: 'object',
            properties: {
                chatId: {
                    type: 'string',
                    description: 'The ID of the chat to send the message to',
                },
                username: {
                    type: 'string',
                    description: 'Username (without @) for private chats. Required if chat not previously opened.',
                },
                text: {
                    type: 'string',
                    description: 'The message text. Use @username format for mentions.',
                },
                replyToMessageId: {
                    type: 'number',
                    description: 'Optional message ID to reply to',
                },
            },
            required: ['chatId', 'text'],
        },
    },
    {
        name: 'forwardMessages',
        description: 'Forward messages from one chat to another.',
        inputSchema: {
            type: 'object',
            properties: {
                fromChatId: {
                    type: 'string',
                    description: 'The ID of the chat to forward messages from',
                },
                toChatId: {
                    type: 'string',
                    description: 'The ID of the chat to forward messages to',
                },
                messageIds: {
                    type: 'array',
                    items: { type: 'number' },
                    description: 'Array of message IDs to forward',
                },
                withoutAuthor: {
                    type: 'boolean',
                    description: 'Forward without showing the original author',
                },
            },
            required: ['fromChatId', 'toChatId', 'messageIds'],
        },
    },
    {
        name: 'deleteMessages',
        description: 'Delete messages from a chat. WARNING: Destructive action.',
        inputSchema: {
            type: 'object',
            properties: {
                chatId: {
                    type: 'string',
                    description: 'The ID of the chat containing the messages',
                },
                messageIds: {
                    type: 'array',
                    items: { type: 'number' },
                    description: 'Array of message IDs to delete',
                },
                forEveryone: {
                    type: 'boolean',
                    description: 'Delete for all participants (if allowed)',
                },
            },
            required: ['chatId', 'messageIds'],
        },
    },
    {
        name: 'searchMessages',
        description: 'Search for messages by text. Can search globally or in a specific chat.',
        inputSchema: {
            type: 'object',
            properties: {
                query: {
                    type: 'string',
                    description: 'The search query text',
                },
                chatId: {
                    type: 'string',
                    description: 'Optional chat ID to search within. If omitted, searches globally.',
                },
                limit: {
                    type: 'number',
                    description: 'Maximum results (default: 50, max: 100)',
                },
            },
            required: ['query'],
        },
    },
    {
        name: 'getRecentMessages',
        description: 'Get recent messages from a chat. Use offset to paginate through older messages.',
        inputSchema: {
            type: 'object',
            properties: {
                chatId: {
                    type: 'string',
                    description: 'The ID of the chat to get messages from',
                },
                limit: {
                    type: 'number',
                    description: 'Maximum messages to return (default: 20, max: 100)',
                },
                offset: {
                    type: 'number',
                    description: 'Number of messages to skip from most recent (for pagination)',
                },
            },
            required: ['chatId'],
        },
    },
    {
        name: 'markChatAsRead',
        description: 'Mark all messages in a chat as read.',
        inputSchema: {
            type: 'object',
            properties: {
                chatId: {
                    type: 'string',
                    description: 'The ID of the chat to mark as read',
                },
            },
            required: ['chatId'],
        },
    },
    // Folder Tools
    {
        name: 'listFolders',
        description: 'Get all chat folders with their IDs, names, and included chat counts.',
        inputSchema: {
            type: 'object',
            properties: {},
        },
    },
    {
        name: 'createFolder',
        description: 'Create a new chat folder. Must provide either includedChatIds or chat type filters.',
        inputSchema: {
            type: 'object',
            properties: {
                title: {
                    type: 'string',
                    description: 'The name of the folder',
                },
                includedChatIds: {
                    type: 'array',
                    items: { type: 'string' },
                    description: 'Array of chat IDs to include',
                },
                excludedChatIds: {
                    type: 'array',
                    items: { type: 'string' },
                    description: 'Array of chat IDs to exclude',
                },
                includeContacts: { type: 'boolean', description: 'Include all contacts' },
                includeNonContacts: { type: 'boolean', description: 'Include non-contacts' },
                includeGroups: { type: 'boolean', description: 'Include groups' },
                includeChannels: { type: 'boolean', description: 'Include channels' },
                includeBots: { type: 'boolean', description: 'Include bots' },
            },
            required: ['title'],
        },
    },
    {
        name: 'addChatToFolder',
        description: 'Add a chat to one or more folders.',
        inputSchema: {
            type: 'object',
            properties: {
                chatId: {
                    type: 'string',
                    description: 'The ID of the chat to add',
                },
                folderIds: {
                    type: 'array',
                    items: { type: 'number' },
                    description: 'Array of folder IDs to add the chat to',
                },
            },
            required: ['chatId', 'folderIds'],
        },
    },
    {
        name: 'removeChatFromFolder',
        description: 'Remove a chat from one or more folders.',
        inputSchema: {
            type: 'object',
            properties: {
                chatId: {
                    type: 'string',
                    description: 'The ID of the chat to remove',
                },
                folderIds: {
                    type: 'array',
                    items: { type: 'number' },
                    description: 'Array of folder IDs to remove the chat from',
                },
            },
            required: ['chatId', 'folderIds'],
        },
    },
    {
        name: 'deleteFolder',
        description: 'Delete a chat folder. WARNING: Destructive action.',
        inputSchema: {
            type: 'object',
            properties: {
                folderId: {
                    type: 'number',
                    description: 'The ID of the folder to delete',
                },
            },
            required: ['folderId'],
        },
    },
    // Member Tools
    {
        name: 'getChatMembers',
        description: 'Get the list of members in a group or channel.',
        inputSchema: {
            type: 'object',
            properties: {
                chatId: {
                    type: 'string',
                    description: 'The ID of the chat to get members for',
                },
                filter: {
                    type: 'string',
                    enum: ['all', 'admins', 'kicked', 'restricted', 'bots', 'recent'],
                    description: 'Filter members by type',
                },
                limit: {
                    type: 'number',
                    description: 'Maximum members to return',
                },
            },
            required: ['chatId'],
        },
    },
    {
        name: 'addChatMembers',
        description: 'Add users to a group.',
        inputSchema: {
            type: 'object',
            properties: {
                chatId: {
                    type: 'string',
                    description: 'The ID of the group',
                },
                userIds: {
                    type: 'array',
                    items: { type: 'string' },
                    description: 'Array of user IDs to add',
                },
            },
            required: ['chatId', 'userIds'],
        },
    },
    {
        name: 'removeChatMember',
        description: 'Remove a user from a group.',
        inputSchema: {
            type: 'object',
            properties: {
                chatId: {
                    type: 'string',
                    description: 'The ID of the group',
                },
                userId: {
                    type: 'string',
                    description: 'The ID of the user to remove',
                },
            },
            required: ['chatId', 'userId'],
        },
    },
    {
        name: 'createGroup',
        description: 'Create a new group chat.',
        inputSchema: {
            type: 'object',
            properties: {
                title: {
                    type: 'string',
                    description: 'The name of the group',
                },
                memberIds: {
                    type: 'array',
                    items: { type: 'string' },
                    description: 'Array of user IDs to add as members',
                },
            },
            required: ['title', 'memberIds'],
        },
    },
    // User Tools
    {
        name: 'searchUsers',
        description: 'Search for users by name or username.',
        inputSchema: {
            type: 'object',
            properties: {
                query: {
                    type: 'string',
                    description: 'Search query',
                },
                limit: {
                    type: 'number',
                    description: 'Maximum results',
                },
            },
            required: ['query'],
        },
    },
    {
        name: 'getUserInfo',
        description: 'Get detailed information about a user.',
        inputSchema: {
            type: 'object',
            properties: {
                userId: {
                    type: 'string',
                    description: 'The ID of the user',
                },
            },
            required: ['userId'],
        },
    },
    // Batch Tools
    {
        name: 'batchSendMessage',
        description: 'Send the same message to multiple chats.',
        inputSchema: {
            type: 'object',
            properties: {
                chatIds: {
                    type: 'array',
                    items: { type: 'string' },
                    description: 'Array of chat IDs to send to',
                },
                text: {
                    type: 'string',
                    description: 'The message text',
                },
                usernames: {
                    type: 'array',
                    items: { type: 'string' },
                    description: 'Optional array of usernames (for private chats)',
                },
                delayMs: {
                    type: 'number',
                    description: 'Delay between sends in milliseconds',
                },
            },
            required: ['chatIds', 'text'],
        },
    },
    {
        name: 'batchAddToFolder',
        description: 'Add multiple chats to a folder.',
        inputSchema: {
            type: 'object',
            properties: {
                chatIds: {
                    type: 'array',
                    items: { type: 'string' },
                    description: 'Array of chat IDs to add',
                },
                folderId: {
                    type: 'number',
                    description: 'The folder ID to add chats to',
                },
            },
            required: ['chatIds', 'folderId'],
        },
    },
    {
        name: 'batchArchive',
        description: 'Archive multiple chats at once.',
        inputSchema: {
            type: 'object',
            properties: {
                chatIds: {
                    type: 'array',
                    items: { type: 'string' },
                    description: 'Array of chat IDs to archive',
                },
            },
            required: ['chatIds'],
        },
    },
];
// WebSocket connection to relay
let ws = null;
let isConnecting = false;
let pendingRequests = new Map();
function generateId() {
    return `${Date.now()}-${Math.random().toString(36).slice(2, 11)}`;
}
function log(msg, ...args) {
    console.error(`[telebiz-mcp] ${msg}`, ...args);
}
function connectToRelay() {
    return new Promise((resolve, reject) => {
        if (ws?.readyState === WebSocket.OPEN) {
            resolve();
            return;
        }
        if (isConnecting) {
            const checkInterval = setInterval(() => {
                if (ws?.readyState === WebSocket.OPEN) {
                    clearInterval(checkInterval);
                    resolve();
                }
            }, 100);
            return;
        }
        isConnecting = true;
        log(`Connecting to relay at ${RELAY_URL}`);
        ws = new WebSocket(RELAY_URL);
        ws.on('open', () => {
            isConnecting = false;
            log('Connected to relay');
            // Register as client
            ws.send(JSON.stringify({ type: 'register', role: 'client' }));
            resolve();
        });
        ws.on('message', (data) => {
            try {
                const message = JSON.parse(data.toString());
                if (message.id && pendingRequests.has(message.id)) {
                    const pending = pendingRequests.get(message.id);
                    clearTimeout(pending.timeout);
                    pendingRequests.delete(message.id);
                    if (message.success === false) {
                        pending.reject(new Error(message.error || 'Unknown error'));
                    }
                    else {
                        pending.resolve(message);
                    }
                }
            }
            catch (e) {
                log('Failed to parse message:', e);
            }
        });
        ws.on('close', () => {
            isConnecting = false;
            ws = null;
            log('Disconnected from relay');
            // Reject all pending requests
            for (const [id, pending] of pendingRequests) {
                clearTimeout(pending.timeout);
                pending.reject(new Error('Connection closed'));
            }
            pendingRequests.clear();
        });
        ws.on('error', (error) => {
            isConnecting = false;
            log('WebSocket error:', error.message);
            if (!ws || ws.readyState !== WebSocket.OPEN) {
                reject(error);
            }
        });
    });
}
async function executeToolOnRelay(tool, args) {
    await connectToRelay();
    if (!ws || ws.readyState !== WebSocket.OPEN) {
        throw new Error('Not connected to relay');
    }
    return new Promise((resolve, reject) => {
        const id = generateId();
        const timeout = setTimeout(() => {
            pendingRequests.delete(id);
            reject(new Error('Request timeout'));
        }, 60000);
        pendingRequests.set(id, { resolve, reject, timeout });
        ws.send(JSON.stringify({
            id,
            type: 'execute',
            tool,
            args,
        }));
    });
}
// Create MCP server
const server = new Server({
    name: 'telebiz',
    version: '1.0.0',
}, {
    capabilities: {
        tools: {},
    },
});
// Handle list tools
server.setRequestHandler(ListToolsRequestSchema, async () => {
    return { tools: TOOLS };
});
// Handle call tool
server.setRequestHandler(CallToolRequestSchema, async (request) => {
    const { name, arguments: args } = request.params;
    try {
        const result = await executeToolOnRelay(name, args || {});
        return {
            content: [
                {
                    type: 'text',
                    text: JSON.stringify(result, null, 2),
                },
            ],
        };
    }
    catch (error) {
        return {
            content: [
                {
                    type: 'text',
                    text: `Error: ${error instanceof Error ? error.message : 'Unknown error'}`,
                },
            ],
            isError: true,
        };
    }
});
// Run server
async function main() {
    const transport = new StdioServerTransport();
    await server.connect(transport);
    log('MCP server running on stdio');
}
main().catch((error) => {
    log('Fatal error:', error);
    process.exit(1);
});
