#!/usr/bin/env node
/**
 * Telebiz HTTP MCP Server
 *
 * Wraps telebiz-mcp in an HTTP server so mcporter can connect without spawning
 * a new process. This keeps the browser connection persistent.
 *
 * Usage:
 *   node http-server.js [port]
 *
 * mcporter config:
 *   { "telebiz": { "url": "http://localhost:9718/mcp" } }
 */
import * as http from 'http';
import { spawn } from 'child_process';
import * as readline from 'readline';
const HTTP_PORT = parseInt(process.env.TELEBIZ_HTTP_PORT || '9718');
const LOG_PREFIX = '[telebiz-http]';
let mcpProcess = null;
let isConnected = false;
let toolsCount = 0;
let requestId = 0;
const pendingRequests = new Map();
function log(msg) {
    console.error(`${LOG_PREFIX} ${msg}`);
}
function startMcpProcess() {
    return new Promise((resolve, reject) => {
        log('Starting telebiz-mcp subprocess...');
        mcpProcess = spawn('telebiz-mcp', [], {
            stdio: ['pipe', 'pipe', 'pipe'],
        });
        // Handle stderr (logs from telebiz-mcp)
        mcpProcess.stderr?.on('data', (data) => {
            const msg = data.toString().trim();
            for (const line of msg.split('\n')) {
                log(`[mcp] ${line}`);
                if (line.includes('Loaded') && line.includes('tools')) {
                    const match = line.match(/Loaded (\d+) tools/);
                    if (match) {
                        toolsCount = parseInt(match[1]);
                        isConnected = true;
                        log(`Browser connected with ${toolsCount} tools`);
                    }
                }
                if (line.includes('disconnected')) {
                    isConnected = false;
                    toolsCount = 0;
                    log('Browser disconnected');
                }
            }
        });
        // Handle stdout (MCP JSON-RPC responses)
        const rl = readline.createInterface({ input: mcpProcess.stdout });
        rl.on('line', (line) => {
            try {
                const response = JSON.parse(line);
                if (response.id !== undefined && pendingRequests.has(response.id)) {
                    const pending = pendingRequests.get(response.id);
                    clearTimeout(pending.timer);
                    pendingRequests.delete(response.id);
                    pending.resolve(response);
                }
            }
            catch (e) {
                // Ignore non-JSON lines
            }
        });
        mcpProcess.on('error', (err) => {
            log(`MCP process error: ${err.message}`);
            reject(err);
        });
        mcpProcess.on('exit', (code) => {
            log(`MCP process exited with code ${code}`);
            isConnected = false;
            mcpProcess = null;
            // Auto-restart after delay
            setTimeout(() => {
                if (!mcpProcess) {
                    log('Auto-restarting MCP process...');
                    startMcpProcess().catch(e => log(`Restart failed: ${e}`));
                }
            }, 3000);
        });
        // Give it time to start
        setTimeout(() => resolve(), 500);
    });
}
function sendMcpRequest(method, params) {
    return new Promise((resolve, reject) => {
        if (!mcpProcess || !mcpProcess.stdin) {
            reject(new Error('MCP process not running'));
            return;
        }
        const id = ++requestId;
        const request = {
            jsonrpc: '2.0',
            id,
            method,
            params: params || {},
        };
        const timer = setTimeout(() => {
            pendingRequests.delete(id);
            reject(new Error('Request timeout (30s)'));
        }, 30000);
        pendingRequests.set(id, { resolve, reject, timer });
        mcpProcess.stdin.write(JSON.stringify(request) + '\n');
    });
}
function makeTextResult(id, obj) {
    return {
        jsonrpc: '2.0',
        id,
        result: {
            content: [
                {
                    type: 'text',
                    text: JSON.stringify(obj, null, 2),
                },
            ],
        },
    };
}
async function callTool(name, args) {
    return sendMcpRequest('tools/call', { name, arguments: args });
}
async function handleMcpRequest(body) {
    const request = JSON.parse(body);
    // Intercept tool calls to apply local workarounds for known upstream issues.
    if (request?.method === 'tools/call' && request?.params?.name) {
        const toolName = request.params.name;
        const toolArgs = (request.params.arguments || {});
        // Workaround: upstream `batchAddToFolder` fails when chatIds.length > 1.
        // Implement it here as sequential `addChatToFolder` calls.
        if (toolName === 'batchAddToFolder') {
            const folderId = toolArgs.folderId;
            const chatIds = toolArgs.chatIds;
            if (folderId && Array.isArray(chatIds) && chatIds.length > 1) {
                const results = [];
                for (const chatId of chatIds) {
                    try {
                        await callTool('addChatToFolder', { chatId, folderIds: [folderId] });
                        results.push({ chatId, success: true });
                    }
                    catch (e) {
                        const msg = e instanceof Error ? e.message : String(e);
                        results.push({ chatId, success: false, error: msg });
                    }
                }
                return makeTextResult(request.id, {
                    added: results.filter(r => r.success).length,
                    total: results.length,
                    folderId,
                    results,
                    note: 'Workaround applied: implemented batchAddToFolder as sequential addChatToFolder calls.',
                });
            }
        }
        // Workaround: `linkEntityToChat` appears to expect entityType="organization" instead of "company".
        if (toolName === 'linkEntityToChat') {
            const entityType = toolArgs.entityType;
            if (entityType === 'company') {
                request.params.arguments = { ...toolArgs, entityType: 'organization' };
            }
        }
    }
    // Default: forward to telebiz-mcp
    const response = await sendMcpRequest(request.method, request.params);
    return response;
}
function createHttpServer() {
    const server = http.createServer(async (req, res) => {
        // CORS headers
        res.setHeader('Access-Control-Allow-Origin', '*');
        res.setHeader('Access-Control-Allow-Methods', 'GET, POST, OPTIONS');
        res.setHeader('Access-Control-Allow-Headers', 'Content-Type');
        if (req.method === 'OPTIONS') {
            res.writeHead(204);
            res.end();
            return;
        }
        // Status endpoint
        if (req.url === '/status' && req.method === 'GET') {
            res.setHeader('Content-Type', 'application/json');
            res.end(JSON.stringify({
                running: !!mcpProcess,
                connected: isConnected,
                tools: toolsCount,
            }));
            return;
        }
        // MCP endpoint
        if (req.url === '/mcp' && req.method === 'POST') {
            let body = '';
            req.on('data', chunk => body += chunk);
            req.on('end', async () => {
                res.setHeader('Content-Type', 'application/json');
                try {
                    const result = await handleMcpRequest(body);
                    res.end(JSON.stringify(result));
                }
                catch (err) {
                    const error = err instanceof Error ? err.message : String(err);
                    res.end(JSON.stringify({
                        jsonrpc: '2.0',
                        error: { code: -32000, message: error },
                        id: null,
                    }));
                }
            });
            return;
        }
        // Not found
        res.writeHead(404);
        res.end('Not Found');
    });
    return server;
}
async function main() {
    log('Starting Telebiz HTTP MCP Server...');
    // Start the MCP subprocess
    await startMcpProcess();
    // Start HTTP server
    const server = createHttpServer();
    server.listen(HTTP_PORT, () => {
        log(`HTTP server listening on port ${HTTP_PORT}`);
        log(`MCP endpoint: http://localhost:${HTTP_PORT}/mcp`);
        log(`Status endpoint: http://localhost:${HTTP_PORT}/status`);
        log('Waiting for browser to connect to telebiz-mcp...');
    });
    // Handle shutdown
    const shutdown = () => {
        log('Shutting down...');
        mcpProcess?.kill();
        server.close();
        process.exit(0);
    };
    process.on('SIGTERM', shutdown);
    process.on('SIGINT', shutdown);
}
main().catch(err => {
    log(`Fatal error: ${err}`);
    process.exit(1);
});
