/**
 * Health Check Module for telebiz-mcp
 *
 * Provides status checking and health monitoring for the relay and executor.
 */
import WebSocket from 'ws';
const RELAY_URL = process.env.TELEBIZ_RELAY_URL || 'ws://localhost:9716';
const HEALTH_TIMEOUT = 5000;
/**
 * Check if the relay server is running and if an executor is connected
 */
export async function checkHealth() {
    return new Promise((resolve) => {
        const status = {
            relay: 'unknown',
            executor: 'unknown',
            timestamp: Date.now(),
        };
        const timeout = setTimeout(() => {
            status.relay = 'down';
            status.error = 'Connection timeout';
            resolve(status);
        }, HEALTH_TIMEOUT);
        try {
            const ws = new WebSocket(RELAY_URL);
            ws.on('open', () => {
                status.relay = 'up';
                // Register as client and request status
                ws.send(JSON.stringify({ type: 'register', role: 'client' }));
                ws.send(JSON.stringify({ id: 'health-check', type: 'ping' }));
            });
            ws.on('message', (data) => {
                try {
                    const message = JSON.parse(data.toString());
                    if (message.type === 'pong') {
                        // Ping worked - executor is connected and responsive
                        status.executor = 'connected';
                        clearTimeout(timeout);
                        ws.close();
                        resolve(status);
                    }
                    else if (message.type === 'error' && message.error?.includes('Executor not connected')) {
                        status.executor = 'disconnected';
                        clearTimeout(timeout);
                        ws.close();
                        resolve(status);
                    }
                }
                catch (e) {
                    // Ignore parse errors
                }
            });
            ws.on('error', (error) => {
                status.relay = 'down';
                status.error = error.message;
                clearTimeout(timeout);
                resolve(status);
            });
            ws.on('close', () => {
                if (status.relay === 'unknown') {
                    status.relay = 'down';
                }
                clearTimeout(timeout);
                resolve(status);
            });
        }
        catch (error) {
            status.relay = 'down';
            status.error = error instanceof Error ? error.message : 'Unknown error';
            clearTimeout(timeout);
            resolve(status);
        }
    });
}
/**
 * Get detailed status including tool count
 */
export async function getDetailedStatus() {
    const health = await checkHealth();
    const detailed = {
        ...health,
        relayUrl: RELAY_URL,
    };
    if (health.relay === 'up' && health.executor === 'connected') {
        // Try to get tool list
        try {
            const tools = await listToolsQuick();
            detailed.tools = tools;
        }
        catch (e) {
            // Ignore
        }
    }
    return detailed;
}
/**
 * Quick tool count check
 */
async function listToolsQuick() {
    return new Promise((resolve, reject) => {
        const timeout = setTimeout(() => reject(new Error('Timeout')), 3000);
        const ws = new WebSocket(RELAY_URL);
        ws.on('open', () => {
            ws.send(JSON.stringify({ type: 'register', role: 'client' }));
            ws.send(JSON.stringify({ id: 'list-tools', type: 'list_tools' }));
        });
        ws.on('message', (data) => {
            try {
                const message = JSON.parse(data.toString());
                if (message.type === 'tools' && Array.isArray(message.tools)) {
                    clearTimeout(timeout);
                    ws.close();
                    resolve(message.tools.length);
                }
            }
            catch (e) {
                // Ignore
            }
        });
        ws.on('error', (error) => {
            clearTimeout(timeout);
            reject(error);
        });
    });
}
/**
 * Format status for display
 */
export function formatStatus(status) {
    const lines = [
        '📱 Telebiz MCP Status',
        '─'.repeat(20),
        `Relay: ${status.relay === 'up' ? '✅ Running' : '❌ Down'}`,
        `Executor: ${status.executor === 'connected' ? '✅ Connected' : '⚠️ Not connected'}`,
    ];
    if (status.tools !== undefined) {
        lines.push(`Tools: ${status.tools} available`);
    }
    if (status.error) {
        lines.push(`Error: ${status.error}`);
    }
    lines.push(`URL: ${status.relayUrl}`);
    lines.push(`Checked: ${new Date(status.timestamp).toISOString()}`);
    return lines.join('\n');
}
// CLI entry point
if (process.argv[1]?.endsWith('health.ts') || process.argv[1]?.endsWith('health.js')) {
    getDetailedStatus().then((status) => {
        console.log(formatStatus(status));
        // Exit with appropriate code
        if (status.relay === 'down') {
            process.exit(2);
        }
        else if (status.executor !== 'connected') {
            process.exit(1);
        }
        else {
            process.exit(0);
        }
    });
}
