#!/usr/bin/env node
/**
 * Telebiz MCP Daemon
 *
 * Keeps telebiz-mcp running persistently and exposes an HTTP API for tool calls.
 * This solves the browser reconnection timing issue.
 *
 * Usage:
 *   node daemon.js start   - Start the daemon
 *   node daemon.js stop    - Stop the daemon
 *   node daemon.js status  - Check status
 *   node daemon.js call <tool> [args]  - Call a tool
 */
import { spawn } from 'child_process';
import * as http from 'http';
import * as fs from 'fs';
import * as path from 'path';
const PORT = 9717; // HTTP API port (9716 is WebSocket)
const PID_FILE = path.join(process.env.HOME || '/tmp', '.telebiz-daemon.pid');
const LOG_FILE = path.join(process.env.HOME || '/tmp', '.telebiz-daemon.log');
let mcpProcess = null;
let isConnected = false;
let toolsLoaded = 0;
let pendingRequests = new Map();
function log(msg) {
    const line = `[${new Date().toISOString()}] ${msg}\n`;
    fs.appendFileSync(LOG_FILE, line);
    console.log(msg);
}
function generateId() {
    return `req_${Date.now()}_${Math.random().toString(36).slice(2, 9)}`;
}
function startMcpProcess() {
    return new Promise((resolve, reject) => {
        log('Starting telebiz-mcp process...');
        mcpProcess = spawn('telebiz-mcp', [], {
            stdio: ['pipe', 'pipe', 'pipe'],
        });
        mcpProcess.stderr?.on('data', (data) => {
            const msg = data.toString();
            log(`[MCP] ${msg.trim()}`);
            if (msg.includes('Loaded') && msg.includes('tools')) {
                const match = msg.match(/Loaded (\d+) tools/);
                if (match) {
                    toolsLoaded = parseInt(match[1]);
                    isConnected = true;
                    log(`Browser connected with ${toolsLoaded} tools`);
                }
            }
            if (msg.includes('disconnected')) {
                isConnected = false;
                toolsLoaded = 0;
                log('Browser disconnected');
            }
        });
        mcpProcess.stdout?.on('data', (data) => {
            try {
                const lines = data.toString().trim().split('\n');
                for (const line of lines) {
                    if (!line)
                        continue;
                    const msg = JSON.parse(line);
                    // Handle MCP responses
                    if (msg.id && pendingRequests.has(msg.id.toString())) {
                        const pending = pendingRequests.get(msg.id.toString());
                        clearTimeout(pending.timeout);
                        pendingRequests.delete(msg.id.toString());
                        pending.resolve(msg);
                    }
                }
            }
            catch (e) {
                // Ignore parse errors
            }
        });
        mcpProcess.on('error', (err) => {
            log(`MCP process error: ${err.message}`);
            reject(err);
        });
        mcpProcess.on('exit', (code) => {
            log(`MCP process exited with code ${code}`);
            isConnected = false;
            mcpProcess = null;
            // Auto-restart after 5 seconds
            setTimeout(() => {
                if (!mcpProcess) {
                    log('Auto-restarting MCP process...');
                    startMcpProcess().catch(log);
                }
            }, 5000);
        });
        // Wait for process to start
        setTimeout(resolve, 1000);
    });
}
async function callTool(tool, args) {
    if (!mcpProcess || !mcpProcess.stdin) {
        throw new Error('MCP process not running');
    }
    const id = generateId();
    // Build MCP request
    const request = {
        jsonrpc: '2.0',
        id,
        method: 'tools/call',
        params: {
            name: tool,
            arguments: args,
        },
    };
    return new Promise((resolve, reject) => {
        const timeout = setTimeout(() => {
            pendingRequests.delete(id);
            reject(new Error('Request timeout'));
        }, 30000);
        pendingRequests.set(id, { resolve, reject, timeout });
        mcpProcess.stdin.write(JSON.stringify(request) + '\n');
    });
}
function startHttpServer() {
    const server = http.createServer(async (req, res) => {
        res.setHeader('Content-Type', 'application/json');
        if (req.url === '/status') {
            res.end(JSON.stringify({
                running: !!mcpProcess,
                connected: isConnected,
                tools: toolsLoaded,
                pid: mcpProcess?.pid,
            }));
            return;
        }
        if (req.url === '/call' && req.method === 'POST') {
            let body = '';
            req.on('data', chunk => body += chunk);
            req.on('end', async () => {
                try {
                    const { tool, args } = JSON.parse(body);
                    const result = await callTool(tool, args || {});
                    res.end(JSON.stringify({ success: true, result }));
                }
                catch (err) {
                    res.statusCode = 500;
                    res.end(JSON.stringify({
                        success: false,
                        error: err instanceof Error ? err.message : String(err)
                    }));
                }
            });
            return;
        }
        res.statusCode = 404;
        res.end(JSON.stringify({ error: 'Not found' }));
    });
    server.listen(PORT, () => {
        log(`HTTP API listening on port ${PORT}`);
    });
    return server;
}
async function main() {
    const cmd = process.argv[2];
    if (cmd === 'status') {
        try {
            const res = await fetch(`http://localhost:${PORT}/status`);
            const status = await res.json();
            console.log(JSON.stringify(status, null, 2));
            process.exit(status.connected ? 0 : 1);
        }
        catch {
            console.log('Daemon not running');
            process.exit(2);
        }
        return;
    }
    if (cmd === 'stop') {
        if (fs.existsSync(PID_FILE)) {
            const pid = fs.readFileSync(PID_FILE, 'utf-8').trim();
            try {
                process.kill(parseInt(pid), 'SIGTERM');
                fs.unlinkSync(PID_FILE);
                console.log('Daemon stopped');
            }
            catch {
                console.log('Daemon not running');
            }
        }
        return;
    }
    if (cmd === 'call') {
        const tool = process.argv[3];
        const argsJson = process.argv[4] || '{}';
        try {
            const res = await fetch(`http://localhost:${PORT}/call`, {
                method: 'POST',
                body: JSON.stringify({ tool, args: JSON.parse(argsJson) }),
            });
            const result = await res.json();
            console.log(JSON.stringify(result, null, 2));
            process.exit(result.success ? 0 : 1);
        }
        catch (err) {
            console.error('Error:', err instanceof Error ? err.message : err);
            process.exit(1);
        }
        return;
    }
    // Default: start daemon
    log('Starting Telebiz MCP Daemon...');
    // Save PID
    fs.writeFileSync(PID_FILE, process.pid.toString());
    // Start MCP process
    await startMcpProcess();
    // Start HTTP server
    startHttpServer();
    // Handle shutdown
    process.on('SIGTERM', () => {
        log('Received SIGTERM, shutting down...');
        mcpProcess?.kill();
        process.exit(0);
    });
    process.on('SIGINT', () => {
        log('Received SIGINT, shutting down...');
        mcpProcess?.kill();
        process.exit(0);
    });
    log('Daemon started. Waiting for browser to connect...');
}
main().catch(err => {
    log(`Fatal error: ${err}`);
    process.exit(1);
});
