---
name: telebiz-mcp
description: Access Telegram data via MCP using the telebiz-tt browser client. Lists chats, reads messages, searches, manages folders, and sends messages through an authenticated Telegram session.
metadata: {"clawdbot":{"emoji":"📱"}}
---

# telebiz-mcp

MCP integration for Telegram via telebiz-tt browser client.

## Quick Rules (read this first)
- **Rate limits are strict**: max 20 calls/request, 30 calls/min, 500ms between calls, heavy ops 1s.
- For adding many chats to folders: **do NOT use `batchAddToFolder` with multiple chatIds** (known bug). Loop `addChatToFolder` sequentially.
- For CRM linking: `linkEntityToChat` is **unstable** in our tests. We observed `company` failing with Validation error, and at one point `organization` succeeding — but later `organization` also failed. Treat `linkEntityToChat` as unreliable until upstream clarifies schema/feature flags.
- Prefer reversible operations and clean up test artifacts (folders, groups) immediately.

## Architecture

```
┌──────────────┐     ┌──────────────────┐     ┌─────────────────┐
│ Clawdbot     │────▶│ MCP Server       │────▶│ WebSocket Relay │
│ (mcporter)   │     │ (stdio)          │     │ (port 9716)     │
└──────────────┘     └──────────────────┘     └────────┬────────┘
                                                       │
                                                       ▼
                                              ┌─────────────────┐
                                              │ Browser         │
                                              │ (telebiz-tt)    │
                                              │ [executor]      │
                                              └─────────────────┘
```

## Quick Setup

### Prerequisites
- Node.js 18+
- Telegram account

### 1. Install telebiz-mcp

```bash
npm install -g @telebiz/telebiz-mcp
```

### 2. Open Telebiz in browser

Go to **https://telebiz.io** and login with your Telegram account.

### 3. Start the HTTP server

```bash
cd ~/clawd/skills/telebiz-mcp
./start-http.sh
```

This starts a persistent server that:
- Runs telebiz-mcp internally
- Keeps browser connection alive  
- Exposes HTTP API on port 9718

### 4. Enable MCP in Telebiz

In telebiz.io: **Settings → Agent → Local MCP**

The status should show "Connected" once the server is running.

### 4. Verify connection

```bash
# Quick health check
cd ~/clawd/skills/telebiz-mcp
npm run health

# Should show:
# 📱 Telebiz MCP Status
# ────────────────────
# Relay: ✅ Running
# Executor: ✅ Connected
# Tools: 31 available
```

### 5. Test via mcporter

```bash
cd ~/clawd
mcporter call telebiz.listChats limit:5
```

## Health Monitoring

### Manual Check

```bash
# Check status
npm run health

# JSON output for automation
node dist/health.js --json
```

### Monitor Script

The monitor tracks state changes and can be used with cron:

```bash
# Check and alert on changes
npm run monitor

# Quiet mode - only output on state change
node dist/monitor.js --quiet

# JSON output
node dist/monitor.js --json
```

Exit codes:
- `0` = Healthy (relay up, executor connected)
- `1` = Degraded (relay up, executor disconnected)
- `2` = Down (relay not running)
- `3` = State changed (for alerting)

### Cron Integration

Add to crontab for periodic monitoring:

```bash
# Check every 5 minutes, alert on changes
*/5 * * * * cd ~/clawd/skills/telebiz-mcp && node dist/monitor.js --quiet >> /var/log/telebiz-monitor.log 2>&1
```

### Heartbeat Integration

Add to `HEARTBEAT.md` for Clawdbot monitoring:

```markdown
### Telebiz MCP (every 2h)
- [ ] Run: `cd ~/clawd/skills/telebiz-mcp && npm run health`
- [ ] If degraded/down: Alert Albert via Telegram
```

## Available Tools

### Chat Tools
| Tool | Description |
|------|-------------|
| `listChats` | List chats with filters (type, unread, archived, etc.) |
| `getChatInfo` | Get detailed chat information |
| `getCurrentChat` | Get currently open chat |
| `openChat` | Navigate to a chat |
| `archiveChat` | Archive a chat |
| `unarchiveChat` | Unarchive a chat |
| `pinChat` | Pin a chat |
| `unpinChat` | Unpin a chat |
| `muteChat` | Mute notifications |
| `unmuteChat` | Unmute notifications |
| `deleteChat` | Delete/leave chat ⚠️ |

### Message Tools
| Tool | Description |
|------|-------------|
| `sendMessage` | Send text message (markdown supported) |
| `forwardMessages` | Forward messages between chats |
| `deleteMessages` | Delete messages ⚠️ |
| `searchMessages` | Search globally or in a chat |
| `getRecentMessages` | Get message history |
| `markChatAsRead` | Mark all messages as read |

### Folder Tools
| Tool | Description |
|------|-------------|
| `listFolders` | List all chat folders |
| `createFolder` | Create a new folder |
| `addChatToFolder` | Add chat to folders |
| `removeChatFromFolder` | Remove chat from folders |
| `deleteFolder` | Delete a folder ⚠️ |

### Member Tools
| Tool | Description |
|------|-------------|
| `getChatMembers` | List group/channel members |
| `addChatMembers` | Add users to group |
| `removeChatMember` | Remove user from group |
| `createGroup` | Create a new group |

### User Tools
| Tool | Description |
|------|-------------|
| `searchUsers` | Search by name/username |
| `getUserInfo` | Get user details |

### Batch Tools
| Tool | Description |
|------|-------------|
| `batchSendMessage` | Send to multiple chats |
| `batchAddToFolder` | Add multiple chats to folder |
| `batchArchive` | Archive multiple chats |

## Usage Examples

### Find chats waiting for my reply
```bash
mcporter call telebiz.listChats iAmLastSender=false hasUnread=true limit:20
```

### Find stale conversations I started
```bash
mcporter call telebiz.listChats iAmLastSender=true lastMessageOlderThanDays:7 limit:20
```

### Search all messages
```bash
mcporter call telebiz.searchMessages query="invoice" limit:20
```

### Search in specific chat
```bash
mcporter call telebiz.searchMessages query="meeting" chatId=-1001234567890 limit:10
```

### Send message
```bash
mcporter call telebiz.sendMessage chatId=-1001234567890 text="Hello from Clawdbot!"
```

### Get recent messages
```bash
mcporter call telebiz.getRecentMessages chatId=-1001234567890 limit:50
```

### Paginate through history
```bash
# Page 1 (newest 50)
mcporter call telebiz.getRecentMessages chatId=-1001234567890 limit:50 offset:0

# Page 2 (messages 51-100)
mcporter call telebiz.getRecentMessages chatId=-1001234567890 limit:50 offset:50
```

### Organize chats
```bash
# List folders
mcporter call telebiz.listFolders

# Add chats to folder
mcporter call telebiz.batchAddToFolder chatIds='["-1001234","-1001235"]' folderId:5
```

## Rate Limiting

The browser enforces rate limits to prevent Telegram flood protection:
- **Max calls per request**: 20
- **Max calls per minute**: 30
- **Min delay between calls**: 500ms
- **Delay for heavy operations** (send/forward/delete): 1s

(These values come from the Telebiz UI and are the effective limits we observed in practice.)

## Known Issues / Workarounds (Feb 2026)

### `batchAddToFolder` fails for multiple chatIds
Observed behavior:
- `batchAddToFolder(folderId, chatIds=[one])` works (or reports `alreadyIncluded`)
- `batchAddToFolder(folderId, chatIds=[two or more])` fails with: **"Error: Failed to update folder"**
- Repro confirmed for both:
  - Auto + another **group**
  - Auto + a **private** chat

**Workaround:** loop sequentially:
- `addChatToFolder(chatId=A, folderIds=[folderId])`
- `addChatToFolder(chatId=B, folderIds=[folderId])`

### `linkEntityToChat` is unstable / schema mismatch
Observed behavior (Feb 2026):
- `linkEntityToChat` returns **"Validation error"** for `entityType=deal`, `contact`, and `company`.
- In one run, using `entityType="organization"` successfully linked a HubSpot company to a chat — but later `organization` also returned **"Validation error"**.

**Implication:** this tool is either behind a feature flag, has changing server-side validation, or the published schema/enums don’t match what the backend expects.

**Workaround:**
- Prefer linking via `createContact/createDeal/createCompany` (these link to the chat at creation time).
- Use `associateEntities` to connect deal↔company/contact.
- Don’t depend on `linkEntityToChat` until upstream provides a stable contract + better error messages.

## Troubleshooting

### Relay not starting
```bash
# Check if port is in use
ss -tlnp | grep 9716

# Kill existing process
pkill -f "relay.js"

# Start fresh
./start-relay.sh
```

### Browser not connecting
1. Verify relay is running: `npm run health`
2. Check browser console (F12) for WebSocket errors
3. Ensure MCP is enabled in Settings → Agent → Enable MCP
4. Try refreshing the telebiz-tt page

### "Executor not connected" error
The browser tab with telebiz-tt must be:
- Open and visible (not suspended)
- Logged into Telegram
- MCP enabled in settings

### Rate limit errors
- Reduce batch sizes
- Add delays between operations
- Be more specific in filters to reduce API calls

### Session expired
If Telegram session expires:
1. Refresh the telebiz-tt browser page
2. Re-login if prompted
3. Re-enable MCP in settings

## Configuration

### Environment Variables

| Variable | Default | Description |
|----------|---------|-------------|
| `TELEBIZ_PORT` | `9716` | Relay WebSocket port |
| `TELEBIZ_RELAY_URL` | `ws://localhost:9716` | Relay URL for MCP server |
| `TELEBIZ_STATE_FILE` | `~/.telebiz-mcp-state.json` | Monitor state file |

### mcporter Config

Located at `~/clawd/config/mcporter.json`:

```json
{
  "mcpServers": {
    "telebiz": {
      "url": "http://localhost:9718/mcp"
    }
  }
}
```

**Note**: Use the HTTP URL (not stdio) to avoid spawning conflicts.

## Security Notes

- The browser holds your Telegram session - keep it secure
- Don't expose the relay port (9716) to the internet
- Review tool calls before executing destructive operations
- Rate limits help prevent accidental spam
