#!/usr/bin/env python3
"""
OpenClaw TaskMaster Integration Example

This shows how to use TaskMaster from within OpenClaw with real token tracking.
"""

import json
from delegate_task import TaskMaster, ModelTier

def example_workflow():
    """Example workflow showing real TaskMaster usage."""
    
    print("TaskMaster + OpenClaw Integration Demo")
    print("=" * 50)
    
    # 1. Create TaskMaster
    tm = TaskMaster(total_budget=5.0)
    
    # 2. Create a task
    task_id = "pdf_research"
    task = tm.create_task(
        task_id, 
        "Research Python PDF processing libraries and compare PyPDF2, pdfplumber, and pymupdf"
    )
    
    print(f"\nCreated task: {task_id}")
    print(f"   Model: {task.model.value}")
    print(f"   Estimated: ${task.estimated_cost:.3f}")
    
    # 3. Generate spawn command for OpenClaw
    spawn_cmd = tm.generate_spawn_command(task_id)
    print(f"\nOpenClaw Command:")
    print("sessions_spawn(")
    spawn_data = json.loads(spawn_cmd)
    for key, value in spawn_data.items():
        if isinstance(value, str):
            print(f'    {key}="{value}",')
        else:
            print(f"    {key}={value},")
    print(")")
    
    # 4. After OpenClaw sessions_spawn completes, you would:
    # session_key = "isolated-abc123"  # From spawn response
    # tm.update_with_actual_cost(task_id, session_key)
    
    print(f"\nAfter completion, call:")
    print(f"tm.update_with_actual_cost('{task_id}', session_key)")
    print(f"This will fetch real tokens and update cost tracking.")

def real_session_example(session_key: str):
    """Example showing how to update with real session data."""
    tm = TaskMaster()
    
    # This is what would be called after a task completes
    task_id = "example_task"
    
    # In the real implementation, this would call:
    # session_status(sessionKey=session_key)
    # And parse the response like "8 in / 583 out"
    
    print(f"Getting actual costs for session: {session_key}")
    print(f"Would call: session_status(sessionKey='{session_key}')")
    print(f"Parse tokens and update TaskResult with real data")

if __name__ == "__main__":
    example_workflow()