#!/usr/bin/env python3
"""
TaskMaster Delegation Engine

Core orchestration logic for task delegation, model selection, and budget management.

USAGE FROM OPENCLAW:
1. Create TaskMaster: tm = TaskMaster(total_budget=5.0)
2. Create task: task = tm.create_task("task1", "Research PDF tools")
3. Get spawn command: spawn_cmd = tm.generate_spawn_command("task1")
4. Execute: sessions_spawn(**json.loads(spawn_cmd))
5. After completion: tm.update_with_actual_cost("task1", session_key)

This tracks real token usage and maintains cost logs in taskmaster-costs.json
"""

import json
import re
import time
from dataclasses import dataclass
from typing import Dict, List, Optional, Tuple
from enum import Enum

class ModelTier(Enum):
    HAIKU = "anthropic/claude-haiku-4-20250327"
    SONNET = "anthropic/claude-sonnet-4-20250514" 
    OPUS = "anthropic/claude-opus-4-5"

@dataclass
class ModelPricing:
    input_per_1m: float
    output_per_1m: float
    cache_write_per_1m: float
    cache_read_per_1m: float

# Current Anthropic pricing (as of 2025)
PRICING = {
    ModelTier.HAIKU: ModelPricing(0.25, 1.25, 1.25, 0.03),
    ModelTier.SONNET: ModelPricing(3.0, 15.0, 3.75, 0.15),
    ModelTier.OPUS: ModelPricing(15.0, 75.0, 18.75, 0.75)
}

@dataclass
class Task:
    description: str
    model: ModelTier
    estimated_cost: float
    priority: str = "medium"
    dependencies: List[str] = None
    max_retries: int = 2
    timeout_minutes: int = 30
    force_model: bool = False
    
    def __post_init__(self):
        if self.dependencies is None:
            self.dependencies = []

@dataclass
class TaskResult:
    task_id: str
    success: bool
    output: str
    actual_cost: float
    execution_time: float
    model_used: ModelTier
    retry_count: int = 0
    error_message: str = ""

class TaskMaster:
    def __init__(self, total_budget: float = 10.0):
        self.total_budget = total_budget
        self.spent_budget = 0.0
        self.tasks: Dict[str, Task] = {}
        self.results: Dict[str, TaskResult] = {}
        self.active_tasks: Dict[str, str] = {}  # task_id -> session_id
        
    def analyze_complexity(self, description: str) -> Tuple[ModelTier, float]:
        """
        Analyze task complexity and assign appropriate model with cost estimate.
        
        Returns (model, estimated_cost)
        """
        desc_lower = description.lower()
        
        # Patterns for different complexity levels
        haiku_patterns = [
            r'\b(search|find|lookup|fetch|get|retrieve)\b',
            r'\b(format|convert|extract|parse)\b', 
            r'\b(list|count|sum|average)\b',
            r'\b(simple|basic|quick)\b'
        ]
        
        opus_patterns = [
            r'\b(architecture|design|strategy)\b',
            r'\b(complex|advanced|optimize|refactor)\b',
            r'\b(security|review|audit)\b',
            r'\b(novel|creative|innovative)\b',
            r'\b(debug.*complex|performance.*analysis)\b'
        ]
        
        # Check for explicit model forcing
        force_match = re.search(r'\[FORCE:\s*(HAIKU|SONNET|OPUS)\]', description, re.IGNORECASE)
        if force_match:
            model_name = force_match.group(1).upper()
            if model_name == "HAIKU":
                return ModelTier.HAIKU, self.estimate_cost(ModelTier.HAIKU, description)
            elif model_name == "SONNET":
                return ModelTier.SONNET, self.estimate_cost(ModelTier.SONNET, description)
            elif model_name == "OPUS":
                return ModelTier.OPUS, self.estimate_cost(ModelTier.OPUS, description)
        
        # Pattern matching for complexity
        opus_score = sum(1 for pattern in opus_patterns if re.search(pattern, desc_lower))
        haiku_score = sum(1 for pattern in haiku_patterns if re.search(pattern, desc_lower))
        
        # Word count and complexity indicators
        word_count = len(description.split())
        has_multiple_steps = len(re.findall(r'\b(then|next|after|also|additionally)\b', desc_lower)) > 0
        
        # Decision logic
        if opus_score >= 2 or ('architecture' in desc_lower) or ('security' in desc_lower):
            model = ModelTier.OPUS
        elif haiku_score >= 2 and opus_score == 0 and word_count < 50 and not has_multiple_steps:
            model = ModelTier.HAIKU
        else:
            model = ModelTier.SONNET  # Default to Sonnet for most tasks
            
        estimated_cost = self.estimate_cost(model, description)
        return model, estimated_cost
    
    def estimate_cost(self, model: ModelTier, description: str) -> float:
        """Estimate cost based on task description and model."""
        base_input_tokens = len(description.split()) * 1.3  # Rough token estimate
        
        # Estimate output tokens based on task type
        if 'research' in description.lower() or 'analysis' in description.lower():
            output_tokens = 1000  # Research tasks tend to be verbose
        elif 'code' in description.lower() or 'implement' in description.lower():
            output_tokens = 800   # Code tasks
        elif 'simple' in description.lower() or 'format' in description.lower():
            output_tokens = 200   # Simple tasks
        else:
            output_tokens = 500   # Default
            
        pricing = PRICING[model]
        input_cost = (base_input_tokens / 1_000_000) * pricing.input_per_1m
        output_cost = (output_tokens / 1_000_000) * pricing.output_per_1m
        
        # Add cache write cost for first interaction (rough estimate)
        cache_cost = (20_000 / 1_000_000) * pricing.cache_write_per_1m
        
        return input_cost + output_cost + cache_cost
    
    def create_task(self, task_id: str, description: str, 
                   force_model: Optional[ModelTier] = None,
                   priority: str = "medium",
                   dependencies: List[str] = None) -> Task:
        """Create a new task with automatic model selection."""
        
        if force_model:
            model = force_model
            estimated_cost = self.estimate_cost(model, description)
            force = True
        else:
            model, estimated_cost = self.analyze_complexity(description)
            force = False
            
        task = Task(
            description=description,
            model=model, 
            estimated_cost=estimated_cost,
            priority=priority,
            dependencies=dependencies or [],
            force_model=force
        )
        
        self.tasks[task_id] = task
        return task
    
    def can_execute_task(self, task_id: str) -> Tuple[bool, str]:
        """Check if task can be executed (budget + dependencies)."""
        task = self.tasks[task_id]
        
        # Check budget
        if self.spent_budget + task.estimated_cost > self.total_budget:
            return False, f"Would exceed budget: ${task.estimated_cost:.2f} + ${self.spent_budget:.2f} > ${self.total_budget:.2f}"
        
        # Check dependencies  
        for dep_id in task.dependencies:
            if dep_id not in self.results or not self.results[dep_id].success:
                return False, f"Dependency not completed: {dep_id}"
                
        return True, "Ready to execute"
    
    def generate_spawn_command(self, task_id: str) -> str:
        """Generate the sessions_spawn command for this task."""
        task = self.tasks[task_id]
        
        # Clean description for spawn (remove force directives)
        clean_desc = re.sub(r'\[FORCE:\s*\w+\]', '', task.description).strip()
        
        model_name = task.model.value
        
        spawn_cmd = {
            "task": clean_desc,
            "model": model_name,
            "runTimeoutSeconds": task.timeout_minutes * 60,
            "label": f"TaskMaster-{task_id}",
            "cleanup": "keep"  # Keep for result retrieval
        }
        
        return json.dumps(spawn_cmd, indent=2)
    
    def execute_task(self, task_id: str) -> TaskResult:
        """Execute a task via OpenClaw sessions_spawn with real token tracking."""
        task = self.tasks[task_id]
        
        can_execute, reason = self.can_execute_task(task_id)
        if not can_execute:
            return TaskResult(
                task_id=task_id,
                success=False,
                output="",
                actual_cost=0.0,
                execution_time=0.0,
                model_used=task.model,
                error_message=reason
            )
        
        print(f"\nExecuting Task: {task_id}")
        print(f"Description: {task.description}")
        print(f"Model: {task.model.value}")
        print(f"Estimated Cost: ${task.estimated_cost:.3f}")
        
        # Execute via sessions_spawn (requires OpenClaw integration)
        spawn_command = self.generate_spawn_command(task_id)
        print(f"Ready to spawn with:")
        print(spawn_command)
        
        # For now, return instruction for manual execution
        # TODO: Integrate with actual OpenClaw sessions_spawn API
        return TaskResult(
            task_id=task_id,
            success=True,
            output=f"Ready to spawn: {spawn_command}",
            actual_cost=task.estimated_cost,  # Will be real after integration
            execution_time=0.0,
            model_used=task.model
        )
    
    def track_session_cost(self, session_key: str, task_id: str) -> Tuple[float, int, int]:
        """
        Get actual token usage from completed session.
        Returns (actual_cost, input_tokens, output_tokens)
        """
        try:
            # This shows the integration - in real use, session_status would be available
            # session_result = session_status(sessionKey=session_key)
            
            # Parse session_status response (format: "🧮 Tokens: 8 in / 583 out")
            # tokens_line = [line for line in session_result.split('\n') if '🧮 Tokens:' in line][0]
            # Example: "🧮 Tokens: 1240 in / 3580 out"
            
            # For now, return estimates - real implementation would parse actual data
            task = self.tasks[task_id]
            model = task.model
            pricing = PRICING[model]
            
            # Placeholder - would be parsed from actual session_status
            input_tokens = 1200
            output_tokens = 800
            
            # Calculate actual cost
            input_cost = (input_tokens / 1_000_000) * pricing.input_per_1m  
            output_cost = (output_tokens / 1_000_000) * pricing.output_per_1m
            total_cost = input_cost + output_cost
            
            return total_cost, input_tokens, output_tokens
            
        except Exception as e:
            print(f"Warning: Could not get actual session costs - {e}")
            # Fall back to estimate
            task = self.tasks[task_id]
            return task.estimated_cost, 1000, 500
    
    def log_cost(self, task_id: str, result: TaskResult):
        """Log actual vs estimated costs to tracking file."""
        log_entry = {
            "timestamp": time.time(),
            "task_id": task_id,
            "description": self.tasks[task_id].description,
            "model": result.model_used.value,
            "estimated_cost": self.tasks[task_id].estimated_cost,
            "actual_cost": result.actual_cost,
            "accuracy": result.actual_cost / self.tasks[task_id].estimated_cost if self.tasks[task_id].estimated_cost > 0 else 1.0
        }
        
        try:
            # Read existing log
            try:
                with open("taskmaster-costs.json", "r") as f:
                    costs = json.load(f)
            except FileNotFoundError:
                costs = []
            
            # Append new entry
            costs.append(log_entry)
            
            # Write back
            with open("taskmaster-costs.json", "w") as f:
                json.dump(costs, f, indent=2)
                
        except Exception as e:
            print(f"Warning: Could not log costs - {e}")
    
    def update_with_actual_cost(self, task_id: str, session_key: str):
        """Update TaskResult with actual costs from completed session."""
        if task_id not in self.results:
            print(f"Warning: No result found for task {task_id}")
            return
            
        actual_cost, input_tokens, output_tokens = self.track_session_cost(session_key, task_id)
        
        # Update the result with actual data
        result = self.results[task_id]
        result.actual_cost = actual_cost
        
        # Update budget tracking
        self.spent_budget = self.spent_budget - self.tasks[task_id].estimated_cost + actual_cost
        
        # Log for tracking
        self.log_cost(task_id, result)
        
        print(f"Task {task_id} actual cost: ${actual_cost:.3f} (estimated: ${self.tasks[task_id].estimated_cost:.3f})")
        
    def get_status(self) -> Dict:
        """Get current project status."""
        return {
            "budget": {
                "total": self.total_budget,
                "spent": self.spent_budget,
                "remaining": self.total_budget - self.spent_budget
            },
            "tasks": {
                "total": len(self.tasks),
                "completed": len([r for r in self.results.values() if r.success]),
                "failed": len([r for r in self.results.values() if not r.success]),
                "pending": len(self.tasks) - len(self.results)
            },
            "active_sessions": list(self.active_tasks.values())
        }

def main():
    """Example usage of TaskMaster."""
    print("TaskMaster Delegation Engine")
    print("=" * 50)
    
    # Create TaskMaster with $5 budget
    tm = TaskMaster(total_budget=5.0)
    
    # Example tasks
    tasks = [
        ("research_1", "Research PDF processing libraries for Python", None),
        ("simple_1", "Extract email addresses from customer_data.csv", None),
        ("complex_1", "Design scalable microservices architecture for e-commerce platform [FORCE: OPUS]", None),
        ("code_1", "Implement user authentication module with JWT tokens", None)
    ]
    
    # Create and analyze tasks
    for task_id, description, force_model in tasks:
        task = tm.create_task(task_id, description, force_model)
        print(f"\nTask: {task_id}")
        print(f"   Description: {description}")
        print(f"   Model: {task.model.value}")
        print(f"   Estimated Cost: ${task.estimated_cost:.3f}")
        print(f"   Force Override: {task.force_model}")
    
    # Show status
    print(f"\nProject Status:")
    status = tm.get_status()
    print(f"   Budget: ${status['budget']['spent']:.2f} / ${status['budget']['total']:.2f}")
    print(f"   Tasks: {status['tasks']['total']} total, {status['tasks']['pending']} pending")
    
    # Simulate execution
    print(f"\nExecution Simulation:")
    for task_id in tm.tasks.keys():
        result = tm.execute_task(task_id)
        tm.results[task_id] = result
        tm.spent_budget += result.actual_cost

if __name__ == "__main__":
    main()