---
name: task-runner
description: Manage tasks and projects across sessions with persistent task tracking. Use when you need to organize work, track progress, and maintain todo lists that persist between conversations. Features: add tasks with projects and priorities, list pending/completed tasks, mark tasks complete, export projects to markdown. Security: file exports are restricted to safe directories only (workspace, home, /tmp). Perfect for multi-session projects, experiment tracking, and maintaining productivity.
---

# Task Runner

Manage tasks and projects across sessions with persistent tracking.

## Quick Start

### Add a task
```bash
task_runner.py add "<description>" [project] [priority]
```

### List all tasks
```bash
task_runner.py list
```

### List tasks for a specific project
```bash
task_runner.py list "<project>"
```

### Complete a task
```bash
task_runner.py complete <task_id>
```

### Change task priority
```bash
task_runner.py priority <task_id> <low|medium|high>
```

### Export project to markdown
```bash
task_runner.py export "<project>" "<output_file>"
```

## Features

- **Persistent storage** - Tasks survive session restarts (stored in `~/.openclaw/workspace/tasks_db.json`)
- **Project organization** - Group tasks by project for better organization
- **Priority levels** - low, medium (default), high
- **Status tracking** - pending vs completed with timestamps
- **Flexible filtering** - View all tasks or filter by project
- **Markdown export** - Export projects to clean markdown for sharing

## Security

### Path Validation (v1.0.1+)
The `export` function validates output paths to prevent malicious writes:
- ✅ Allowed: `~/.openclaw/workspace/`, `/tmp/`, and home directory
- ❌ Blocked: System paths (`/etc/`, `/usr/`, `/var/`, etc.)
- ❌ Blocked: Sensitive dotfiles (`~/.bashrc`, `~/.ssh`, etc.)

This prevents prompt injection attacks that could attempt to write to system files for privilege escalation.

### Task Storage
The task storage is JSON-based and only writes to `~/.openclaw/workspace/tasks_db.json`.

## Usage Patterns

### For multi-session projects
```bash
# Add experiment tasks
task_runner.py add "Setup development environment" "project-x" "high"
task_runner.py add "Write initial tests" "project-x" "medium"
task_runner.py add "Document API endpoints" "project-x" "low"

# List project progress
task_runner.py list "project-x"

# Complete tasks as you go
task_runner.py complete 1
task_runner.py complete 2
```

### For autonomous agent workflows
Track your own tasks across sessions:
```bash
# Plan experiments
task_runner.py add "Build and publish skill" "income-experiments" "high"
task_runner.py add "Test content pipeline" "income-experiments" "medium"

# Update priorities based on learning
task_runner.py priority 2 "high"

# Export progress reports
task_runner.py export "income-experiments" "./progress-report.md"
```

### For sprint planning
```bash
# Plan week's work
task_runner.py.py add "Build feature X" "sprint-5" "high"
task_runner.py.py add "Fix bug Y" "sprint-5" "high"
task_runner.py.py add "Update documentation" "sprint-5" "medium"

# Review progress
task_runner.py list "sprint-5"

# Export for standup
task_runner.py export "sprint-5" "./standup.md"
```

## Task Priorities

| Priority | Emoji | When to Use |
|----------|-------|-------------|
| high | 🔴 | Blocking issues, urgent, must do now |
| medium | 🟡 | Normal work, do soon | 
| low | 🟢 | Nice to have, backlog items |

## Output Format

Task listing shows:
- Status icon (✅ completed, ⏳ pending)
- Project name
- Task ID number
- Priority emoji
- Creation date
- Task description
- Completion date (if completed)

## Export Format

Markdown export includes:
- Project title with task counts
- Pending tasks section
- Completed tasks section (most recent first)
- Task IDs, priorities, and timestamps

## Examples

### Managing a coding project
```bash
# Setup
task_runner.py add "Clone repository" "my-project" "high"
task_runner.py add "Install dependencies" "my-project" "high"
task_runner.py add "Set up database" "my-project" "medium"

# Track progress
task_runner.py list "my-project"
task_runner.py complete 1
task_runner.py complete 2

# Export for documentation
task_runner.py export "my-project" "./my-project-tasks.md"
```

### Tracking autonomous agent experiments
```bash
# Plan experiments
task_runner.py add "Experiment 1: Publish skills" "autonomous-income" "high"
task_runner.py add "Experiment 2: Content automation" "autonomous-income" "medium"
task_runner.py add "Experiment 3: Service MVP" "autonomous-income" "low"

# Work through them
task_runner.py list "autonomous-income"
task_runner.py complete 1

# Adjust based on learning
task_runner.py add "Experiment 2a: Research tools without API keys" "autonomous-income" "high"
task_runner.py priority 2 "low"
```

### Daily task management
```bash
# Plan the day
task_runner.py add "Review pull requests" "daily" "high"
task_runner.py add "Write documentation" "daily" "medium"
task_runner.py add "Respond to emails" "daily" "low"

# End-of-day review
task_runner.py list

# Archive completed work
task_runner.py export "daily" "./$(date +%Y-%m-%d)-tasks.md"
```

## Best Practices

1. **Use meaningful project names** - `income-experiments` not `ideas`
2. **Set priorities consistently** - helps with focus
3. **Mark tasks complete promptly** - keeps list clean
4. **Export before major changes** - backup progress
5. **Review and clean up** - archive old projects regularly

## Integration with Other Skills

Combine with **research-assistant** for complete project management:
- Use `research-assistant` for notes and knowledge
- Use `task-runner` for actionable tasks
- Export both to create comprehensive project docs
