#!/usr/bin/env node
/**
 * update-dashboard.js
 * 
 * Actualiza DASHBOARD.md con el estado actual de las sesiones de OpenClaw.
 * Este script se ejecuta vía cron para mantener el dashboard sincronizado.
 */

const fs = require('fs');
const path = require('path');

const DASHBOARD_PATH = path.join(__dirname, '../../DASHBOARD.md');
const SESSION_DATA_PATH = process.argv[2]; // Opcional: ruta a JSON con datos de sesiones

/**
 * Formatea un timestamp a fecha legible
 */
function formatDate(timestamp) {
  if (!timestamp) return 'N/A';
  const date = new Date(timestamp);
  return date.toISOString().replace('T', ' ').substring(0, 19) + ' UTC';
}

/**
 * Genera el contenido del dashboard
 */
function generateDashboard(sessions) {
  const now = new Date().toISOString().replace('T', ' ').substring(0, 19) + ' UTC';
  
  let md = `# DASHBOARD.md\n\n`;
  md += `**Última actualización:** ${now}\n\n`;
  md += `---\n\n`;
  md += `## Sesiones Activas (${sessions.count})\n\n`;

  if (sessions.count === 0) {
    md += `*No hay sesiones activas.*\n\n`;
    return md;
  }

  // Agrupar por tipo
  const byType = {
    'cron': [],
    'main': [],
    'discord': [],
    'subagent': [],
    'other': []
  };

  sessions.sessions.forEach(s => {
    if (s.key.includes('cron:')) byType.cron.push(s);
    else if (s.key.includes('discord:')) byType.discord.push(s);
    else if (s.key.includes('subagent:')) byType.subagent.push(s);
    else if (s.key === 'agent:main:main') byType.main.push(s);
    else byType.other.push(s);
  });

  // Main session
  if (byType.main.length > 0) {
    md += `### 🏠 Sesión Principal\n\n`;
    byType.main.forEach(s => {
      md += `- **${s.displayName || s.key}**\n`;
      md += `  - Modelo: \`${s.model || 'N/A'}\`\n`;
      md += `  - Tokens: ${s.totalTokens || 0}\n`;
      md += `  - Última actividad: ${formatDate(s.updatedAt)}\n\n`;
    });
  }

  // Discord sessions
  if (byType.discord.length > 0) {
    md += `### 💬 Sesiones Discord\n\n`;
    byType.discord.forEach(s => {
      md += `- **${s.displayName || s.key}**\n`;
      md += `  - Tokens: ${s.totalTokens || 0}\n`;
      md += `  - Última actividad: ${formatDate(s.updatedAt)}\n\n`;
    });
  }

  // Cron jobs
  if (byType.cron.length > 0) {
    md += `### ⏱️ Tareas Programadas (Cron)\n\n`;
    byType.cron.forEach(s => {
      const jobId = s.key.split(':').pop();
      md += `- **Cron Job:** \`${jobId.substring(0, 8)}...\`\n`;
      md += `  - Modelo: \`${s.model || 'N/A'}\`\n`;
      md += `  - Tokens: ${s.totalTokens || 0}\n`;
      md += `  - Última ejecución: ${formatDate(s.updatedAt)}\n\n`;
    });
  }

  // Subagents
  if (byType.subagent.length > 0) {
    md += `### 🤖 Sub-agentes\n\n`;
    byType.subagent.forEach(s => {
      md += `- **${s.displayName || s.key}**\n`;
      md += `  - Tokens: ${s.totalTokens || 0}\n`;
      md += `  - Última actividad: ${formatDate(s.updatedAt)}\n\n`;
    });
  }

  // Other
  if (byType.other.length > 0) {
    md += `### 📋 Otras Sesiones\n\n`;
    byType.other.forEach(s => {
      md += `- **${s.displayName || s.key}**\n`;
      md += `  - Tokens: ${s.totalTokens || 0}\n`;
      md += `  - Última actividad: ${formatDate(s.updatedAt)}\n\n`;
    });
  }

  md += `---\n\n`;
  md += `*Dashboard generado automáticamente por task-monitor*\n`;

  return md;
}

/**
 * Main
 */
async function main() {
  let sessions;

  if (SESSION_DATA_PATH && fs.existsSync(SESSION_DATA_PATH)) {
    // Leer desde archivo JSON
    const data = fs.readFileSync(SESSION_DATA_PATH, 'utf8');
    sessions = JSON.parse(data);
  } else {
    // Leer desde stdin (tool result)
    console.error('ERROR: No session data provided');
    process.exit(1);
  }

  const dashboardContent = generateDashboard(sessions);
  fs.writeFileSync(DASHBOARD_PATH, dashboardContent, 'utf8');
  
  console.log(`✅ DASHBOARD.md actualizado: ${sessions.count} sesiones`);
}

main().catch(err => {
  console.error('ERROR:', err.message);
  process.exit(1);
});
