#!/usr/bin/env node

const express = require('express');
const { exec } = require('child_process');
const { promisify } = require('util');
const fs = require('fs');
const path = require('path');

const execAsync = promisify(exec);

const app = express();
const PORT = 3030;
const HOST = '0.0.0.0';

// Cache configuration
const CACHE_TTL_MS = 30000; // 30 seconds
const CRON_CACHE_TTL_MS = 300000; // 5 minutes (cron jobs change rarely)
let cache = {
  data: null,
  timestamp: 0,
  isRefreshing: false
};
let cronJobsCache = {
  jobs: [],
  timestamp: 0
};

// Fetch fresh data
async function fetchStatusData() {
  try {
    const { stdout: output } = await execAsync('openclaw sessions list --json', { timeout: 10000 });
    const data = JSON.parse(output);
    const sessions = data.sessions || [];

    // Get cron job names (with persistent cache)
    let cronJobs = cronJobsCache.jobs; // Start with cached value
    const cronCacheAge = Date.now() - cronJobsCache.timestamp;
    
    // Only refresh if cache is old or empty
    if (cronJobsCache.jobs.length === 0 || cronCacheAge > CRON_CACHE_TTL_MS) {
      try {
        const { stdout: cronOutput } = await execAsync('openclaw cron list --json', { timeout: 10000 });
        const cronData = JSON.parse(cronOutput);
        const freshJobs = cronData.jobs || [];
        
        if (freshJobs.length > 0) {
          cronJobsCache.jobs = freshJobs;
          cronJobsCache.timestamp = Date.now();
          cronJobs = freshJobs;
          console.log(`Cron cache updated: ${freshJobs.length} jobs (age was ${Math.round(cronCacheAge / 1000)}s)`);
        } else {
          console.warn('Cron list returned empty, keeping cached data');
        }
      } catch (e) {
        console.warn(`Failed to load cron jobs (using cached: ${cronJobsCache.jobs.length}):`, e.message);
        // Keep existing cronJobs from cache
      }
    } else {
      console.log(`Using cached cron jobs (${cronJobs.length}, age: ${Math.round(cronCacheAge / 1000)}s)`);
    }

    // Sort by updatedAt descending
    sessions.sort((a, b) => b.updatedAt - a.updatedAt);

    // Classify sessions
    const mainSession = sessions.find(s => s.key === 'agent:main:main');
    const discordSession = sessions.find(s => s.key.includes(':discord:channel:'));
    const subagents = sessions.filter(s => s.key.includes(':subagent:'));
    const cronSessions = sessions.filter(s => s.key.includes(':cron:')).slice(0, 5);

    // Enhance subagents with descriptions
    const enhancedSubagents = subagents.map(s => ({
      ...s,
      description: getTaskDescription(s.sessionId)
    }));

    // Enhance cron sessions with job names
    const enhancedCrons = cronSessions.map(s => {
      const jobId = s.key.split(':cron:')[1];
      const cronJob = cronJobs.find(j => j.id === jobId);
      const name = cronJob ? cronJob.name : 'Unknown Job';
      if (!cronJob && cronJobs.length > 0) {
        console.log(`No match for jobId: ${jobId}, available IDs: ${cronJobs.slice(0, 3).map(j => j.id).join(', ')}`);
      }
      return {
        ...s,
        name: name
      };
    });

    return {
      timestamp: new Date().toISOString(),
      totalSessions: data.count,
      main: mainSession,
      discord: discordSession,
      subagents: enhancedSubagents,
      recentCrons: enhancedCrons
    };
  } catch (error) {
    console.error('Error fetching status data:', error);
    throw error;
  }
}

// Background refresh
async function refreshCache() {
  if (cache.isRefreshing) return;
  
  cache.isRefreshing = true;
  try {
    const data = await fetchStatusData();
    cache.data = data;
    cache.timestamp = Date.now();
    console.log(`Cache refreshed at ${new Date().toISOString()}`);
  } catch (error) {
    console.error('Cache refresh failed:', error.message);
  } finally {
    cache.isRefreshing = false;
  }
}

// API endpoint with caching
app.get('/api/status', async (req, res) => {
  try {
    const now = Date.now();
    const cacheAge = now - cache.timestamp;
    
    // Return cached data if valid
    if (cache.data && cacheAge < CACHE_TTL_MS) {
      console.log(`Cache hit (age: ${Math.round(cacheAge / 1000)}s)`);
      return res.json({
        ...cache.data,
        cached: true,
        cacheAge: Math.round(cacheAge / 1000)
      });
    }
    
    // Cache expired or missing - trigger refresh
    if (!cache.isRefreshing) {
      refreshCache(); // Fire and forget
    }
    
    // If we have stale data, return it while refreshing in background
    if (cache.data) {
      console.log(`Returning stale cache (age: ${Math.round(cacheAge / 1000)}s) while refreshing`);
      return res.json({
        ...cache.data,
        cached: true,
        cacheAge: Math.round(cacheAge / 1000),
        refreshing: true
      });
    }
    
    // No cache at all - wait for fresh data
    console.log('No cache available, fetching fresh data...');
    const data = await fetchStatusData();
    cache.data = data;
    cache.timestamp = Date.now();
    
    res.json({
      ...data,
      cached: false
    });
  } catch (error) {
    console.error('Error in /api/status:', error);
    res.status(500).json({ 
      error: error.message,
      timestamp: new Date().toISOString()
    });
  }
});

// HTML Dashboard
app.get('/', (req, res) => {
  res.send(`<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>🤖 Task Monitor</title>
  <style>
    * {
      margin: 0;
      padding: 0;
      box-sizing: border-box;
    }
    body {
      font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
      background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
      color: #fff;
      padding: 1rem;
      min-height: 100vh;
    }
    .container {
      max-width: 1200px;
      margin: 0 auto;
    }
    header {
      text-align: center;
      margin-bottom: 2rem;
      padding: 1rem;
      background: rgba(255,255,255,0.1);
      border-radius: 12px;
      backdrop-filter: blur(10px);
    }
    h1 {
      font-size: 2rem;
      margin-bottom: 0.5rem;
    }
    .timestamp {
      font-size: 0.9rem;
      opacity: 0.8;
    }
    .grid {
      display: grid;
      grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
      gap: 1rem;
      margin-bottom: 1rem;
    }
    .card {
      background: rgba(255,255,255,0.15);
      border-radius: 12px;
      padding: 1.5rem;
      backdrop-filter: blur(10px);
      box-shadow: 0 8px 32px rgba(0,0,0,0.1);
    }
    .card h2 {
      font-size: 1.3rem;
      margin-bottom: 1rem;
      display: flex;
      align-items: center;
      gap: 0.5rem;
    }
    .stat {
      margin-bottom: 0.8rem;
      padding-bottom: 0.8rem;
      border-bottom: 1px solid rgba(255,255,255,0.2);
    }
    .stat:last-child {
      border-bottom: none;
    }
    .stat-label {
      font-size: 0.85rem;
      opacity: 0.8;
      margin-bottom: 0.3rem;
    }
    .stat-value {
      font-size: 1.1rem;
      font-weight: 600;
    }
    .badge {
      display: inline-block;
      padding: 0.25rem 0.75rem;
      border-radius: 12px;
      font-size: 0.8rem;
      font-weight: 600;
      background: rgba(255,255,255,0.2);
    }
    .badge-success {
      background: rgba(76, 217, 100, 0.3);
    }
    .badge-warning {
      background: rgba(255, 204, 0, 0.3);
    }
    .badge-error {
      background: rgba(255, 59, 48, 0.3);
    }
    .task-item {
      margin-bottom: 1rem;
      padding: 1rem;
      background: rgba(255,255,255,0.1);
      border-radius: 8px;
    }
    .task-item:last-child {
      margin-bottom: 0;
    }
    .task-desc {
      font-size: 0.9rem;
      opacity: 0.9;
      margin: 0.5rem 0;
    }
    .task-meta {
      font-size: 0.8rem;
      opacity: 0.7;
      display: flex;
      gap: 1rem;
      flex-wrap: wrap;
    }
    .empty {
      text-align: center;
      opacity: 0.6;
      padding: 2rem;
      font-style: italic;
    }
    @media (max-width: 768px) {
      body {
        padding: 0.5rem;
      }
      h1 {
        font-size: 1.5rem;
      }
      .grid {
        grid-template-columns: 1fr;
      }
      .card {
        padding: 1rem;
      }
    }
    .loading {
      text-align: center;
      padding: 2rem;
      opacity: 0.7;
    }
  </style>
</head>
<body>
  <div class="container">
    <header>
      <h1>🤖 OpenClaw Task Monitor</h1>
      <div class="timestamp" id="timestamp">Loading...</div>
    </header>
    
    <div id="content">
      <div class="loading">⏳ Loading status...</div>
    </div>
  </div>

  <script>
    async function fetchStatus() {
      try {
        const response = await fetch('/api/status');
        const data = await response.json();
        renderDashboard(data);
      } catch (error) {
        document.getElementById('content').innerHTML = 
          '<div class="card"><h2>❌ Error</h2><p>' + error.message + '</p></div>';
      }
    }

    function renderDashboard(data) {
      const timestamp = new Date(data.timestamp).toLocaleString();
      let statusText = 'Last updated: ' + timestamp;
      if (data.cached) {
        statusText += ' (cached: ' + data.cacheAge + 's ago';
        if (data.refreshing) {
          statusText += ', refreshing...';
        }
        statusText += ')';
      }
      document.getElementById('timestamp').textContent = statusText;

      let html = '<div class="grid">';

      // Main Session Card
      if (data.main) {
        const age = formatAge(data.main.ageMs);
        html += \`
          <div class="card">
            <h2>💻 Main Session</h2>
            <div class="stat">
              <div class="stat-label">Model</div>
              <div class="stat-value">\${data.main.model}</div>
            </div>
            <div class="stat">
              <div class="stat-label">Tokens</div>
              <div class="stat-value">\${(data.main.totalTokens / 1000).toFixed(1)}k / \${(data.main.contextTokens / 1000).toFixed(0)}k</div>
            </div>
            <div class="stat">
              <div class="stat-label">Age</div>
              <div class="stat-value">\${age}</div>
            </div>
          </div>
        \`;
      }

      // Discord Session Card
      if (data.discord) {
        const age = formatAge(data.discord.ageMs);
        html += \`
          <div class="card">
            <h2>💬 Discord Session</h2>
            <div class="stat">
              <div class="stat-label">Model</div>
              <div class="stat-value">\${data.discord.model}</div>
            </div>
            <div class="stat">
              <div class="stat-label">Tokens</div>
              <div class="stat-value">\${(data.discord.totalTokens / 1000).toFixed(1)}k / \${(data.discord.contextTokens / 1000).toFixed(0)}k</div>
            </div>
            <div class="stat">
              <div class="stat-label">Age</div>
              <div class="stat-value">\${age}</div>
            </div>
          </div>
        \`;
      }

      // Summary Card
      html += \`
        <div class="card">
          <h2>📊 Summary</h2>
          <div class="stat">
            <div class="stat-label">Total Sessions</div>
            <div class="stat-value">\${data.totalSessions}</div>
          </div>
          <div class="stat">
            <div class="stat-label">Active Tasks</div>
            <div class="stat-value">\${data.subagents.length}</div>
          </div>
          <div class="stat">
            <div class="stat-label">Recent Crons</div>
            <div class="stat-value">\${data.recentCrons.length}</div>
          </div>
        </div>
      \`;

      html += '</div>';

      // Background Tasks
      html += '<div class="card">';
      html += '<h2>🛠️ Background Tasks</h2>';
      if (data.subagents.length > 0) {
        data.subagents.forEach(task => {
          const age = formatAge(task.ageMs);
          const status = task.abortedLastRun ? 
            '<span class="badge badge-error">Aborted</span>' : 
            '<span class="badge badge-success">Running</span>';
          html += \`
            <div class="task-item">
              <div><strong>Task \${task.sessionId.substring(0, 8)}</strong> \${status}</div>
              <div class="task-desc">\${task.description}</div>
              <div class="task-meta">
                <span>Model: \${task.model}</span>
                <span>Age: \${age}</span>
                <span>Tokens: \${task.totalTokens}</span>
              </div>
            </div>
          \`;
        });
      } else {
        html += '<div class="empty">No active background tasks</div>';
      }
      html += '</div>';

      // Recent Cron Jobs
      html += '<div class="card">';
      html += '<h2>🕒 Recent Cron Jobs</h2>';
      if (data.recentCrons.length > 0) {
        data.recentCrons.forEach(cron => {
          const lastRun = new Date(cron.updatedAt).toLocaleString();
          const status = cron.abortedLastRun ? 
            '<span class="badge badge-error">Error</span>' : 
            '<span class="badge badge-success">OK</span>';
          const jobName = cron.name || 'Unknown Job';
          html += \`
            <div class="task-item">
              <div><strong>\${jobName}</strong> \${status}</div>
              <div class="task-meta">
                <span>Last run: \${lastRun}</span>
                <span>Model: \${cron.model}</span>
              </div>
            </div>
          \`;
        });
      } else {
        html += '<div class="empty">No recent cron jobs</div>';
      }
      html += '</div>';

      document.getElementById('content').innerHTML = html;
    }

    function formatAge(ms) {
      const sec = Math.floor(ms / 1000);
      const min = Math.floor(sec / 60);
      const hr = Math.floor(min / 60);
      const day = Math.floor(hr / 24);

      if (day > 0) return \`\${day}d \${hr % 24}h\`;
      if (hr > 0) return \`\${hr}h \${min % 60}m\`;
      if (min > 0) return \`\${min}m \${sec % 60}s\`;
      return \`\${sec}s\`;
    }

    // Initial fetch
    fetchStatus();

    // Auto-refresh every 60 seconds
    setInterval(fetchStatus, 60000);
  </script>
</body>
</html>`);
});

// Helper function
function getTaskDescription(sessionId) {
  try {
    const transcriptPath = path.join(
      process.env.HOME,
      '.openclaw/agents/main/sessions',
      `${sessionId}.jsonl`
    );

    if (!fs.existsSync(transcriptPath)) {
      return 'Unknown task (transcript not found)';
    }

    const content = fs.readFileSync(transcriptPath, 'utf-8');
    const lines = content.trim().split('\n');

    if (lines.length === 0) return 'Empty transcript';

    // Find first user message in new format
    for (const line of lines) {
      const entry = JSON.parse(line);
      
      // New format: { type: "message", message: { role: "user", content: [...] } }
      if (entry.type === 'message' && entry.message?.role === 'user') {
        const contentArray = entry.message.content;
        if (Array.isArray(contentArray) && contentArray.length > 0) {
          const textContent = contentArray.find(c => c.type === 'text');
          if (textContent?.text) {
            const text = textContent.text.substring(0, 100);
            return text + (textContent.text.length > 100 ? '...' : '');
          }
        }
      }
      
      // Old format fallback: { role: "user", content: "..." }
      if (entry.role === 'user' && entry.content) {
        const text = entry.content.substring(0, 100);
        return text + (entry.content.length > 100 ? '...' : '');
      }
    }

    return 'Task description unavailable';
  } catch (error) {
    return `Error: ${error.message}`;
  }
}

// Start server
app.listen(PORT, HOST, async () => {
  console.log(`🤖 Task Monitor running at http://${HOST}:${PORT}`);
  console.log(`Local access: http://localhost:${PORT}`);
  console.log(`LAN access: http://<your-ip>:${PORT}`);
  
  // Pre-warm cache on startup
  console.log('Pre-warming cache...');
  await refreshCache();
  console.log('Cache ready');
});
