#!/usr/bin/env node

const { execSync } = require('child_process');
const fs = require('fs');
const path = require('path');

try {
  // 1. Get sessions
  const output = execSync('openclaw sessions list --json', { encoding: 'utf-8' });
  const data = JSON.parse(output);
  const sessions = data.sessions || [];

  // 2. Filter & Sort
  // Sort by updatedAt descending
  sessions.sort((a, b) => b.updatedAt - a.updatedAt);

  // 3. Generate Markdown
  let md = '# 🚀 OpenClaw Task Dashboard\n\n';
  md += `**Last Updated:** ${new Date().toLocaleString()}\n`;
  md += `**Total Sessions:** ${data.count}\n\n`;

  // Active Sub-agents (Tasks)
  const subagents = sessions.filter(s => s.key.includes(':subagent:'));
  
  if (subagents.length > 0) {
    md += '## 🛠️ Active Background Tasks\n\n';
    
    subagents.forEach(s => {
      const age = formatAge(s.ageMs);
      const status = s.abortedLastRun ? '❌ Aborted' : '✅ Running/Idle';
      const shortId = s.sessionId.substring(0, 8);
      const description = getTaskDescription(s.sessionId);
      
      md += `### 📌 Task: \`${shortId}\`\n`;
      md += `**Description:** ${description}\n`;
      md += `**Model:** ${s.model} | **Age:** ${age} | **Status:** ${status} | **Tokens:** ${s.totalTokens}\n\n`;
    });
  } else {
    md += '## 🛠️ Active Background Tasks\n\n_No active sub-agents._\n\n';
  }

  // Cron Jobs (Recent)
  const crons = sessions.filter(s => s.key.includes(':cron:'));
  if (crons.length > 0) {
    md += '## 🕒 Recent Cron Jobs\n\n';
    md += '| Job ID | Last Run | Model |\n';
    md += '|--------|----------|-------|\n';
    
    crons.slice(0, 5).forEach(s => { // Show top 5
       const jobId = s.key.split(':').pop();
       const date = new Date(s.updatedAt).toLocaleString();
       md += `| \`${jobId}\` | ${date} | ${s.model} |\n`;
    });
    md += '\n';
  }

  // Write to file
  const dashboardPath = path.resolve(process.env.HOME, 'clawd', 'DASHBOARD.md');
  fs.writeFileSync(dashboardPath, md);
  
  console.log(`Dashboard updated at: ${dashboardPath}`);

} catch (error) {
  console.error('Error generating dashboard:', error);
  process.exit(1);
}

function formatAge(ms) {
  const sec = Math.floor(ms / 1000);
  const min = Math.floor(sec / 60);
  const hr = Math.floor(min / 60);
  
  if (hr > 0) return `${hr}h ${min % 60}m`;
  if (min > 0) return `${min}m ${sec % 60}s`;
  return `${sec}s`;
}

function getTaskDescription(sessionId) {
  try {
    const transcriptPath = path.join(
      process.env.HOME,
      '.openclaw/agents/main/sessions',
      `${sessionId}.jsonl`
    );
    
    if (!fs.existsSync(transcriptPath)) {
      return 'Unknown task (transcript not found)';
    }
    
    // Read first line of JSONL (user's initial message)
    const content = fs.readFileSync(transcriptPath, 'utf-8');
    const lines = content.trim().split('\n');
    
    if (lines.length === 0) return 'Empty transcript';
    
    const firstMessage = JSON.parse(lines[0]);
    
    // Look for user message
    if (firstMessage.role === 'user' && firstMessage.content) {
      // Truncate to 80 chars
      const text = firstMessage.content.substring(0, 80);
      return text + (firstMessage.content.length > 80 ? '...' : '');
    }
    
    return 'Task description unavailable';
  } catch (error) {
    return `Error reading task: ${error.message}`;
  }
}
