#!/usr/bin/env python3
"""
SendGrid Webhook Server for Hour Meter

Receives SendGrid email events (opens, clicks, bounces, unsubscribes, spam reports)
and posts notifications to a configured Discord channel via OpenClaw.

Usage:
    # Start the webhook server
    python sendgrid_webhook.py --port 8089 --channel 1234567890
    
    # Or use environment variables
    WEBHOOK_PORT=8089 WEBHOOK_CHANNEL=1234567890 python sendgrid_webhook.py

Environment Variables:
    SENDGRID_WEBHOOK_PUBLIC_KEY - For signature verification (optional but recommended)
    SENDGRID_WEBHOOK_MAX_AGE_SECONDS - Max age for timestamp validation (default: 300)
    WEBHOOK_PORT - Server port (default: 8089)
    WEBHOOK_CHANNEL - Discord channel ID for notifications
    WEBHOOK_LOG_FILE - Path to log file (default: ~/.openclaw/sendgrid-webhook.log)
    OPENCLAW_GATEWAY_URL - OpenClaw gateway URL (default: http://localhost:4440)
    OPENCLAW_GATEWAY_TOKEN - OpenClaw gateway token for sending messages

Attribution:
    Concept & Design: Ross (@rm289)
    Implementation: Claude (OpenClaw Agent)
    Based on patterns from AIRM project
"""

import argparse
import base64
import hashlib
import json
import os
import sys
import time
import urllib.request
import urllib.error
from datetime import datetime, timezone
from http.server import HTTPServer, BaseHTTPRequestHandler
from pathlib import Path
from typing import Optional, Tuple, Dict, Any, List

# Optional cryptography imports for signature verification
try:
    from cryptography.hazmat.primitives.asymmetric import ec
    from cryptography.hazmat.primitives import hashes
    from cryptography.hazmat.primitives.serialization import load_der_public_key, load_pem_public_key
    from cryptography.exceptions import InvalidSignature
    CRYPTO_AVAILABLE = True
except ImportError:
    CRYPTO_AVAILABLE = False

# Default configuration
DEFAULT_PORT = 8089
DEFAULT_LOG_FILE = os.path.expanduser("~/.openclaw/sendgrid-webhook.log")
DEFAULT_GATEWAY_URL = "http://localhost:4440"
DEFAULT_MAX_AGE_SECONDS = 300

# Event type emoji mapping
EVENT_EMOJI = {
    "delivered": "✅",
    "open": "👀",
    "click": "🔗",
    "bounce": "⚠️",
    "dropped": "🚫",
    "deferred": "⏳",
    "unsubscribe": "🔕",
    "group_unsubscribe": "🔕",
    "spamreport": "🚨",
    "processed": "📤",
}

# Configuration holder
config: Dict[str, Any] = {}


def log(message: str, level: str = "INFO") -> None:
    """Log to file and stdout."""
    timestamp = datetime.now(timezone.utc).strftime("%Y-%m-%d %H:%M:%S UTC")
    log_line = f"[{timestamp}] [{level}] {message}"
    print(log_line)
    
    log_file = config.get("log_file", DEFAULT_LOG_FILE)
    if log_file:
        try:
            Path(log_file).parent.mkdir(parents=True, exist_ok=True)
            with open(log_file, "a") as f:
                f.write(log_line + "\n")
        except Exception as e:
            print(f"[WARN] Could not write to log file: {e}")


def get_webhook_public_key() -> str:
    """Get SendGrid webhook public key from environment."""
    return os.environ.get("SENDGRID_WEBHOOK_PUBLIC_KEY", "").strip()


def get_max_age_seconds() -> int:
    """Get max age for timestamp validation."""
    raw = os.environ.get("SENDGRID_WEBHOOK_MAX_AGE_SECONDS", "").strip()
    if not raw:
        return DEFAULT_MAX_AGE_SECONDS
    try:
        return max(0, int(raw))
    except ValueError:
        return DEFAULT_MAX_AGE_SECONDS


def validate_timestamp(timestamp_value: str) -> bool:
    """Validate webhook timestamp is within acceptable range."""
    max_age = get_max_age_seconds()
    if max_age <= 0:
        return True
    try:
        ts = int(timestamp_value)
    except (TypeError, ValueError):
        return False
    return abs(int(time.time()) - ts) <= max_age


def verify_signature(payload: bytes, signature: str, timestamp: str, public_key: str) -> bool:
    """Verify SendGrid webhook signature using ECDSA."""
    if not CRYPTO_AVAILABLE:
        log("Cryptography library not available - skipping signature verification", "WARN")
        return True
    
    if not public_key or not signature or not timestamp:
        return False
    
    key_text = public_key.strip()
    if not key_text:
        return False
    
    try:
        if "BEGIN PUBLIC KEY" in key_text:
            public_key_obj = load_pem_public_key(key_text.encode("utf-8"))
        else:
            key_bytes = base64.b64decode(key_text)
            public_key_obj = load_der_public_key(key_bytes)
        sig_bytes = base64.b64decode(signature)
    except (TypeError, ValueError) as e:
        log(f"Failed to decode signature or key: {e}", "ERROR")
        return False
    
    signed_payload = timestamp.encode("utf-8") + payload
    try:
        public_key_obj.verify(sig_bytes, signed_payload, ec.ECDSA(hashes.SHA256()))
        return True
    except InvalidSignature:
        return False
    except Exception as e:
        log(f"Signature verification error: {e}", "ERROR")
        return False


def format_event_message(event: Dict[str, Any]) -> str:
    """Format a SendGrid event as a Discord message."""
    event_type = event.get("event", "unknown")
    emoji = EVENT_EMOJI.get(event_type, "📧")
    recipient = event.get("email", "unknown")
    timestamp = event.get("timestamp")
    
    # Format timestamp in Eastern Time
    time_str = ""
    if timestamp:
        try:
            from zoneinfo import ZoneInfo
            dt = datetime.fromtimestamp(int(timestamp), tz=timezone.utc)
            dt_eastern = dt.astimezone(ZoneInfo("America/New_York"))
            tz_name = "EDT" if dt_eastern.dst() else "EST"
            time_str = dt_eastern.strftime(f'%Y-%m-%d %H:%M:%S {tz_name}')
        except (TypeError, ValueError):
            pass
    
    # Simple events: one line
    if event_type in ("delivered", "processed", "open", "deferred"):
        parts = [f"{emoji} **{event_type.upper()}**", f"`{recipient}`"]
        if time_str:
            parts.append(time_str)
        return " | ".join(parts)
    
    # Complex events: multi-line
    lines = [f"{emoji} **{event_type.upper()}** | `{recipient}`"]
    
    if event_type == "click":
        url = event.get("url", "N/A")
        lines.append(f"🔗 {url[:80]}..." if len(url) > 80 else f"🔗 {url}")
    
    if event_type == "bounce":
        reason = event.get("reason", "Unknown")
        bounce_type = event.get("type", "unknown")
        lines.append(f"⚠️ {bounce_type}: {reason[:100]}")
    
    if event_type in ("unsubscribe", "group_unsubscribe"):
        asm_group = event.get("asm_group_id", "N/A")
        lines.append(f"🔕 Group: {asm_group}")
    
    if event_type == "spamreport":
        lines.append("🚨 Marked as spam!")
    
    if time_str:
        lines[0] += f" | {time_str}"
    
    return "\n".join(lines)


def send_to_discord(message: str) -> Tuple[bool, str]:
    """Send a message to Discord via webhook URL or OpenClaw gateway."""
    
    # Method 1: Direct Discord Webhook (preferred - no dependencies)
    discord_webhook_url = config.get("discord_webhook_url")
    if discord_webhook_url:
        payload = {
            "content": message,
            "username": "TARDIS",
            "avatar_url": "https://upload.wikimedia.org/wikipedia/commons/thumb/7/76/Tardis_BBC.svg/240px-Tardis_BBC.svg.png"
        }
        data = json.dumps(payload).encode("utf-8")
        
        try:
            req = urllib.request.Request(
                discord_webhook_url,
                data=data,
                headers={
                    "Content-Type": "application/json",
                    "User-Agent": "TARDIS-Webhook/1.0"
                },
                method="POST"
            )
            with urllib.request.urlopen(req, timeout=10) as response:
                return True, "Message sent via Discord webhook"
        except urllib.error.HTTPError as e:
            error_body = e.read().decode("utf-8") if e.fp else str(e)
            return False, f"Discord webhook HTTP {e.code}: {error_body}"
        except urllib.error.URLError as e:
            return False, f"Discord webhook network error: {e.reason}"
        except Exception as e:
            return False, f"Discord webhook error: {e}"
    
    # Method 2: OpenClaw Gateway API
    channel_id = config.get("channel")
    if not channel_id:
        return False, "No channel or webhook URL configured"
    
    gateway_url = config.get("gateway_url", DEFAULT_GATEWAY_URL)
    gateway_token = config.get("gateway_token", "")
    
    payload = {
        "action": "send",
        "channel": "discord",
        "target": str(channel_id),
        "message": message
    }
    
    data = json.dumps(payload).encode("utf-8")
    
    headers = {"Content-Type": "application/json"}
    if gateway_token:
        headers["Authorization"] = f"Bearer {gateway_token}"
    
    try:
        req = urllib.request.Request(
            f"{gateway_url}/api/message",
            data=data,
            headers=headers,
            method="POST"
        )
        with urllib.request.urlopen(req, timeout=10) as response:
            return True, "Message sent via gateway"
    except urllib.error.HTTPError as e:
        error_body = e.read().decode("utf-8") if e.fp else str(e)
        return False, f"Gateway HTTP {e.code}: {error_body}"
    except urllib.error.URLError as e:
        return False, f"Gateway network error: {e.reason}"
    except Exception as e:
        return False, f"Gateway error: {e}"


def write_event_to_log(event: Dict[str, Any]) -> None:
    """Write event to a JSON log file for persistence."""
    events_file = os.path.expanduser("~/.openclaw/sendgrid-events.jsonl")
    try:
        Path(events_file).parent.mkdir(parents=True, exist_ok=True)
        with open(events_file, "a") as f:
            record = {
                "received_at": int(time.time()),
                "event": event
            }
            f.write(json.dumps(record) + "\n")
    except Exception as e:
        log(f"Could not write event to log: {e}", "WARN")


class WebhookHandler(BaseHTTPRequestHandler):
    """HTTP request handler for SendGrid webhooks."""
    
    def log_message(self, format: str, *args) -> None:
        """Override to use our logging."""
        log(f"HTTP: {format % args}", "DEBUG")
    
    def do_GET(self) -> None:
        """Handle GET requests - health check."""
        if self.path == "/health" or self.path == "/":
            self.send_response(200)
            self.send_header("Content-Type", "application/json")
            self.end_headers()
            response = {
                "status": "ok",
                "service": "hour-meter-sendgrid-webhook",
                "channel": config.get("channel"),
                "signature_verification": bool(get_webhook_public_key())
            }
            self.wfile.write(json.dumps(response).encode())
        else:
            self.send_response(404)
            self.end_headers()
    
    def do_POST(self) -> None:
        """Handle POST requests - webhook events."""
        if self.path != "/webhooks/sendgrid":
            self.send_response(404)
            self.end_headers()
            return
        
        # Read body
        content_length = int(self.headers.get("Content-Length", 0))
        raw_body = self.rfile.read(content_length)
        
        # Signature verification
        public_key = get_webhook_public_key()
        if public_key:
            signature = self.headers.get("X-Twilio-Email-Event-Webhook-Signature", "").strip()
            timestamp = self.headers.get("X-Twilio-Email-Event-Webhook-Timestamp", "").strip()
            
            if not signature or not timestamp:
                log("Webhook denied: missing signature headers", "WARN")
                self.send_response(403)
                self.send_header("Content-Type", "application/json")
                self.end_headers()
                self.wfile.write(json.dumps({"status": "denied", "reason": "missing_signature"}).encode())
                return
            
            if not validate_timestamp(timestamp):
                log("Webhook denied: timestamp invalid", "WARN")
                self.send_response(403)
                self.send_header("Content-Type", "application/json")
                self.end_headers()
                self.wfile.write(json.dumps({"status": "denied", "reason": "timestamp_invalid"}).encode())
                return
            
            if not verify_signature(raw_body, signature, timestamp, public_key):
                log("Webhook denied: signature invalid", "WARN")
                self.send_response(403)
                self.send_header("Content-Type", "application/json")
                self.end_headers()
                self.wfile.write(json.dumps({"status": "denied", "reason": "signature_invalid"}).encode())
                return
            
            log("Webhook signature verified ✓")
        else:
            log("Webhook received (no signature verification configured)", "WARN")
        
        # Parse payload
        try:
            payload_str = raw_body.decode("utf-8", errors="replace").strip()
            if not payload_str:
                raise ValueError("Empty payload")
            parsed = json.loads(payload_str)
        except (json.JSONDecodeError, ValueError) as e:
            log(f"Webhook error: invalid JSON - {e}", "ERROR")
            self.send_response(400)
            self.send_header("Content-Type", "application/json")
            self.end_headers()
            self.wfile.write(json.dumps({"status": "error", "reason": "invalid_json"}).encode())
            return
        
        # Process events
        events = parsed if isinstance(parsed, list) else [parsed]
        processed = 0
        
        for event in events:
            if not isinstance(event, dict):
                continue
            
            event_type = event.get("event", "unknown")
            recipient = event.get("email", "unknown")
            log(f"Processing event: {event_type} for {recipient}")
            
            # Write to persistent log
            write_event_to_log(event)
            
            # Format and send to Discord
            message = format_event_message(event)
            success, result = send_to_discord(message)
            
            if success:
                log(f"Posted to Discord: {event_type}")
                processed += 1
            else:
                log(f"Failed to post to Discord: {result}", "ERROR")
        
        # Respond
        self.send_response(200)
        self.send_header("Content-Type", "application/json")
        self.end_headers()
        self.wfile.write(json.dumps({"status": "ok", "processed": processed}).encode())


def run_server(port: int) -> None:
    """Run the webhook server."""
    server_address = ("", port)
    httpd = HTTPServer(server_address, WebhookHandler)
    
    log(f"Starting SendGrid webhook server on port {port}")
    log(f"Webhook endpoint: http://localhost:{port}/webhooks/sendgrid")
    log(f"Health check: http://localhost:{port}/health")
    log(f"Discord channel: {config.get('channel', 'NOT SET')}")
    log(f"Signature verification: {'ENABLED' if get_webhook_public_key() else 'DISABLED'}")
    
    try:
        httpd.serve_forever()
    except KeyboardInterrupt:
        log("Shutting down webhook server")
        httpd.shutdown()


def get_pending_events(mark_processed: bool = False) -> List[Dict[str, Any]]:
    """Get pending events from the events log file."""
    events_file = os.path.expanduser("~/.openclaw/sendgrid-events.jsonl")
    processed_file = os.path.expanduser("~/.openclaw/sendgrid-events-processed.txt")
    
    if not os.path.exists(events_file):
        return []
    
    # Load processed event IDs
    processed_ids = set()
    if os.path.exists(processed_file):
        with open(processed_file, "r") as f:
            processed_ids = set(line.strip() for line in f if line.strip())
    
    pending = []
    with open(events_file, "r") as f:
        for line in f:
            if not line.strip():
                continue
            try:
                record = json.loads(line)
                event = record.get("event", {})
                event_id = event.get("sg_event_id") or f"{record.get('received_at')}_{event.get('email', '')}"
                
                if event_id not in processed_ids:
                    pending.append({
                        "id": event_id,
                        "received_at": record.get("received_at"),
                        "event": event
                    })
                    if mark_processed:
                        processed_ids.add(event_id)
            except json.JSONDecodeError:
                continue
    
    # Write back processed IDs
    if mark_processed and pending:
        with open(processed_file, "w") as f:
            f.write("\n".join(sorted(processed_ids)))
    
    return pending


def cmd_process_events(output_format: str = "text") -> int:
    """Process pending events and output them."""
    pending = get_pending_events(mark_processed=True)
    
    if not pending:
        if output_format == "json":
            print(json.dumps({"events": [], "count": 0}))
        else:
            print("No pending events.")
        return 0
    
    if output_format == "json":
        print(json.dumps({"events": pending, "count": len(pending)}, indent=2))
    else:
        for record in pending:
            event = record["event"]
            print(format_event_message(event))
            print("-" * 40)
    
    return 0


def main() -> int:
    parser = argparse.ArgumentParser(
        description="SendGrid Webhook Server for Hour Meter",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
    # Start server with Discord webhook (recommended)
    python sendgrid_webhook.py --port 8089 --discord-webhook https://discord.com/api/webhooks/xxx/yyy
    
    # Start server with channel ID (requires gateway)
    python sendgrid_webhook.py --port 8089 --channel 1234567890
    
    # Process pending events (for agent to post)
    python sendgrid_webhook.py --process-events
    python sendgrid_webhook.py --process-events --json
    
    # Test health endpoint
    curl http://localhost:8089/health
    
SendGrid Setup:
    1. Go to SendGrid > Settings > Mail Settings > Event Webhook
    2. Set the HTTP POST URL to: https://your-domain.com/webhooks/sendgrid
    3. Select events: Delivered, Opens, Clicks, Bounces, Unsubscribes, Spam Reports
    4. (Optional) Enable Signed Event Webhook and set SENDGRID_WEBHOOK_PUBLIC_KEY

Discord Webhook Setup (recommended):
    1. In Discord channel, go to Settings > Integrations > Webhooks
    2. Create a webhook, copy the URL
    3. Pass to --discord-webhook or set DISCORD_WEBHOOK_URL env var
        """
    )
    
    parser.add_argument(
        "--port", "-p",
        type=int,
        default=int(os.environ.get("WEBHOOK_PORT", DEFAULT_PORT)),
        help=f"Server port (default: {DEFAULT_PORT})"
    )
    parser.add_argument(
        "--channel", "-c",
        default=os.environ.get("WEBHOOK_CHANNEL", ""),
        help="Discord channel ID for notifications (via gateway)"
    )
    parser.add_argument(
        "--discord-webhook",
        default=os.environ.get("DISCORD_WEBHOOK_URL", ""),
        help="Discord webhook URL (direct posting, no gateway needed)"
    )
    parser.add_argument(
        "--gateway-url",
        default=os.environ.get("OPENCLAW_GATEWAY_URL", DEFAULT_GATEWAY_URL),
        help=f"OpenClaw gateway URL (default: {DEFAULT_GATEWAY_URL})"
    )
    parser.add_argument(
        "--gateway-token",
        default=os.environ.get("OPENCLAW_GATEWAY_TOKEN", ""),
        help="OpenClaw gateway token"
    )
    parser.add_argument(
        "--log-file",
        default=os.environ.get("WEBHOOK_LOG_FILE", DEFAULT_LOG_FILE),
        help=f"Log file path (default: {DEFAULT_LOG_FILE})"
    )
    parser.add_argument(
        "--process-events",
        action="store_true",
        help="Process pending events from log file and exit"
    )
    parser.add_argument(
        "--json",
        action="store_true",
        help="Output in JSON format (with --process-events)"
    )
    
    args = parser.parse_args()
    
    # Process events mode
    if args.process_events:
        return cmd_process_events("json" if args.json else "text")
    
    # Validate required config for server mode
    if not args.channel and not args.discord_webhook:
        print("ERROR: Need --channel or --discord-webhook for server mode.", file=sys.stderr)
        print("       Or use --process-events to just process pending events.", file=sys.stderr)
        return 1
    
    # Set global config
    config["port"] = args.port
    config["channel"] = args.channel
    config["discord_webhook_url"] = args.discord_webhook
    config["gateway_url"] = args.gateway_url
    config["gateway_token"] = args.gateway_token
    config["log_file"] = args.log_file
    
    # Run server
    run_server(args.port)
    return 0


if __name__ == "__main__":
    sys.exit(main())
