#!/usr/bin/env python3
"""
TARDIS - Tamper-evident elapsed time tracker with milestones (formerly Hour Meter)

Three modes:
- COUNT UP: Time since an event (quit smoking, project start)
- COUNT DOWN: Time until an event (baby due date, deadline)
- COUNT BETWEEN: Journey from start to end (career, pregnancy, project)

With milestone notifications and multiple tamper-evidence storage options.
"""

import json
import hashlib
import time
import argparse
import os
import sys
import base64
import urllib.request
import urllib.error
from datetime import datetime, timezone
from pathlib import Path

# Default storage location
DEFAULT_STORAGE = os.path.expanduser("~/.openclaw/meters.json")

# Auto-load .env file if SENDGRID_API_KEY not in environment
def _load_dotenv():
    """Load .env file if SENDGRID_API_KEY not already set."""
    if os.environ.get("SENDGRID_API_KEY"):
        return
    env_paths = [
        os.path.expanduser("~/.env"),
        os.path.expanduser("/root/.env"),
        ".env"
    ]
    for env_path in env_paths:
        if os.path.exists(env_path):
            try:
                with open(env_path) as f:
                    for line in f:
                        line = line.strip()
                        if line and not line.startswith("#") and "=" in line:
                            key, value = line.split("=", 1)
                            key = key.strip()
                            value = value.strip().strip('"').strip("'")
                            if key and key not in os.environ:
                                os.environ[key] = value
                break
            except Exception:
                pass

_load_dotenv()

# Witness file location (for cloud sync)
WITNESS_FILE = os.path.expanduser("~/.openclaw/meter-witness.txt")

def get_storage_path():
    """Get the meters storage file path."""
    return Path(os.environ.get("METER_STORAGE", DEFAULT_STORAGE))

def load_meters():
    """Load meters from storage."""
    path = get_storage_path()
    if path.exists():
        with open(path, 'r') as f:
            return json.load(f)
    return {"meters": {}}

def save_meters(data):
    """Save meters to storage."""
    path = get_storage_path()
    path.parent.mkdir(parents=True, exist_ok=True)
    with open(path, 'w') as f:
        json.dump(data, f, indent=2)

def compute_hash(name: str, epoch_ms: int, salt: str) -> str:
    """Compute SHA-256 integrity hash."""
    payload = f"{name}:{epoch_ms}:{salt}"
    return hashlib.sha256(payload.encode()).hexdigest()

def hash_to_paper_code(full_hash: str) -> str:
    """
    Convert full hash to human-writable paper code.
    Format: XXXX-XXXX-XXXX-XXXX-C (16 chars + checksum)
    
    Uses first 16 hex chars (64 bits) - plenty for personal use.
    Checksum digit catches transcription errors.
    """
    short = full_hash[:16].upper()
    
    # Simple checksum: sum of char values mod 36, encoded as 0-9A-Z
    checksum_val = sum(ord(c) for c in short) % 36
    if checksum_val < 10:
        checksum = str(checksum_val)
    else:
        checksum = chr(ord('A') + checksum_val - 10)
    
    # Format as XXXX-XXXX-XXXX-XXXX-C
    formatted = f"{short[0:4]}-{short[4:8]}-{short[8:12]}-{short[12:16]}-{checksum}"
    return formatted

def paper_code_to_hash_prefix(paper_code: str) -> tuple[str, bool]:
    """
    Validate paper code and extract hash prefix.
    Returns (hash_prefix, is_valid).
    """
    # Remove dashes and spaces
    clean = paper_code.replace("-", "").replace(" ", "").upper()
    
    if len(clean) != 17:
        return None, False
    
    short = clean[:16]
    provided_checksum = clean[16]
    
    # Verify checksum
    checksum_val = sum(ord(c) for c in short) % 36
    if checksum_val < 10:
        expected_checksum = str(checksum_val)
    else:
        expected_checksum = chr(ord('A') + checksum_val - 10)
    
    if provided_checksum != expected_checksum:
        return short, False  # Return prefix anyway for debugging
    
    return short.lower(), True

def generate_qr_code(data: str, filename: str) -> bool:
    """
    Generate QR code PNG if qrcode library available.
    Returns True if successful.
    """
    try:
        import qrcode
        qr = qrcode.QRCode(version=1, box_size=10, border=4)
        qr.add_data(data)
        qr.make(fit=True)
        img = qr.make_image(fill_color="black", back_color="white")
        img.save(filename)
        return True
    except ImportError:
        return False

def append_witness(meter_name: str, paper_code: str, full_hash: str, timestamp: str):
    """Append to witness file for cloud sync verification."""
    witness_path = Path(os.environ.get("METER_WITNESS", WITNESS_FILE))
    witness_path.parent.mkdir(parents=True, exist_ok=True)
    
    with open(witness_path, 'a') as f:
        f.write(f"\n{'='*60}\n")
        f.write(f"METER: {meter_name}\n")
        f.write(f"LOCKED: {timestamp}\n")
        f.write(f"PAPER CODE: {paper_code}\n")
        f.write(f"FULL HASH: {full_hash}\n")
        f.write(f"{'='*60}\n")
    
    return witness_path

def parse_time(time_str: str) -> int:
    """Parse time string to milliseconds."""
    try:
        dt = datetime.fromisoformat(time_str.replace('Z', '+00:00'))
        return int(dt.timestamp() * 1000)
    except ValueError:
        try:
            return int(float(time_str) * 1000)
        except ValueError:
            raise ValueError(f"Cannot parse time '{time_str}'")

def format_elapsed(seconds: float, short: bool = False) -> str:
    """Format elapsed time in human-readable form."""
    if seconds < 0:
        return "not started yet" if not short else "-"
    
    days = int(seconds // 86400)
    hours = int((seconds % 86400) // 3600)
    minutes = int((seconds % 3600) // 60)
    secs = int(seconds % 60)
    
    if short:
        if days > 0:
            return f"{days}d {hours}h"
        elif hours > 0:
            return f"{hours}h {minutes}m"
        else:
            return f"{minutes}m {secs}s"
    
    parts = []
    if days > 0:
        parts.append(f"{days}d")
    if hours > 0 or days > 0:
        parts.append(f"{hours}h")
    if minutes > 0 or hours > 0 or days > 0:
        parts.append(f"{minutes}m")
    parts.append(f"{secs}s")
    
    return " ".join(parts)

def format_hours(seconds: float) -> str:
    """Format as decimal hours."""
    if seconds < 0:
        return "0.0 hours"
    hours = seconds / 3600
    return f"{hours:,.1f} hours"

def format_money(amount: float) -> str:
    """Format as currency."""
    return f"${amount:,.2f}"

def format_progress_bar(percent: float, width: int = 20) -> str:
    """Create a text progress bar."""
    filled = int(width * percent / 100)
    empty = width - filled
    bar = "█" * filled + "░" * empty
    return f"[{bar}] {percent:.1f}%"

def cmd_create(args):
    """Create a new meter."""
    data = load_meters()
    
    if args.name in data["meters"]:
        print(f"Error: Meter '{args.name}' already exists.", file=sys.stderr)
        sys.exit(1)
    
    # Parse start time
    if args.start:
        try:
            start_ms = parse_time(args.start)
        except ValueError as e:
            print(f"Error: {e}", file=sys.stderr)
            sys.exit(1)
    else:
        start_ms = int(time.time() * 1000)
    
    # Parse end time
    end_ms = None
    if args.end:
        try:
            end_ms = parse_time(args.end)
        except ValueError as e:
            print(f"Error: {e}", file=sys.stderr)
            sys.exit(1)
    
    # Determine mode
    if args.mode:
        mode = args.mode
    elif end_ms:
        mode = "between"
    else:
        mode = "up"
    
    if mode in ["down", "between"] and not end_ms:
        print(f"Error: Mode '{mode}' requires --end time.", file=sys.stderr)
        sys.exit(1)
    
    salt = hashlib.sha256(os.urandom(32)).hexdigest()[:16]
    
    meter = {
        "name": args.name,
        "description": args.description or "",
        "start_ms": start_ms,
        "end_ms": end_ms,
        "created_ms": int(time.time() * 1000),
        "locked": False,
        "locked_ms": None,
        "salt": salt,
        "integrity_hash": None,
        "paper_code": None,
        "mode": mode,
        "milestones": [],
        "notify_channel": args.channel,
        "notify_target": args.target,
        "notify_email": args.notify_email,
        "notify_from_email": args.from_email
    }
    
    data["meters"][args.name] = meter
    save_meters(data)
    
    start_dt = datetime.fromtimestamp(start_ms / 1000, tz=timezone.utc)
    print(f"✓ Created meter '{args.name}'")
    print(f"  Mode: {mode.upper()}")
    print(f"  Start: {start_dt.isoformat()}")
    if end_ms:
        end_dt = datetime.fromtimestamp(end_ms / 1000, tz=timezone.utc)
        print(f"  End: {end_dt.isoformat()}")
    print(f"  Status: UNLOCKED")

def cmd_milestone(args):
    """Add a milestone to a meter."""
    data = load_meters()
    
    if args.name not in data["meters"]:
        print(f"Error: Meter '{args.name}' not found.", file=sys.stderr)
        sys.exit(1)
    
    meter = data["meters"][args.name]
    
    milestone = {
        "id": hashlib.sha256(os.urandom(8)).hexdigest()[:8],
        "type": args.type,
        "value": args.value,
        "message": args.message,
        "fired": False,
        "fired_ms": None
    }
    
    meter["milestones"].append(milestone)
    save_meters(data)
    
    if args.type == "hours":
        print(f"✓ Added milestone: notify at {args.value:,.0f} hours")
    else:
        print(f"✓ Added milestone: notify at {args.value}%")

def cmd_milestones_list(args):
    """List milestones for a meter."""
    data = load_meters()
    
    if args.name not in data["meters"]:
        print(f"Error: Meter '{args.name}' not found.", file=sys.stderr)
        sys.exit(1)
    
    meter = data["meters"][args.name]
    
    if not meter["milestones"]:
        print(f"No milestones for meter '{args.name}'")
        return
    
    print(f"Milestones for '{args.name}':")
    for m in meter["milestones"]:
        status = "✓" if m["fired"] else "○"
        val = f"{m['value']:,.0f}h" if m["type"] == "hours" else f"{m['value']}%"
        print(f"  {status} {val}: {m['message']}")

def cmd_check_milestones(args):
    """Check all meters for milestone triggers AND timer completions."""
    data = load_meters()
    now_ms = int(time.time() * 1000)
    triggered = []
    email_results = []
    
    for name, meter in data["meters"].items():
        elapsed_ms = now_ms - meter["start_ms"]
        elapsed_hours = elapsed_ms / (1000 * 3600)
        elapsed_str = format_elapsed(elapsed_ms / 1000)
        
        mode = meter.get("mode", "up")
        percent = None
        
        # Get email settings for this meter
        notify_email = meter.get("notify_email")
        notify_from_email = meter.get("notify_from_email")
        
        if mode in ["down", "between"] and meter.get("end_ms"):
            total_ms = meter["end_ms"] - meter["start_ms"]
            percent = (elapsed_ms / total_ms) * 100 if total_ms > 0 else 0
            
            # Check for timer completion (countdown/between reached end)
            if now_ms >= meter["end_ms"] and not meter.get("completed_fired"):
                meter["completed_fired"] = True
                desc = meter.get("description", name)
                if mode == "down":
                    msg = f"⏰ COUNTDOWN COMPLETE: {desc}"
                else:
                    msg = f"🏁 JOURNEY COMPLETE: {desc} (100%)"
                
                triggered.append({
                    "meter": name,
                    "milestone_id": "_completion_",
                    "type": "completion",
                    "value": 100,
                    "message": msg,
                    "channel": meter.get("notify_channel"),
                    "target": meter.get("notify_target"),
                    "notify_email": notify_email,
                    "description": desc
                })
                
                # Send email notification if configured
                if notify_email:
                    success, result = send_milestone_email(
                        notify_email, name, msg, elapsed_str, desc, notify_from_email
                    )
                    email_results.append({"meter": name, "email": notify_email, "success": success, "result": result})
        
        # Check milestones
        for m in meter.get("milestones", []):
            if m["fired"]:
                continue
            
            should_fire = False
            if m["type"] == "hours" and elapsed_hours >= m["value"]:
                should_fire = True
            elif m["type"] == "percent" and percent is not None and percent >= m["value"]:
                should_fire = True
            
            if should_fire:
                m["fired"] = True
                m["fired_ms"] = now_ms
                triggered.append({
                    "meter": name,
                    "milestone_id": m["id"],
                    "type": m["type"],
                    "value": m["value"],
                    "message": m["message"],
                    "channel": meter.get("notify_channel"),
                    "target": meter.get("notify_target"),
                    "notify_email": notify_email,
                    "description": meter.get("description", "")
                })
                
                # Send email notification if configured
                if notify_email:
                    success, result = send_milestone_email(
                        notify_email, name, m["message"], elapsed_str, 
                        meter.get("description", ""), notify_from_email
                    )
                    email_results.append({"meter": name, "email": notify_email, "success": success, "result": result})
    
    if triggered:
        save_meters(data)
    
    output = {"triggered": triggered}
    if email_results:
        output["email_notifications"] = email_results
    
    print(json.dumps(output, indent=2))

def send_email_sendgrid(to_email: str, subject: str, html_content: str, text_content: str,
                        from_email: str = None) -> tuple[bool, str]:
    """
    Send email via SendGrid API.
    Requires SENDGRID_API_KEY environment variable.
    Returns (success, message).
    """
    api_key = os.environ.get("SENDGRID_API_KEY")
    if not api_key:
        return False, "SENDGRID_API_KEY environment variable not set"
    
    # Default from email
    if not from_email:
        from_email = os.environ.get("SENDGRID_FROM_EMAIL", "tardis@noreply.example.com")
    
    payload = {
        "personalizations": [{"to": [{"email": to_email}]}],
        "from": {"email": from_email, "name": "TARDIS"},
        "subject": subject,
        "content": [
            {"type": "text/plain", "value": text_content},
            {"type": "text/html", "value": html_content}
        ],
        "tracking_settings": {
            "subscription_tracking": {
                "enable": True,
                "substitution_tag": "<%asm_group_unsubscribe_url%>"
            }
        }
    }
    
    data = json.dumps(payload).encode('utf-8')
    
    req = urllib.request.Request(
        "https://api.sendgrid.com/v3/mail/send",
        data=data,
        headers={
            "Authorization": f"Bearer {api_key}",
            "Content-Type": "application/json"
        },
        method="POST"
    )
    
    try:
        with urllib.request.urlopen(req) as response:
            return True, f"Email sent to {to_email}"
    except urllib.error.HTTPError as e:
        error_body = e.read().decode('utf-8') if e.fp else str(e)
        return False, f"SendGrid error {e.code}: {error_body}"
    except urllib.error.URLError as e:
        return False, f"Network error: {e.reason}"


def generate_verification_email(meter_name: str, paper_code: str, full_hash: str,
                                 description: str, lock_time: str) -> tuple[str, str, str]:
    """Generate email subject, HTML content, and text content for verification email."""
    
    subject = f"🔒 TARDIS Verification Code: {meter_name}"
    
    text_content = f"""TARDIS VERIFICATION CODE
{'='*40}

Meter: {meter_name}
Description: {description or 'N/A'}
Locked: {lock_time}

📋 PAPER CODE (save this!):

    {paper_code}

To verify later, run:
    meter.py verify {meter_name} "{paper_code}"

Full hash: {full_hash}

---
This code proves the meter hasn't been tampered with.
Keep this email - you'll need the paper code to verify.

Sent by TARDIS


To unsubscribe: <%asm_group_unsubscribe_raw_url%>
"""
    
    html_content = f"""
<!DOCTYPE html>
<html>
<head>
    <style>
        body {{ font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; padding: 20px; background: #f5f5f5; }}
        .container {{ max-width: 600px; margin: 0 auto; background: white; border-radius: 10px; padding: 30px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }}
        .header {{ text-align: center; margin-bottom: 30px; }}
        .header h1 {{ color: #333; margin: 0; }}
        .code-box {{ background: #1a1a2e; color: #00ff88; padding: 30px; border-radius: 8px; text-align: center; margin: 20px 0; }}
        .code {{ font-family: 'SF Mono', Monaco, monospace; font-size: 28px; letter-spacing: 3px; font-weight: bold; }}
        .details {{ background: #f8f9fa; padding: 20px; border-radius: 8px; margin: 20px 0; }}
        .details table {{ width: 100%; border-collapse: collapse; }}
        .details td {{ padding: 8px 0; }}
        .details td:first-child {{ color: #666; width: 120px; }}
        .verify-cmd {{ background: #2d2d2d; color: #e0e0e0; padding: 15px; border-radius: 5px; font-family: monospace; font-size: 14px; overflow-x: auto; }}
        .footer {{ text-align: center; color: #999; font-size: 12px; margin-top: 30px; padding-top: 20px; border-top: 1px solid #eee; }}
        .hash {{ font-family: monospace; font-size: 11px; color: #999; word-break: break-all; }}
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>🔒 Meter Locked</h1>
            <p style="color: #666;">Your verification code is below</p>
        </div>
        
        <div class="code-box">
            <div class="code">{paper_code}</div>
            <p style="color: #888; margin: 10px 0 0 0; font-size: 14px;">Paper Code</p>
        </div>
        
        <div class="details">
            <table>
                <tr><td>Meter</td><td><strong>{meter_name}</strong></td></tr>
                <tr><td>Description</td><td>{description or 'N/A'}</td></tr>
                <tr><td>Locked</td><td>{lock_time}</td></tr>
            </table>
        </div>
        
        <h3>To verify later:</h3>
        <div class="verify-cmd">
            meter.py verify {meter_name} "{paper_code}"
        </div>
        
        <div class="footer">
            <p>This code proves your meter hasn't been tampered with.</p>
            <p>Keep this email safe - you'll need the paper code to verify.</p>
            <p class="hash">Full hash: {full_hash}</p>
            <p style="margin-top: 15px;">⏱️ Sent by TARDIS</p>
            <br><br>
            <p style="font-size: 11px; color: #aaa;">
                <a href="<%asm_group_unsubscribe_url%>" style="color: #aaa;">Click here to unsubscribe</a> from TARDIS notifications.
            </p>
        </div>
    </div>
</body>
</html>
"""
    
    return subject, html_content, text_content


def send_milestone_email(to_email: str, meter_name: str, milestone_message: str,
                         elapsed_str: str, description: str, from_email: str = None) -> tuple[bool, str]:
    """
    Send milestone notification email via SendGrid.
    Returns (success, message).
    """
    api_key = os.environ.get("SENDGRID_API_KEY")
    if not api_key:
        return False, "SENDGRID_API_KEY not set"
    
    if not from_email:
        from_email = os.environ.get("SENDGRID_FROM_EMAIL", "hour-meter@noreply.example.com")
    
    subject = f"🎯 Milestone: {meter_name}"
    
    text_content = f"""TARDIS MILESTONE REACHED!
{'='*40}

Meter: {meter_name}
Description: {description or 'N/A'}

🎯 MILESTONE:
{milestone_message}

⏱️ Elapsed: {elapsed_str}

---
Sent by TARDIS


To unsubscribe from these notifications: <%asm_group_unsubscribe_raw_url%>
"""
    
    html_content = f"""
<!DOCTYPE html>
<html>
<head>
    <style>
        body {{ font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; padding: 20px; background: #f5f5f5; }}
        .container {{ max-width: 600px; margin: 0 auto; background: white; border-radius: 10px; padding: 30px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }}
        .header {{ text-align: center; margin-bottom: 20px; }}
        .milestone-box {{ background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 25px; border-radius: 8px; text-align: center; margin: 20px 0; }}
        .milestone-msg {{ font-size: 24px; font-weight: bold; margin: 0; }}
        .details {{ background: #f8f9fa; padding: 15px; border-radius: 8px; margin: 20px 0; }}
        .footer {{ text-align: center; color: #999; font-size: 12px; margin-top: 20px; }}
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>🎯 Milestone Reached!</h1>
        </div>
        
        <div class="milestone-box">
            <p class="milestone-msg">{milestone_message}</p>
        </div>
        
        <div class="details">
            <p><strong>Meter:</strong> {meter_name}</p>
            <p><strong>Description:</strong> {description or 'N/A'}</p>
            <p><strong>Elapsed:</strong> {elapsed_str}</p>
        </div>
        
        <div class="footer">
            <p>⏱️ Sent by TARDIS</p>
            <br><br>
            <p style="font-size: 11px; color: #aaa;">
                <a href="<%asm_group_unsubscribe_url%>" style="color: #aaa;">Click here to unsubscribe</a> from TARDIS milestone notifications.
            </p>
        </div>
    </div>
</body>
</html>
"""
    
    payload = {
        "personalizations": [{"to": [{"email": to_email}]}],
        "from": {"email": from_email, "name": "TARDIS"},
        "subject": subject,
        "content": [
            {"type": "text/plain", "value": text_content},
            {"type": "text/html", "value": html_content}
        ],
        "tracking_settings": {
            "subscription_tracking": {
                "enable": True,
                "substitution_tag": "<%asm_group_unsubscribe_url%>"
            }
        }
    }
    
    data = json.dumps(payload).encode('utf-8')
    
    req = urllib.request.Request(
        "https://api.sendgrid.com/v3/mail/send",
        data=data,
        headers={
            "Authorization": f"Bearer {api_key}",
            "Content-Type": "application/json"
        },
        method="POST"
    )
    
    try:
        with urllib.request.urlopen(req) as response:
            return True, f"Milestone email sent to {to_email}"
    except urllib.error.HTTPError as e:
        error_body = e.read().decode('utf-8') if e.fp else str(e)
        return False, f"SendGrid error {e.code}: {error_body}"
    except urllib.error.URLError as e:
        return False, f"Network error: {e.reason}"


def generate_mailto_link(meter_name: str, paper_code: str, full_hash: str, 
                         description: str, lock_time: str) -> str:
    """Generate a mailto: link for emailing the verification code to yourself."""
    import urllib.parse
    
    subject = f"🔒 TARDIS Verification Code: {meter_name}"
    
    body = f"""TARDIS VERIFICATION CODE
============================

Meter: {meter_name}
Description: {description or 'N/A'}
Locked: {lock_time}

📋 PAPER CODE (save this!):

    {paper_code}

To verify later, run:
    meter.py verify {meter_name} "{paper_code}"

Full hash: {full_hash}

---
This code proves the meter hasn't been tampered with.
Keep this email - you'll need the paper code to verify.

Sent by TARDIS
"""
    
    mailto = f"mailto:?subject={urllib.parse.quote(subject)}&body={urllib.parse.quote(body)}"
    return mailto

def cmd_lock(args):
    """Lock a meter with tamper-evident hash."""
    data = load_meters()
    
    if args.name not in data["meters"]:
        print(f"Error: Meter '{args.name}' not found.", file=sys.stderr)
        sys.exit(1)
    
    meter = data["meters"][args.name]
    
    if meter["locked"]:
        print(f"Error: Meter '{args.name}' is already locked.", file=sys.stderr)
        sys.exit(1)
    
    full_hash = compute_hash(meter["name"], meter["start_ms"], meter["salt"])
    paper_code = hash_to_paper_code(full_hash)
    lock_time = datetime.now(timezone.utc)
    lock_time_str = lock_time.strftime('%Y-%m-%d %H:%M:%S UTC')
    
    meter["locked"] = True
    meter["locked_ms"] = int(time.time() * 1000)
    meter["integrity_hash"] = full_hash
    meter["paper_code"] = paper_code
    
    save_meters(data)
    
    print(f"🔒 LOCKED: {args.name}")
    print()
    print(f"╔══════════════════════════════════════════════════════════════╗")
    print(f"║  PAPER CODE (write this down):                               ║")
    print(f"║                                                              ║")
    print(f"║     {paper_code}                          ║")
    print(f"║                                                              ║")
    print(f"╚══════════════════════════════════════════════════════════════╝")
    print()
    print(f"📋 FOUR WAYS TO SAVE THIS:")
    print()
    print(f"   1️⃣  PAPER: Write the code above on paper/sticky note")
    print(f"       It has a checksum - we'll catch typos when you verify")
    print()
    print(f"   2️⃣  PHOTO: Take a screenshot or photo of this screen")
    print(f"       Store in your camera roll or cloud photos")
    print()
    print(f"   3️⃣  WITNESS FILE: Auto-saved to:")
    
    # Save to witness file
    witness_path = append_witness(
        args.name, 
        paper_code, 
        full_hash,
        lock_time.isoformat()
    )
    print(f"       {witness_path}")
    print(f"       (Sync this folder to Dropbox/iCloud/Google Drive)")
    print()
    
    # Generate mailto link
    mailto_link = generate_mailto_link(
        args.name, paper_code, full_hash,
        meter.get("description", ""), lock_time_str
    )
    print(f"   4️⃣  EMAIL TO SELF: Click or copy this link:")
    print(f"       {mailto_link[:80]}...")
    print()
    
    # Also output a compact version for easy copying
    print(f"   📧 Or copy this ready-to-paste message:")
    print(f"   ─────────────────────────────────────────")
    print(f"   🔒 {args.name} | Code: {paper_code} | Locked: {lock_time_str}")
    print(f"   ─────────────────────────────────────────")
    print()
    
    # Try to generate QR code
    qr_path = Path(f"{args.name}-verify.png")
    if generate_qr_code(f"meter:{args.name}:{paper_code}", str(qr_path)):
        print(f"   🔲 QR CODE: Saved to {qr_path}")
        print()
    
    print(f"   Full hash (for nerds): {full_hash}")
    print()
    print(f"   To verify later: meter.py verify {args.name} {paper_code}")
    
    # Send email if requested
    if hasattr(args, 'email') and args.email:
        print()
        print(f"   📧 Sending verification email...")
        subject, html_content, text_content = generate_verification_email(
            args.name, paper_code, full_hash,
            meter.get("description", ""), lock_time_str
        )
        from_email = getattr(args, 'from_email', None)
        success, message = send_email_sendgrid(args.email, subject, html_content, text_content, from_email)
        if success:
            print(f"   ✅ {message}")
        else:
            print(f"   ❌ {message}")

def cmd_check(args):
    """Check a meter's status."""
    data = load_meters()
    
    if args.name not in data["meters"]:
        print(f"Error: Meter '{args.name}' not found.", file=sys.stderr)
        sys.exit(1)
    
    meter = data["meters"][args.name]
    now_ms = int(time.time() * 1000)
    
    start_dt = datetime.fromtimestamp(meter["start_ms"] / 1000, tz=timezone.utc)
    elapsed_ms = now_ms - meter["start_ms"]
    elapsed_seconds = elapsed_ms / 1000
    
    mode = meter.get("mode", "up")
    
    print(f"⏱️  Meter: {meter['name']} ({mode.upper()})")
    if meter["description"]:
        print(f"   {meter['description']}")
    print()
    
    if mode == "up":
        print(f"   📍 Started: {start_dt.strftime('%Y-%m-%d %H:%M:%S')} UTC")
        print(f"   ⏱️  Elapsed: {format_elapsed(elapsed_seconds)}")
        print(f"   🕐 Hours:   {format_hours(elapsed_seconds)}")
        if elapsed_seconds > 86400:
            print(f"   📅 Days:    {elapsed_seconds / 86400:,.1f}")
        
    elif mode == "down":
        end_dt = datetime.fromtimestamp(meter["end_ms"] / 1000, tz=timezone.utc)
        remaining_ms = meter["end_ms"] - now_ms
        remaining_seconds = remaining_ms / 1000
        
        print(f"   🎯 Target: {end_dt.strftime('%Y-%m-%d %H:%M:%S')} UTC")
        
        if remaining_seconds <= 0:
            print(f"   ✅ COMPLETE! ({format_elapsed(-remaining_seconds)} ago)")
        else:
            print(f"   ⏳ Remaining: {format_elapsed(remaining_seconds)}")
            print(f"   🕐 Hours:     {format_hours(remaining_seconds)}")
        
    elif mode == "between":
        end_dt = datetime.fromtimestamp(meter["end_ms"] / 1000, tz=timezone.utc)
        total_ms = meter["end_ms"] - meter["start_ms"]
        remaining_ms = meter["end_ms"] - now_ms
        remaining_seconds = remaining_ms / 1000
        
        percent = (elapsed_ms / total_ms) * 100 if total_ms > 0 else 0
        percent = min(100, max(0, percent))
        
        print(f"   📍 Start:     {start_dt.strftime('%Y-%m-%d %H:%M:%S')} UTC")
        print(f"   🎯 End:       {end_dt.strftime('%Y-%m-%d %H:%M:%S')} UTC")
        print()
        print(f"   {format_progress_bar(percent)}")
        print()
        print(f"   ✅ Elapsed:   {format_elapsed(elapsed_seconds)} ({elapsed_seconds/3600:,.0f} hrs)")
        
        if remaining_seconds <= 0:
            print(f"   🎉 COMPLETE!")
        else:
            print(f"   ⏳ Remaining: {format_elapsed(remaining_seconds)} ({remaining_seconds/3600:,.0f} hrs)")
    
    if meter["milestones"]:
        print()
        print(f"   Milestones:")
        for m in meter["milestones"]:
            status = "✓" if m["fired"] else "○"
            val = f"{m['value']:,.0f}h" if m["type"] == "hours" else f"{m['value']}%"
            print(f"   {status} {val}: {m['message'][:40]}")
    
    print()
    if meter["locked"]:
        expected_hash = compute_hash(meter["name"], meter["start_ms"], meter["salt"])
        if expected_hash == meter["integrity_hash"]:
            print(f"   🔒 LOCKED ✓ (integrity verified)")
            print(f"   📋 Paper code: {meter.get('paper_code', 'N/A')}")
        else:
            print(f"   ⚠️  TAMPERED! Hash mismatch!")
    else:
        print(f"   🔓 UNLOCKED")

def cmd_verify(args):
    """Verify against paper code or full hash."""
    data = load_meters()
    
    if args.name not in data["meters"]:
        print(f"Error: Meter '{args.name}' not found.", file=sys.stderr)
        sys.exit(1)
    
    meter = data["meters"][args.name]
    
    if not meter["locked"]:
        print(f"Error: Meter '{args.name}' is not locked.", file=sys.stderr)
        sys.exit(1)
    
    expected_hash = compute_hash(meter["name"], meter["start_ms"], meter["salt"])
    input_code = args.code
    
    # Check if it's a paper code (has dashes or is ~17 chars) or full hash
    clean_input = input_code.replace("-", "").replace(" ", "")
    
    if len(clean_input) <= 20:
        # Treat as paper code
        prefix, checksum_valid = paper_code_to_hash_prefix(input_code)
        
        if not checksum_valid:
            print(f"⚠️  CHECKSUM ERROR!")
            print(f"   The paper code has a typo. Double-check what you wrote.")
            print(f"   You entered: {input_code}")
            if prefix:
                print(f"   Expected checksum for '{prefix}' doesn't match.")
            sys.exit(1)
        
        if expected_hash.startswith(prefix):
            print(f"✅ VERIFIED!")
            print(f"   Paper code matches. Meter '{args.name}' has NOT been tampered with.")
            print()
            print(f"   Paper code: {meter.get('paper_code')}")
            print(f"   Created:    {datetime.fromtimestamp(meter['start_ms']/1000, tz=timezone.utc).strftime('%Y-%m-%d')}")
        else:
            print(f"❌ MISMATCH!")
            print(f"   Paper code does NOT match stored hash.")
            print(f"   The meter may have been TAMPERED with!")
            print()
            print(f"   Your code:   {input_code}")
            print(f"   Stored code: {meter.get('paper_code')}")
            sys.exit(1)
    else:
        # Treat as full hash
        if clean_input.lower() == expected_hash.lower():
            print(f"✅ VERIFIED! Full hash matches.")
        else:
            print(f"❌ MISMATCH! Hashes do not match.")
            print(f"   External:  {input_code}")
            print(f"   Computed:  {expected_hash}")
            sys.exit(1)

def cmd_witness(args):
    """Show or manage the witness file."""
    witness_path = Path(os.environ.get("METER_WITNESS", WITNESS_FILE))
    
    if args.show:
        if witness_path.exists():
            print(f"📋 Witness file: {witness_path}")
            print()
            print(witness_path.read_text())
        else:
            print(f"No witness file found at {witness_path}")
    elif args.path:
        print(witness_path)
    else:
        print(f"Witness file location: {witness_path}")
        if witness_path.exists():
            lines = len(witness_path.read_text().strip().split('\n'))
            print(f"Status: exists ({lines} lines)")
        else:
            print(f"Status: not created yet (lock a meter to create)")

def cmd_list(args):
    """List all meters."""
    data = load_meters()
    
    if not data["meters"]:
        print("No meters found.")
        return
    
    now_ms = int(time.time() * 1000)
    
    print(f"{'Name':<20} {'Mode':<8} {'Status':<10} {'Progress':<25}")
    print("-" * 70)
    
    for name, meter in sorted(data["meters"].items()):
        mode = meter.get("mode", "up")
        status = "🔒" if meter["locked"] else "🔓"
        
        elapsed_ms = now_ms - meter["start_ms"]
        elapsed_hours = elapsed_ms / (1000 * 3600)
        
        if mode == "up":
            progress = f"{elapsed_hours:,.0f} hrs elapsed"
        elif mode == "down":
            remaining_ms = meter["end_ms"] - now_ms
            remaining_hours = remaining_ms / (1000 * 3600)
            if remaining_hours <= 0:
                progress = "COMPLETE"
            else:
                progress = f"{remaining_hours:,.0f} hrs remaining"
        else:
            total_ms = meter["end_ms"] - meter["start_ms"]
            pct = (elapsed_ms / total_ms) * 100 if total_ms > 0 else 0
            pct = min(100, max(0, pct))
            progress = f"{pct:.1f}% complete"
        
        print(f"{name:<20} {mode:<8} {status:<10} {progress:<25}")

def cmd_delete(args):
    """Delete a meter."""
    data = load_meters()
    
    if args.name not in data["meters"]:
        print(f"Error: Meter '{args.name}' not found.", file=sys.stderr)
        sys.exit(1)
    
    meter = data["meters"][args.name]
    
    if meter["locked"] and not args.force:
        print(f"Error: Meter '{args.name}' is locked. Use --force.", file=sys.stderr)
        sys.exit(1)
    
    del data["meters"][args.name]
    save_meters(data)
    print(f"✓ Deleted meter '{args.name}'")

def cmd_export(args):
    """Export meter data as JSON."""
    data = load_meters()
    
    if args.name:
        if args.name not in data["meters"]:
            print(f"Error: Meter '{args.name}' not found.", file=sys.stderr)
            sys.exit(1)
        export_data = {"meters": {args.name: data["meters"][args.name]}}
    else:
        export_data = data
    
    print(json.dumps(export_data, indent=2))

def cmd_career(args):
    """Career inventory projection."""
    data = load_meters()
    
    if args.meter:
        if args.meter not in data["meters"]:
            print(f"Error: Meter '{args.meter}' not found.", file=sys.stderr)
            sys.exit(1)
        meter = data["meters"][args.meter]
        now_ms = int(time.time() * 1000)
        elapsed_seconds = (now_ms - meter["start_ms"]) / 1000
        elapsed_years = elapsed_seconds / (365.25 * 24 * 3600)
        hours_worked = elapsed_years * args.hours_per_year
        career_start = datetime.fromtimestamp(meter["start_ms"] / 1000, tz=timezone.utc)
        print(f"📊 Career Inventory Projection")
        print(f"   Based on: {args.meter}")
        print(f"   Started:  {career_start.strftime('%Y-%m-%d')}")
    else:
        hours_worked = args.hours_worked or 0
        print(f"📊 Career Inventory Projection")
    
    total_hours = args.total_hours
    hours_per_year = args.hours_per_year
    hourly_rate = args.rate
    annual_raise = args.raise_pct / 100
    
    hours_remaining = max(0, total_hours - hours_worked)
    years_remaining = hours_remaining / hours_per_year
    years_worked = hours_worked / hours_per_year
    
    print()
    print(f"   ⏱️  Hours worked:     {hours_worked:>12,.0f} hrs ({years_worked:.1f} yrs)")
    print(f"   ⏳ Hours remaining:  {hours_remaining:>12,.0f} hrs ({years_remaining:.1f} yrs)")
    print(f"   📦 Total inventory:  {total_hours:>12,.0f} hrs ({total_hours/hours_per_year:.0f} yrs)")
    print()
    print(f"   {format_progress_bar((hours_worked/total_hours)*100)}")
    print()
    print(f"   💰 Current rate:     {format_money(hourly_rate)}/hr")
    print(f"   📈 Annual raise:     {args.raise_pct}%")
    print()
    
    total_earnings = 0
    current_rate = hourly_rate
    remaining = hours_remaining
    year = 1
    
    print(f"   {'Year':<6} {'Rate':>10} {'Earnings':>14} {'Cumulative':>14}")
    print(f"   " + "-" * 50)
    
    while remaining > 0:
        year_hours = min(hours_per_year, remaining)
        year_earnings = year_hours * current_rate
        total_earnings += year_earnings
        
        if year <= 3 or remaining <= hours_per_year or year % 5 == 0:
            print(f"   {year:<6} {format_money(current_rate):>10} {format_money(year_earnings):>14} {format_money(total_earnings):>14}")
        elif year == 4:
            print(f"   {'...':^6}")
        
        remaining -= year_hours
        current_rate *= (1 + annual_raise)
        year += 1
    
    print()
    print(f"   🎯 REMAINING EARNING POTENTIAL: {format_money(total_earnings)}")

def main():
    parser = argparse.ArgumentParser(description="TARDIS - Life event tracker")
    subparsers = parser.add_subparsers(dest="command", required=True)
    
    # Create
    p_create = subparsers.add_parser("create", help="Create a new meter")
    p_create.add_argument("name", help="Meter name")
    p_create.add_argument("--start", "-s", help="Start time (ISO/unix, default: now)")
    p_create.add_argument("--end", "-e", help="End time (for countdown/between)")
    p_create.add_argument("--description", "-d", help="Description")
    p_create.add_argument("--mode", "-m", choices=["up", "down", "between"])
    p_create.add_argument("--channel", help="Notification channel")
    p_create.add_argument("--target", help="Notification target")
    p_create.add_argument("--notify-email", help="Email address for milestone notifications (requires SENDGRID_API_KEY)")
    p_create.add_argument("--from-email", help="From email for notifications (default: SENDGRID_FROM_EMAIL)")
    p_create.set_defaults(func=cmd_create)
    
    # Milestone
    p_milestone = subparsers.add_parser("milestone", help="Add milestone")
    p_milestone.add_argument("name", help="Meter name")
    p_milestone.add_argument("--type", "-t", choices=["hours", "percent"], required=True)
    p_milestone.add_argument("--value", "-v", type=float, required=True)
    p_milestone.add_argument("--message", "-m", required=True)
    p_milestone.set_defaults(func=cmd_milestone)
    
    # Milestones list
    p_milestones = subparsers.add_parser("milestones", help="List milestones")
    p_milestones.add_argument("name", help="Meter name")
    p_milestones.set_defaults(func=cmd_milestones_list)
    
    # Check milestones
    p_check_ms = subparsers.add_parser("check-milestones", help="Check all milestones (JSON)")
    p_check_ms.set_defaults(func=cmd_check_milestones)
    
    # Lock
    p_lock = subparsers.add_parser("lock", help="Lock meter")
    p_lock.add_argument("name", help="Meter name")
    p_lock.add_argument("--email", "-e", help="Send verification code to this email address (requires SENDGRID_API_KEY)")
    p_lock.add_argument("--from-email", help="From email address (default: SENDGRID_FROM_EMAIL or hour-meter@noreply.example.com)")
    p_lock.set_defaults(func=cmd_lock)
    
    # Check
    p_check = subparsers.add_parser("check", help="Check meter status")
    p_check.add_argument("name", help="Meter name")
    p_check.set_defaults(func=cmd_check)
    
    # Verify
    p_verify = subparsers.add_parser("verify", help="Verify with paper code or hash")
    p_verify.add_argument("name", help="Meter name")
    p_verify.add_argument("code", help="Paper code (XXXX-XXXX-XXXX-XXXX-C) or full hash")
    p_verify.set_defaults(func=cmd_verify)
    
    # Witness
    p_witness = subparsers.add_parser("witness", help="Manage witness file")
    p_witness.add_argument("--show", action="store_true", help="Show witness file contents")
    p_witness.add_argument("--path", action="store_true", help="Just print path")
    p_witness.set_defaults(func=cmd_witness)
    
    # List
    p_list = subparsers.add_parser("list", help="List all meters")
    p_list.set_defaults(func=cmd_list)
    
    # Delete
    p_delete = subparsers.add_parser("delete", help="Delete meter")
    p_delete.add_argument("name", help="Meter name")
    p_delete.add_argument("--force", "-f", action="store_true")
    p_delete.set_defaults(func=cmd_delete)
    
    # Export
    p_export = subparsers.add_parser("export", help="Export as JSON")
    p_export.add_argument("name", nargs="?")
    p_export.set_defaults(func=cmd_export)
    
    # Career
    p_career = subparsers.add_parser("career", help="Career projection")
    p_career.add_argument("--meter", "-m")
    p_career.add_argument("--hours-worked", "-w", type=float)
    p_career.add_argument("--total-hours", "-t", type=float, default=80000)
    p_career.add_argument("--hours-per-year", "-y", type=float, default=2000)
    p_career.add_argument("--rate", "-r", type=float, default=50)
    p_career.add_argument("--raise-pct", "-p", type=float, default=2)
    p_career.set_defaults(func=cmd_career)
    
    args = parser.parse_args()
    args.func(args)

if __name__ == "__main__":
    main()
