# Hour Meter: Technical Whitepaper

**A Tamper-Evident Elapsed Time Tracker with Milestone Notifications**

*Version 1.0 — February 2026*

---

## Abstract

Hour Meter is an OpenClaw skill that provides tamper-evident elapsed time tracking with three operational modes, configurable milestone notifications, and human-verifiable integrity proofs. Inspired by analog Hobbs meters used in aviation and industrial equipment, it digitizes the concept while adding cryptographic verification and integration with modern notification channels.

This document covers the technical architecture, integrity model, operational modes, milestone system, and integration patterns.

---

## Table of Contents

1. [Technical Overview](#technical-overview)
2. [Operational Modes](#operational-modes)
3. [Integrity Model](#integrity-model)
4. [Paper Code System](#paper-code-system)
5. [Milestone Notifications](#milestone-notifications)
6. [Persistence Architecture](#persistence-architecture)
7. [Use Cases with Examples](#use-cases-with-examples)
8. [Integration Patterns](#integration-patterns)
9. [Cost Considerations](#cost-considerations)
10. [Security Analysis](#security-analysis)

---

## 1. Technical Overview

### Core Components

```
hour-meter/
├── SKILL.md           # OpenClaw skill definition
├── scripts/
│   └── meter.py       # Primary CLI (Python 3)
├── README.md          # User documentation
└── WHITEPAPER.md      # This document
```

### Data Model

Each meter is stored as a JSON object:

```json
{
  "name": "career-start",
  "description": "College graduation - career begins",
  "start_ms": 1274025600000,
  "end_ms": 2146435200000,
  "created_ms": 1738454400000,
  "locked": true,
  "locked_ms": 1738454500000,
  "salt": "a7f3b92c1d4e8f6a",
  "integrity_hash": "318b3229c5232f9c5f3731b1cea8dc9f7b5000b762e9ca57f907775580fa170f",
  "paper_code": "318B-3229-C523-2F9C-V",
  "mode": "between",
  "milestones": [...],
  "notify_channel": "discord",
  "notify_target": "YOUR_CHANNEL_ID",
  "completed_fired": false
}
```

### Storage Location

- **Primary:** `~/.openclaw/meters.json`
- **Witness Log:** `~/.openclaw/meter-witness.txt`
- **Override:** `METER_STORAGE` and `METER_WITNESS` environment variables

---

## 2. Operational Modes

### 2.1 Count Up (Elapsed Time)

Tracks time elapsed since a start epoch. Runs indefinitely.

**Trigger:** `--mode up` (default when no `--end` specified)

**Use cases:** Sobriety counters, project duration, equipment runtime, time since last incident

**Display:**
```
📍 Started: 2025-06-15 08:00:00 UTC
⏱️  Elapsed: 231d 18h 8m 41s
🕐 Hours:   5,562.1 hours
```

### 2.2 Count Down (Time Remaining)

Tracks time remaining until a target end time.

**Trigger:** `--mode down` (requires `--end`)

**Use cases:** Deadlines, due dates, countdowns to events

**Display:**
```
🎯 Target: 2026-10-15 00:00:00 UTC
⏳ Remaining: 254d 21h 51m 8s
🕐 Hours:     6,118.0 hours
```

**Completion:** Fires automatic notification when target is reached.

### 2.3 Count Between (Journey Progress)

Tracks progress through a defined time span with percentage complete.

**Trigger:** `--mode between` or automatic when both `--start` and `--end` provided

**Use cases:** Career span, pregnancy, project phases, fiscal years

**Display:**
```
📍 Start:     2010-05-15 00:00:00 UTC
🎯 End:       2050-05-15 00:00:00 UTC

[█████████████░░░░░░░] 69.3%

✅ Elapsed:   10125d 2h 8m 57s (243,002 hrs)
⏳ Remaining: 4484d 21h 51m 2s (107,638 hrs)
```

**Completion:** Fires automatic notification when 100% is reached.

---

## 3. Integrity Model

### Hash Computation

When a meter is locked, the system computes:

```
integrity_hash = SHA256(name + ":" + start_ms + ":" + salt)
```

- **name:** Meter identifier (string)
- **start_ms:** Start epoch in milliseconds (integer)
- **salt:** 16-character random hex string generated at creation

### Security Properties

| Property | Guarantee |
|----------|-----------|
| **Tamper Detection** | Any modification to name or start_ms changes the hash |
| **Preimage Resistance** | Cannot reverse-engineer the original values from hash |
| **Salt Uniqueness** | Each meter has a unique salt; identical meters produce different hashes |

### Verification Flow

```
1. User provides paper code or full hash
2. System recomputes: SHA256(stored_name + ":" + stored_start_ms + ":" + stored_salt)
3. Compare computed hash to provided/stored hash
4. Match → ✅ VERIFIED | Mismatch → ❌ TAMPERED
```

### Limitations

- **Local Trust:** The integrity model assumes the storage file itself is protected. An attacker with write access to `meters.json` could modify all fields including salt to create a valid hash.
- **Mitigation:** External hash storage (paper, email, cloud sync) provides independent verification.

---

## 4. Paper Code System

### Design Goals

1. **Human-Writable:** Short enough to write on paper without errors
2. **Error-Detecting:** Built-in checksum catches transcription mistakes
3. **Memorable Format:** Grouped with dashes for easy reading

### Format

```
XXXX-XXXX-XXXX-XXXX-C

Where:
- XXXX groups: First 16 hex characters of SHA-256 hash (uppercase)
- C: Single checksum character (0-9, A-Z)
```

### Checksum Algorithm

```python
checksum_val = sum(ord(c) for c in short_hash) % 36
if checksum_val < 10:
    checksum = str(checksum_val)
else:
    checksum = chr(ord('A') + checksum_val - 10)
```

### Error Detection

When verifying, the system:
1. Validates checksum before comparing hash
2. If checksum fails → "CHECKSUM ERROR: You have a typo"
3. If checksum passes but hash mismatches → "MISMATCH: Possible tampering"

This prevents false tampering alerts from transcription errors.

---

## 5. Milestone Notifications

### Milestone Types

| Type | Trigger | Example |
|------|---------|---------|
| **hours** | Elapsed hours from start | `1000` → fires at 1,000 hours |
| **percent** | Percentage of journey (between mode) | `75` → fires at 75% complete |
| **completion** | Auto-generated when timer ends | Countdown/between modes only |

### Milestone Data Structure

```json
{
  "id": "a7f3b92c",
  "type": "hours",
  "value": 720,
  "message": "🎉 30 days smoke-free!",
  "fired": false,
  "fired_ms": null
}
```

### Check Process

The `check-milestones` command:

1. Iterates all meters
2. Computes current elapsed time and percentage
3. Compares against unfired milestone values
4. Marks triggered milestones as fired
5. Outputs JSON array of triggered notifications

**Output Format:**
```json
{
  "triggered": [
    {
      "meter": "smoke-free",
      "milestone_id": "a7f3b92c",
      "type": "hours",
      "value": 720,
      "message": "🎉 30 days smoke-free!",
      "channel": "discord",
      "target": "YOUR_CHANNEL_ID",
      "description": "Last cigarette"
    }
  ]
}
```

---

## 6. Persistence Architecture

### Four Storage Methods

| Method | Location | Sync Strategy | Best For |
|--------|----------|---------------|----------|
| **Paper Code** | Physical note | Manual | Offline verification, legal records |
| **Photo/Screenshot** | Camera roll | Auto (iCloud/Google Photos) | Quick backup, visual reference |
| **Witness File** | `~/.openclaw/meter-witness.txt` | Cloud folder sync | Automated backup, audit trail |
| **Email** | Inbox | Self-sent | Searchable archive, timestamp proof |

### Witness File Format

Append-only log with each lock event:

```
============================================================
METER: smoke-free
LOCKED: 2026-02-02T02:13:54.030248+00:00
PAPER CODE: 318B-3229-C523-2F9C-V
FULL HASH: 318b3229c5232f9c5f3731b1cea8dc9f7b5000b762e9ca57f907775580fa170f
============================================================
```

### Cloud Sync Recommendation

Point Dropbox, iCloud Drive, or Google Drive at `~/.openclaw/` to automatically sync:
- `meters.json` — Full meter data
- `meter-witness.txt` — Append-only verification log

---

## 7. Use Cases with Examples

### 7.1 Quit Smoking Tracker

```bash
# Create with quit date
meter.py create smoke-free --start "2025-06-15T08:00:00Z" \
  -d "Last cigarette - freedom begins" \
  --channel discord --target "YOUR_CHANNEL_ID"

# Add health milestones
meter.py milestone smoke-free -t hours -v 24 \
  -m "🎉 24 hours! Nicotine leaving your body"
meter.py milestone smoke-free -t hours -v 72 \
  -m "🎉 3 days! Breathing easier"
meter.py milestone smoke-free -t hours -v 720 \
  -m "🎉 30 days! Lung function improving"
meter.py milestone smoke-free -t hours -v 8760 \
  -m "🎉 ONE YEAR! Heart disease risk halved"

# Lock and save paper code
meter.py lock smoke-free
```

### 7.2 Pregnancy Countdown

```bash
# Due date in 9 months
meter.py create baby-emma \
  --start "2026-01-15" --end "2026-10-15" \
  -d "Baby Emma arriving!" \
  --channel telegram --target "@family_chat"

# Trimester milestones
meter.py milestone baby-emma -t percent -v 33 \
  -m "👶 First trimester complete! Baby is size of a lime"
meter.py milestone baby-emma -t percent -v 66 \
  -m "👶 Second trimester done! Baby can hear your voice"
meter.py milestone baby-emma -t percent -v 90 \
  -m "👶 Almost there! Pack the hospital bag!"
```

### 7.3 Career Inventory (80,000 Hours)

```bash
# Career span from graduation to retirement
meter.py create my-career \
  --start "1998-05-15" --end "2038-05-15" \
  -d "40-year career journey"

# Career phase milestones
meter.py milestone my-career -t percent -v 25 \
  -m "📊 25% - Establishing expertise"
meter.py milestone my-career -t percent -v 50 \
  -m "📊 HALFTIME - Peak earning years"
meter.py milestone my-career -t percent -v 75 \
  -m "📊 75% - Mentorship phase"
meter.py milestone my-career -t percent -v 90 \
  -m "🎯 90% - Plan retirement seriously"

# Project earnings
meter.py career --meter my-career --rate 85 --raise-pct 2.5
```

### 7.4 Project Billing Hours

```bash
# Consulting engagement
meter.py create acme-project \
  --start "2026-02-01T09:00:00Z" \
  -d "ACME Corp consulting - 500 hour cap"

# Budget milestones
meter.py milestone acme-project -t hours -v 100 \
  -m "📊 100 hours billed - 20% of budget"
meter.py milestone acme-project -t hours -v 400 \
  -m "⚠️ 400 hours - discuss extension with client"
meter.py milestone acme-project -t hours -v 475 \
  -m "🚨 475 hours - STOP and get approval"

# Lock for billing records
meter.py lock acme-project
```

### 7.5 Equipment Service Interval

```bash
# Aircraft engine
meter.py create n12345-engine \
  --start "2026-01-01T00:00:00Z" \
  -d "Lycoming IO-540 - TBO 2000 hours"

meter.py milestone n12345-engine -t hours -v 500 \
  -m "🔧 500h inspection due"
meter.py milestone n12345-engine -t hours -v 1000 \
  -m "🔧 1000h major inspection"
meter.py milestone n12345-engine -t hours -v 1800 \
  -m "⚠️ 200 hours to TBO - plan overhaul"

meter.py lock n12345-engine
# Store paper code in aircraft logbook
```

---

## 8. Integration Patterns

### 8.1 Heartbeat Integration (Recommended)

Add to `HEARTBEAT.md`:

```markdown
### Hour Meter Milestones
- Run `python3 ~/.openclaw/workspace/skills/hour-meter/scripts/meter.py check-milestones`
- If any milestones triggered, notify the configured channel with the message
```

**Pros:** Cost-efficient (batched), ~30 minute resolution
**Cons:** Not precise timing

### 8.2 Cron Integration (Precise)

```bash
# Check every 15 minutes
cron add --schedule '*/15 * * * *' \
  --command 'meter.py check-milestones'
```

**Pros:** Precise timing
**Cons:** Higher API costs (each check = 1 agent turn)

### 8.3 External Notification Routing

The `check-milestones` JSON output includes `channel` and `target` for each triggered milestone. Integrate with:

- **SendGrid/Mailgun:** Email notifications
- **Twilio:** SMS alerts
- **Webhooks:** Zapier, Make, n8n

---

## 9. Cost Considerations

### Token Usage

| Approach | Frequency | Daily API Calls | Monthly Cost* |
|----------|-----------|-----------------|---------------|
| Heartbeat | ~30 min | ~48 | Low |
| Cron (15 min) | 15 min | 96 | Medium |
| Cron (5 min) | 5 min | 288 | High |
| Cron (1 min) | 1 min | 1,440 | Very High |

*Actual cost depends on model pricing and prompt size*

### Recommendation

Use **heartbeat** for milestone checking unless precise timing is required. Reserve cron for:
- One-shot reminders
- Business-critical deadlines
- Isolated tasks with different model requirements

---

## 10. Security Analysis

### Threat Model

| Threat | Mitigation |
|--------|------------|
| **Local file tampering** | External hash storage (paper, email, cloud) |
| **Hash collision** | SHA-256 has no known practical collisions |
| **Transcription errors** | Paper code checksum detects typos |
| **Salt guessing** | 64-bit salt space (16 hex chars) |
| **Replay attacks** | N/A (timestamps are public, integrity is the goal) |

### What This Does NOT Provide

- **Non-repudiation:** No third-party timestamp authority
- **Encrypted storage:** Meter data is stored in plaintext
- **Access control:** Anyone with file access can read meters

### Stronger Guarantees (Future Work)

For legal/compliance use cases, consider:
- **RFC 3161 timestamps:** Third-party timestamp authorities
- **Blockchain anchoring:** Publish hashes to Bitcoin/Ethereum
- **Hardware security modules:** Store salts in TPM/HSM

---

## Conclusion

Hour Meter provides a practical balance between simplicity and verifiability for personal and professional time tracking. The paper code system makes cryptographic verification accessible to non-technical users, while the milestone notification system integrates seamlessly with OpenClaw's channel architecture.

For most use cases—quit tracking, career planning, project hours, equipment runtime—the local integrity model with external hash backup provides sufficient tamper evidence. High-stakes applications may require additional measures outlined in the security analysis.

---

*Document version: 1.0*
*Last updated: February 2026*
*Author: OpenClaw Agent*
*Architect: @rm289*
