// FixClawd Skill - Brave API Integration
//
// This module provides the actual implementation for integrating 
// Brave Search API into the FixClawd system repair workflow.

/**
 * Searches for official documentation and solutions using Brave API
 * @param {string} query - The search query
 * @param {Object} options - Additional search options
 * @returns {Promise<Object>} Search results
 */
async function searchOfficialDocs(query, options = {}) {
    const defaultOptions = {
        count: 5,
        freshness: options.freshness || 'week' // Recent results for up-to-date solutions
    };
    
    const searchQuery = `${query} site:official-docs OR site:github.com`;
    
    try {
        const results = await web_search({
            query: searchQuery,
            ...defaultOptions
        });
        
        return {
            success: true,
            results: results,
            message: `Found ${results.length} official documentation results for: ${query}`
        };
    } catch (error) {
        return {
            success: false,
            error: error.message,
            message: 'Failed to search official documentation'
        };
    }
}

/**
 * Searches GitHub for solutions using Brave API
 * @param {string} query - The search query
 * @returns {Promise<Object>} GitHub search results
 */
async function searchGitHubSolutions(query) {
    const githubQuery = `${query} site:github.com`;
    
    try {
        const results = await web_search({
            query: githubQuery,
            count: 5,
            freshness: 'month' // GitHub solutions can be older but still valid
        });
        
        return {
            success: true,
            results: results,
            message: `Found ${results.length} GitHub solution results for: ${query}`
        };
    } catch (error) {
        return {
            success: false,
            error: error.message,
            message: 'Failed to search GitHub for solutions'
        };
    }
}

/**
 * Searches ClawdHub for existing skills using Brave API
 * @param {string} query - The search query for skills
 * @returns {Promise<Object>} ClawdHub skill search results
 */
async function searchClawdHubSkills(query) {
    const skillQuery = `${query} site:clawdhub.com OR site:github.com/clawdhub`;
    
    try {
        const results = await web_search({
            query: skillQuery,
            count: 3,
            freshness: 'year' // Skills might be older but still applicable
        });
        
        return {
            success: true,
            results: results,
            message: `Found ${results.length} potential skills on ClawdHub/GitHub for: ${query}`
        };
    } catch (error) {
        return {
            success: false,
            error: error.message,
            message: 'Failed to search ClawdHub for existing skills'
        };
    }
}

/**
 * Executes the complete FixClawd workflow with integrated search capabilities
 * @param {string} problemDescription - Description of the problem to solve
 * @returns {Promise<Object>} Complete workflow execution results
 */
async function executeWorkflow(problemDescription) {
    console.log("Starting FixClawd workflow...");
    
    // Step 1: Problem understanding is handled by the calling agent
    
    // Step 2: Search for official solutions
    console.log("Step 2: Searching for official solutions...");
    const officialResults = await searchOfficialDocs(problemDescription);
    
    if (officialResults.success) {
        console.log(officialResults.message);
        // Return early if official solutions found
        if (officialResults.results && officialResults.results.length > 0) {
            return {
                step: 2,
                status: 'official_solutions_found',
                results: officialResults.results,
                message: 'Official solutions found - recommend implementing these first'
            };
        }
    }
    
    // Step 3: Search for existing ClawdHub skills
    console.log("Step 3: Searching for existing skills...");
    const skillResults = await searchClawdHubSkills(problemDescription);
    
    if (skillResults.success) {
        console.log(skillResults.message);
        // Return if relevant skills found
        if (skillResults.results && skillResults.results.length > 0) {
            return {
                step: 3,
                status: 'relevant_skills_found',
                results: skillResults.results,
                message: 'Existing skills found - recommend installing and using these'
            };
        }
    }
    
    // Step 5: Search GitHub for solutions
    console.log("Step 5: Searching GitHub for solutions...");
    const githubResults = await searchGitHubSolutions(problemDescription);
    
    if (githubResults.success) {
        console.log(githubResults.message);
        if (githubResults.results && githubResults.results.length > 0) {
            return {
                step: 5,
                status: 'github_solutions_found',
                results: githubResults.results,
                message: 'Solutions found on GitHub - evaluate these community solutions'
            };
        }
    }
    
    // If no solutions found, proceed to step 6
    return {
        step: 6,
        status: 'no_solutions_found',
        message: 'No solutions found through searches - consider creating a custom fix script'
    };
}

module.exports = {
    searchOfficialDocs,
    searchGitHubSolutions,
    searchClawdHubSkills,
    executeWorkflow
};