#!/usr/bin/env python3
"""
Synthdata CLI — Query volatility forecasts and price data.

Usage:
    python3 synth.py BTC                        # Single asset
    python3 synth.py BTC ETH SOL --compare      # Compare assets
    python3 synth.py --forecast --chart         # Full forecast with chart
    python3 synth.py BTC --simulate             # 24h Monte Carlo simulation
    python3 synth.py BTC --simulate --hours 12  # 12h simulation
    python3 synth.py --all                      # All assets overview

Note: Synthdata provides 24h forecasts, so simulations are capped at 24 hours.
"""

import os
import sys
import json
import math
import random
import argparse
import urllib.request
from datetime import datetime

API_BASE = "https://api.synthdata.co"
API_KEY = os.environ.get("SYNTHDATA_API_KEY", "")

ASSETS = ["BTC", "ETH", "SOL", "XAU", "SPYX", "NVDAX", "GOOGLX", "TSLAX", "AAPLX"]

ASSET_NAMES = {
    "BTC": "Bitcoin",
    "ETH": "Ethereum",
    "SOL": "Solana",
    "XAU": "Gold",
    "SPYX": "S&P 500",
    "NVDAX": "NVIDIA",
    "GOOGLX": "Google",
    "TSLAX": "Tesla",
    "AAPLX": "Apple"
}


def api_get(endpoint):
    """Fetch from Synthdata API"""
    url = f"{API_BASE}{endpoint}"
    
    try:
        req = urllib.request.Request(url, headers={
            "Authorization": f"Apikey {API_KEY}",
            "User-Agent": "SynthdataSkill/1.0"
        })
        with urllib.request.urlopen(req, timeout=10) as resp:
            return json.loads(resp.read().decode())
    except urllib.error.HTTPError as e:
        return {"error": f"HTTP {e.code}: {e.reason}"}
    except Exception as e:
        return {"error": str(e)}


def get_asset(ticker):
    """Get single asset volatility data"""
    return api_get(f"/insights/volatility?asset={ticker.upper()}")


def get_all_assets():
    """Get all assets"""
    results = {}
    for ticker in ASSETS:
        data = get_asset(ticker)
        if "error" not in data:
            results[ticker] = data
    return results


def format_price(price):
    """Format price with appropriate decimals"""
    if price is None:
        return "N/A"
    if price >= 1000:
        return f"${price:,.0f}"
    elif price >= 1:
        return f"${price:,.2f}"
    else:
        return f"${price:.4f}"


def format_change(change):
    """Format percentage change with emoji"""
    if change is None:
        return "N/A"
    sign = "+" if change >= 0 else ""
    emoji = "🟢" if change >= 0 else "🔴"
    return f"{emoji} {sign}{change:.2f}%"


def vol_level(vol):
    """Categorize volatility level"""
    if vol is None:
        return "Unknown"
    if vol < 20:
        return "Low"
    elif vol < 40:
        return "Moderate"
    elif vol < 60:
        return "Elevated"
    elif vol < 80:
        return "High"
    else:
        return "Extreme"


def vol_emoji(vol):
    """Get emoji for vol level"""
    if vol is None:
        return "⚪"
    if vol < 20:
        return "🟢"
    elif vol < 40:
        return "🟡"
    elif vol < 60:
        return "🟠"
    else:
        return "🔴"


def extract_metrics(data):
    """Extract key metrics from API response"""
    if "error" in data:
        return data
    
    result = {}
    
    # Current price
    result["price"] = data.get("current_price")
    
    # Realized volatility (most recent and average)
    realized = data.get("realized", {})
    vol_array = realized.get("volatility", [])
    if vol_array:
        result["vol_current"] = vol_array[-1]  # Most recent
        result["vol_realized_avg"] = realized.get("average_volatility")
    
    # Calculate 24h change from price data
    prices = realized.get("prices", [])
    if len(prices) >= 24:
        old_price = prices[-24].get("price")
        new_price = prices[-1].get("price")
        if old_price and new_price and old_price > 0:
            result["change_24h"] = ((new_price - old_price) / old_price) * 100
    
    # Forecast volatility
    forecast_future = data.get("forecast_future", {})
    forecast_vol = forecast_future.get("volatility", [])
    if forecast_vol:
        # Filter out None values
        valid_forecast = [v for v in forecast_vol if v is not None]
        if valid_forecast:
            result["vol_forecast_avg"] = forecast_future.get("average_volatility")
            result["vol_forecast_next"] = valid_forecast[0] if valid_forecast else None
    
    return result


def print_asset(ticker, data):
    """Pretty print single asset"""
    print(f"\n{'='*50}")
    print(f"  {ticker} — {ASSET_NAMES.get(ticker, ticker)}")
    print(f"{'='*50}")
    
    if "error" in data:
        print(f"  Error: {data['error']}")
        return
    
    metrics = extract_metrics(data)
    
    price = metrics.get("price")
    change = metrics.get("change_24h")
    vol = metrics.get("vol_current")
    vol_avg = metrics.get("vol_realized_avg")
    forecast_avg = metrics.get("vol_forecast_avg")
    
    print(f"  Price:           {format_price(price)}")
    if change is not None:
        print(f"  24h Change:      {format_change(change)}")
    if vol is not None:
        print(f"  Current Vol:     {vol:.1f}% {vol_emoji(vol)} [{vol_level(vol)}]")
    if vol_avg is not None:
        print(f"  Avg Realized:    {vol_avg:.1f}%")
    if forecast_avg is not None:
        print(f"  Forecast Vol:    {forecast_avg:.1f}%")


def print_comparison(assets_data):
    """Print comparison table"""
    print(f"\n{'Asset':<8} {'Name':<12} {'Price':>12} {'Realized':>10} {'Forecast':>10} {'Level':>10}")
    print("-" * 70)
    
    # Extract metrics and sort by current vol descending
    rows = []
    for ticker, data in assets_data.items():
        if "error" in data:
            continue
        metrics = extract_metrics(data)
        vol = metrics.get("vol_current") or metrics.get("vol_realized_avg") or 0
        rows.append((ticker, data, metrics, vol))
    
    sorted_rows = sorted(rows, key=lambda x: x[3], reverse=True)
    
    for ticker, data, metrics, _ in sorted_rows:
        name = ASSET_NAMES.get(ticker, ticker)[:12]
        price = format_price(metrics.get("price"))
        vol = metrics.get("vol_current")
        forecast = metrics.get("vol_forecast_avg")
        vol_str = f"{vol:.1f}%" if vol else "N/A"
        forecast_str = f"{forecast:.1f}%" if forecast else "N/A"
        level = vol_level(vol)
        emoji = vol_emoji(vol)
        
        print(f"{ticker:<8} {name:<12} {price:>12} {vol_str:>10} {forecast_str:>10} {emoji} {level}")





def monte_carlo(price, vol_annual, hours=24, paths=500):
    """Generate Monte Carlo price paths (hourly steps, max 24h)"""
    if price is None or vol_annual is None:
        return []
    
    # Cap at 24 hours (forecast window)
    hours = min(hours, 24)
    
    vol_decimal = vol_annual / 100
    dt = 1/(365 * 24)  # Hourly time step
    results = []
    
    for _ in range(paths):
        path = [price]
        p = price
        for _ in range(hours):
            drift = 0
            shock = random.gauss(0, 1)
            p = p * math.exp((drift - 0.5 * vol_decimal**2) * dt + vol_decimal * math.sqrt(dt) * shock)
            path.append(p)
        results.append(path)
    
    return results


def monte_carlo_stats(paths):
    """Calculate percentile bands from paths"""
    if not paths:
        return None
    
    days = len(paths[0])
    stats = {"p5": [], "p25": [], "p50": [], "p75": [], "p95": []}
    
    for day in range(days):
        day_prices = sorted([p[day] for p in paths])
        n = len(day_prices)
        stats["p5"].append(day_prices[int(n * 0.05)])
        stats["p25"].append(day_prices[int(n * 0.25)])
        stats["p50"].append(day_prices[int(n * 0.50)])
        stats["p75"].append(day_prices[int(n * 0.75)])
        stats["p95"].append(day_prices[int(n * 0.95)])
    
    return stats





def main():
    parser = argparse.ArgumentParser(description="Synthdata Volatility CLI")
    parser.add_argument("assets", nargs="*", help="Asset tickers (e.g., BTC ETH SOL)")
    parser.add_argument("--all", action="store_true", help="Show all assets")
    parser.add_argument("--compare", action="store_true", help="Compare assets in table")
    parser.add_argument("--forecast", action="store_true", help="Full forecast")
    parser.add_argument("--simulate", action="store_true", help="Monte Carlo simulation")
    parser.add_argument("--hours", type=int, default=24, help="Simulation hours (max 24)")
    parser.add_argument("--paths", type=int, default=500, help="Simulation paths")
    parser.add_argument("--json", action="store_true", help="Output as JSON")
    
    args = parser.parse_args()
    
    if not API_KEY:
        print("Error: SYNTHDATA_API_KEY not set")
        print("Export your API key: export SYNTHDATA_API_KEY=your_key_here")
        sys.exit(1)
    
    # Determine which assets to fetch
    if args.all or args.forecast:
        tickers = ASSETS
    elif args.assets:
        tickers = [a.upper() for a in args.assets]
    else:
        parser.print_help()
        sys.exit(0)
    
    # Fetch data
    assets_data = {}
    for ticker in tickers:
        assets_data[ticker] = get_asset(ticker)
    
    # JSON output
    if args.json:
        print(json.dumps(assets_data, indent=2))
        return
    
    # Monte Carlo simulation
    if args.simulate and len(tickers) == 1:
        ticker = tickers[0]
        data = assets_data[ticker]
        if "error" not in data:
            metrics = extract_metrics(data)
            price = metrics.get("price")
            vol = metrics.get("vol_forecast_avg") or metrics.get("vol_current")  # Prefer forecast for simulation
            
            # Cap hours at 24 (forecast window)
            hours = min(args.hours, 24)
            if args.hours > 24:
                print(f"⚠️  Capped to 24h (Synthdata forecast window)")
            
            if price and vol:
                print(f"\n🎲 Monte Carlo Simulation: {ticker}")
                print(f"   Starting Price: {format_price(price)}")
                print(f"   Volatility: {vol:.1f}%")
                print(f"   Hours: {hours}, Paths: {args.paths}")
                
                paths = monte_carlo(price, vol, hours, args.paths)
                stats = monte_carlo_stats(paths)
                
                if stats:
                    print(f"\n   {hours}h Price Ranges:")
                    print(f"   5th percentile:  {format_price(stats['p5'][-1])}")
                    print(f"   25th percentile: {format_price(stats['p25'][-1])}")
                    print(f"   Median:          {format_price(stats['p50'][-1])}")
                    print(f"   75th percentile: {format_price(stats['p75'][-1])}")
                    print(f"   95th percentile: {format_price(stats['p95'][-1])}")
            else:
                print(f"Error: Missing price or volatility data for {ticker}")
        return
    
    # Comparison table
    if args.compare or args.forecast or args.all:
        print_comparison(assets_data)
        return
    
    # Single asset details
    for ticker in tickers:
        print_asset(ticker, assets_data[ticker])


if __name__ == "__main__":
    main()
