# Synthdata Use Cases

## 1. Daily Volatility Report to Slack/Telegram

Schedule a daily briefing for your team:

```javascript
// synth-daily.mjs
const API_KEY = process.env.SYNTHDATA_API_KEY;
const ASSETS = ['BTC', 'ETH', 'SOL', 'SPYX'];

async function getVol(ticker) {
  const res = await fetch(
    `https://api.synthdata.co/insights/volatility?asset=${ticker}`,
    { headers: { Authorization: `Apikey ${API_KEY}` } }
  );
  const data = await res.json();
  return {
    ticker,
    price: data.current_price,
    realized: data.realized?.average_volatility,
    forecast: data.forecast_future?.average_volatility
  };
}

async function generateReport() {
  const results = await Promise.all(ASSETS.map(getVol));
  let msg = '📊 *Daily Volatility Report*\n\n';
  
  for (const r of results.sort((a, b) => b.forecast - a.forecast)) {
    const emoji = r.forecast > 60 ? '🔴' : r.forecast > 40 ? '🟠' : '🟢';
    msg += `${emoji} *${r.ticker}*: $${r.price?.toFixed(2)} | Vol: ${r.forecast?.toFixed(1)}%\n`;
  }
  
  return msg;
}
```

## 2. Volatility Alerts

Trigger notifications when volatility exceeds thresholds:

```python
ALERT_THRESHOLD = 70  # Alert if forecast > 70%

for ticker in ASSETS:
    data = get_asset(ticker)
    forecast = data.get("forecast_future", {}).get("average_volatility")
    
    if forecast and forecast > ALERT_THRESHOLD:
        send_alert(f"⚠️ {ticker} volatility forecast: {forecast:.1f}%")
```

## 3. Polymarket Integration

Use volatility to inform binary option bets on hourly markets:

```python
# High volatility = larger expected moves
# Low volatility = price more likely to stay flat

btc_data = get_asset("BTC")
forecast_vol = btc_data["forecast_future"]["average_volatility"]

if forecast_vol > 60:
    print("High vol - expect significant price movement")
    # Consider betting on UP or DOWN based on trend
elif forecast_vol < 30:
    print("Low vol - price likely to stay range-bound")
    # Hourly direction bet is riskier (more random)
```

## 4. Portfolio Risk Dashboard

Calculate aggregate portfolio volatility:

```python
portfolio = {
    "BTC": 0.5,   # 50% allocation
    "ETH": 0.3,   # 30%
    "SOL": 0.2    # 20%
}

weighted_vol = 0
for ticker, weight in portfolio.items():
    data = get_asset(ticker)
    vol = data["forecast_future"]["average_volatility"]
    weighted_vol += weight * vol

print(f"Portfolio forecast volatility: {weighted_vol:.1f}%")
```

## 5. Monte Carlo Position Sizing

Use volatility for risk-adjusted position sizes:

```python
capital = 10000
max_risk_pct = 0.02  # 2% max risk per trade
vol = forecast_vol / 100

# Expected daily move (1 std dev)
daily_move = vol / math.sqrt(365)

# Position size based on stop-loss
stop_loss_pct = daily_move * 2  # 2 std dev stop
position_size = (capital * max_risk_pct) / stop_loss_pct

print(f"Max position: ${position_size:.2f}")
```

## 6. Volatility Comparison Chart

Generate visual comparisons:

```bash
python3 scripts/synth.py BTC ETH SOL XAU --compare --chart
```

Output includes a QuickChart URL for immediate visualization.

## 7. Historical Backtest Data

The API provides `forecast_past` for backtesting strategies:

```python
# Compare predictions vs realized
realized = data["realized"]["average_volatility"]
predicted = data["forecast_past"]["average_volatility"]
error = abs(realized - predicted) / realized * 100

print(f"Forecast accuracy: {100 - error:.1f}%")
```

## 8. Multi-Timeframe Analysis

Combine with price data for trend + volatility signals:

```python
prices = data["realized"]["prices"]
recent_prices = [p["price"] for p in prices[-24:]]

trend = "up" if recent_prices[-1] > recent_prices[0] else "down"
vol = data["forecast_future"]["average_volatility"]

if trend == "up" and vol > 50:
    print("Bullish momentum with high volatility - strong move likely")
elif trend == "down" and vol > 50:
    print("Bearish momentum with high volatility - sharp drop possible")
```

## 9. Volatility Regime Detection

Classify market conditions:

```python
def get_regime(vol):
    if vol < 25:
        return "calm"
    elif vol < 50:
        return "normal"
    elif vol < 75:
        return "elevated"
    else:
        return "crisis"

# Adapt strategy based on regime
regime = get_regime(forecast_vol)
if regime == "crisis":
    reduce_exposure()
elif regime == "calm":
    consider_selling_options()
```

## 10. Cross-Asset Correlation

Compare volatility across asset classes:

```python
crypto_vol = (btc_vol + eth_vol + sol_vol) / 3
stock_vol = (nvda_vol + googl_vol + aapl_vol) / 3
gold_vol = xau_vol

print(f"Crypto avg: {crypto_vol:.1f}%")
print(f"Stocks avg: {stock_vol:.1f}%")
print(f"Gold: {gold_vol:.1f}%")

# High crypto vol + low stock vol = risk-off sentiment
```
