---
name: synth-data
description: Query volatility forecasts from Synthdata.co for crypto, commodities, and stocks. Compare assets and run Monte Carlo simulations.
metadata:
  {
    "openclaw":
      {
        "emoji": "📊",
        "requires": { "bins": ["python3"], "env": ["SYNTHDATA_API_KEY"] }
      }
  }
---

# Synthdata Volatility Skill

Query and analyze volatility forecasts from Synthdata.co for crypto, commodities, and stock indices.

## Setup

Set your API key:
```bash
export SYNTHDATA_API_KEY=your_key_here
```

## Quick Start

```bash
# Single asset
python3 scripts/synth.py BTC

# Multiple assets comparison
python3 scripts/synth.py BTC ETH SOL --compare

# All assets overview
python3 scripts/synth.py --all

# Monte Carlo simulation (24h max)
python3 scripts/synth.py BTC --simulate --hours 12
```

## Available Assets

| Ticker | Name | Category |
|--------|------|----------|
| BTC | Bitcoin | Crypto |
| ETH | Ethereum | Crypto |
| SOL | Solana | Crypto |
| XAU | Gold | Commodity |
| SPYX | S&P 500 | Index |
| NVDAX | NVIDIA | Stock |
| GOOGLX | Google | Stock |
| TSLAX | Tesla | Stock |
| AAPLX | Apple | Stock |

## Output Example

```
==================================================
  BTC — Bitcoin
==================================================
  Price:           $77,966
  24h Change:      🔴 -0.95%
  Current Vol:     58.4% 🟠 [Elevated]
  Avg Realized:    53.3%
  Forecast Vol:    52.2%
```

## Volatility Levels

| Level | Range | Emoji |
|-------|-------|-------|
| Low | < 20% | 🟢 |
| Moderate | 20-40% | 🟡 |
| Elevated | 40-60% | 🟠 |
| High | 60-80% | 🔴 |
| Extreme | > 80% | 🔴 |

## Use Cases

### 1. Market Overview
```bash
python3 scripts/synth.py --all
```
Get a ranked table of all assets by volatility.

### 2. Trading Signals
- **High forecast → Current low**: Expect volatility spike
- **Low forecast → Current high**: Volatility may decrease
- Use for position sizing and options trading

### 3. Monte Carlo Projections
```bash
python3 scripts/synth.py BTC --simulate --hours 24 --paths 1000
```
Generate probabilistic price ranges using forecast volatility (24h max - Synthdata forecast window).

### 4. Scheduled Reports
Create a cron job for daily Slack/Telegram forecasts (see examples/use-cases.md).

### 5. Risk Alerts
Monitor for assets crossing volatility thresholds and trigger notifications.

## API Reference

See `references/api.md` for full API documentation.

## Direct API Usage

```python
import requests

resp = requests.get(
    "https://api.synthdata.co/insights/volatility",
    params={"asset": "BTC"},
    headers={"Authorization": f"Apikey {API_KEY}"}
)
data = resp.json()

# Key fields:
price = data["current_price"]
realized_vol = data["realized"]["average_volatility"]
forecast_vol = data["forecast_future"]["average_volatility"]
```

## Integration Ideas

- **Polymarket**: Use volatility forecasts to inform up/down market bets
- **Options**: High forecast vol = consider buying options
- **Portfolio**: Rebalance when aggregate volatility spikes
- **Alerts**: Notify when forecast differs significantly from realized
