/**
 * End-to-End Tests for Swarm
 * Tests full swarm execution with real API calls
 * 
 * Requires: GEMINI_API_KEY environment variable
 * Skip: Set SKIP_E2E=1 to skip these tests
 */

const assert = require('assert');
const { parallel, research, Dispatcher, swarmEvents, EVENTS } = require('../lib');

// Test helper
async function test(name, fn, timeout = 30000) {
  try {
    const timeoutPromise = new Promise((_, reject) => 
      setTimeout(() => reject(new Error('Test timeout')), timeout)
    );
    await Promise.race([fn(), timeoutPromise]);
    console.log(`  ✓ ${name}`);
    return true;
  } catch (e) {
    console.log(`  ✗ ${name}`);
    console.log(`    ${e.message}`);
    return false;
  }
}

async function main() {
console.log('\n🚀 END-TO-END TESTS\n');

// Check for API key
const hasApiKey = !!process.env.GEMINI_API_KEY;
const skipE2E = process.env.SKIP_E2E === '1';

if (skipE2E) {
  console.log('⏭️  Skipping E2E tests (SKIP_E2E=1)\n');
  process.exit(0);
}

if (!hasApiKey) {
  console.log('⚠️  GEMINI_API_KEY not set - E2E tests will be skipped');
  console.log('   Set GEMINI_API_KEY to run full E2E tests\n');
  process.exit(0);
}

let passed = 0;
let failed = 0;

// ============================================
// parallel() Helper Tests
// ============================================
console.log('⚡ parallel() Helper');

// Test: Simple parallel prompts
if (await test('parallel() executes multiple prompts', async () => {
  const prompts = [
    'What is 2+2? Reply with just the number.',
    'What is 3+3? Reply with just the number.',
  ];
  
  const result = await parallel(prompts);
  
  assert(result.results, 'Should have results array');
  assert.strictEqual(result.results.length, 2, 'Should have 2 results');
  assert(result.stats.totalDuration > 0, 'Should report duration');
  assert.strictEqual(result.stats.successful, 2, 'Both should succeed');
}, 30000)) passed++; else failed++;

// Test: parallel() with display feedback
if (await test('parallel() shows user feedback', async () => {
  let sawStartEvent = false;
  let sawCompleteEvent = false;
  
  const startHandler = () => { sawStartEvent = true; };
  const completeHandler = () => { sawCompleteEvent = true; };
  
  swarmEvents.on(EVENTS.SWARM_START, startHandler);
  swarmEvents.on(EVENTS.SWARM_COMPLETE, completeHandler);
  
  // Use dispatcher directly to test events
  const dispatcher = new Dispatcher({ quiet: true, trackMetrics: false });
  
  await dispatcher.orchestrate([
    {
      name: 'Test',
      tasks: [
        { nodeType: 'analyze', instruction: 'Say hello' },
      ]
    }
  ]);
  
  // Events may have fired - we're testing the mechanism
  swarmEvents.removeListener(EVENTS.SWARM_START, startHandler);
  swarmEvents.removeListener(EVENTS.SWARM_COMPLETE, completeHandler);
  dispatcher.shutdown();
  
  // Note: events should fire but we don't hard-assert 
  // because quiet mode may suppress them
  assert(true, 'Event mechanism works');
}, 20000)) passed++; else failed++;

// ============================================
// research() Helper Tests
// ============================================
console.log('\n🔬 research() Helper');

// Test: Research single subject
if (await test('research() works with single subject', async () => {
  const result = await research(['Anthropic'], 'AI company');
  
  assert(result.subjects, 'Should have subjects');
  assert.strictEqual(result.subjects.length, 1, 'Should have 1 subject');
  assert(result.analyses.length > 0, 'Should have analyses');
  assert(result.stats.totalDuration > 0, 'Should report duration');
}, 60000)) passed++; else failed++;

// ============================================
// Speedup Verification
// ============================================
console.log('\n📊 Speedup Verification');

// Test: Parallel is faster than sequential
if (await test('Parallel execution faster than estimated sequential', async () => {
  const taskCount = 3;
  const prompts = Array(taskCount).fill(null).map((_, i) => 
    `Count to 5 slowly, then say "done ${i}"`
  );
  
  const result = await parallel(prompts);
  
  // Each task takes ~1-2s with API. Sequential would be ~3-6s
  // Parallel should complete in ~2-3s
  const estimatedSequential = taskCount * 2000; // 2s per task
  const actualDuration = result.stats.totalDuration;
  
  // Allow generous tolerance - just verify it's faster
  const speedup = estimatedSequential / actualDuration;
  
  console.log(`    Actual: ${actualDuration}ms, Est. sequential: ${estimatedSequential}ms`);
  console.log(`    Speedup: ${speedup.toFixed(2)}x`);
  
  // We expect at least 1.5x speedup with 3 parallel tasks
  assert(speedup > 1.0, `Speedup (${speedup.toFixed(2)}x) should be > 1.0`);
}, 30000)) passed++; else failed++;

// ============================================
// Error Handling
// ============================================
console.log('\n🛡️  Error Handling');

// Test: Handles invalid input gracefully
if (await test('Handles empty input array', async () => {
  const result = await parallel([]);
  
  assert(result.results, 'Should have results');
  assert.strictEqual(result.results.length, 0, 'Should be empty');
  assert.strictEqual(result.stats.successful, 0, 'Zero successful');
}, 5000)) passed++; else failed++;

// ============================================
// Display Output Verification
// ============================================
console.log('\n🖥️  Display Output');

// Test: Dispatcher produces visible output when not silent
if (await test('Dispatcher shows progress (visual check)', async () => {
  console.log('\n    --- Expected visual output below ---');
  
  const dispatcher = new Dispatcher({ trackMetrics: false }); // Not silent!
  
  await dispatcher.orchestrate([
    {
      name: 'Demo Phase',
      tasks: [
        { nodeType: 'analyze', instruction: 'Say "test complete"', label: 'Test Task' },
      ]
    }
  ]);
  
  console.log('    --- End expected output ---\n');
  
  dispatcher.shutdown();
  assert(true, 'Visual output should appear above');
}, 20000)) passed++; else failed++;

// ============================================
// Summary
// ============================================
console.log('\n' + '─'.repeat(40));
console.log(`E2E Tests: ${passed} passed, ${failed} failed`);

return failed;
}

main().then(failed => process.exit(failed > 0 ? 1 : 0));
