#!/usr/bin/env node
/**
 * Node Scaling Setup Wizard
 * Interactive configuration for Clawdbot users
 */

const readline = require('readline');
const fs = require('fs');
const path = require('path');
const os = require('os');

const CONFIG_DIR = path.join(os.homedir(), '.config', 'clawdbot');
const CONFIG_PATH = path.join(CONFIG_DIR, 'node-scaling.yaml');

const PROVIDERS = [
  {
    id: 'gemini',
    name: 'Google Gemini Flash',
    cost: '$0.075/1M tokens',
    keyUrl: 'https://aistudio.google.com/apikey',
    envVar: 'GEMINI_API_KEY',
    recommended: true,
  },
  {
    id: 'groq',
    name: 'Groq (FREE tier available)',
    cost: 'Free tier / $0.05/1M tokens',
    keyUrl: 'https://console.groq.com/keys',
    envVar: 'GROQ_API_KEY',
    recommended: false,
  },
  {
    id: 'openai',
    name: 'OpenAI GPT-4o-mini',
    cost: '$0.15/1M tokens',
    keyUrl: 'https://platform.openai.com/api-keys',
    envVar: 'OPENAI_API_KEY',
    recommended: false,
  },
  {
    id: 'anthropic',
    name: 'Anthropic Claude Haiku',
    cost: '$0.25/1M tokens',
    keyUrl: 'https://console.anthropic.com/settings/keys',
    envVar: 'ANTHROPIC_API_KEY',
    recommended: false,
  },
];

const rl = readline.createInterface({
  input: process.stdin,
  output: process.stdout,
});

function ask(question) {
  return new Promise(resolve => rl.question(question, resolve));
}

function print(text) {
  console.log(text);
}

function box(lines, title = '') {
  const maxLen = Math.max(...lines.map(l => l.length), title.length);
  const border = '─'.repeat(maxLen + 2);
  print(`┌${border}┐`);
  if (title) {
    print(`│ ${title.padEnd(maxLen)} │`);
    print(`├${border}┤`);
  }
  for (const line of lines) {
    print(`│ ${line.padEnd(maxLen)} │`);
  }
  print(`└${border}┘`);
}

async function validateApiKey(provider, apiKey) {
  print('  Validating API key...');
  
  try {
    const { createProvider } = require('../lib/providers');
    const client = createProvider({
      provider: provider.id,
      apiKey,
    });
    
    const result = await client.complete('Say "ok" and nothing else.', { maxTokens: 10 });
    return result.toLowerCase().includes('ok');
  } catch (error) {
    print(`  ❌ Error: ${error.message}`);
    return false;
  }
}

function getSystemInfo() {
  const cpus = os.cpus().length;
  const totalMem = Math.round(os.totalmem() / 1024 / 1024 / 1024);
  const freeMem = Math.round(os.freemem() / 1024 / 1024 / 1024);
  
  // Recommend nodes based on system resources
  // Rule: max(2, min(cpus * 2, freeMem / 0.1, 50))
  const recommended = Math.max(2, Math.min(cpus * 2, Math.floor(freeMem / 0.1), 50));
  
  return { cpus, totalMem, freeMem, recommendedNodes: recommended };
}

function saveConfig(config) {
  // Ensure config directory exists
  if (!fs.existsSync(CONFIG_DIR)) {
    fs.mkdirSync(CONFIG_DIR, { recursive: true });
  }
  
  const yaml = `# Node Scaling Configuration for Clawdbot
# Generated by setup wizard on ${new Date().toISOString()}

node_scaling:
  enabled: true
  
  # Resource limits
  limits:
    max_nodes: ${config.maxNodes}
    max_concurrent_api: ${Math.min(config.maxNodes, 20)}
    
  # LLM Provider for worker nodes
  provider:
    name: ${config.provider}
    model: ${config.model}
    api_key_env: ${config.envVar}
    
  # Web Search (Gemini only)
  # When enabled, research tasks use Google Search grounding for live data
  web_search:
    enabled: ${config.webSearch || false}
    # Set to true to enable web search for all parallel tasks by default
    parallel_default: false
    
  # Cost controls (optional)
  cost:
    max_daily_spend: 5.00
    warn_at: 1.00
    
  # Task routing
  routing:
    auto_scale_threshold: 3
    enabled_task_types:
      - research
      - batch_analysis
      - document_processing
      - web_search
`;
  
  fs.writeFileSync(CONFIG_PATH, yaml);
  
  // Also save the API key to a separate file (more secure)
  if (config.apiKey) {
    const keyPath = path.join(CONFIG_DIR, `${config.provider}-key.txt`);
    fs.writeFileSync(keyPath, config.apiKey, { mode: 0o600 });
    print(`  API key saved to: ${keyPath}`);
  }
  
  return CONFIG_PATH;
}

async function main() {
  console.clear();
  print('');
  print('🚀 Node Scaling Setup for Clawdbot');
  print('═'.repeat(50));
  print('');
  print('This will configure parallel task execution using');
  print('cheap LLM workers for faster task completion.');
  print('');
  
  // Step 1: Choose provider
  print('Step 1: Choose your LLM provider for worker nodes');
  print('');
  
  for (let i = 0; i < PROVIDERS.length; i++) {
    const p = PROVIDERS[i];
    const rec = p.recommended ? ' (Recommended)' : '';
    print(`  [${i + 1}] ${p.name} - ${p.cost}${rec}`);
  }
  print('');
  
  let providerIndex = -1;
  while (providerIndex < 0 || providerIndex >= PROVIDERS.length) {
    const answer = await ask('Choose provider [1]: ');
    providerIndex = (parseInt(answer) || 1) - 1;
  }
  
  const provider = PROVIDERS[providerIndex];
  print(`\n  ✓ Selected: ${provider.name}`);
  
  // Step 2: Get API key
  print('\nStep 2: Enter your API key');
  print(`  Get one at: ${provider.keyUrl}`);
  print('');
  
  let apiKey = '';
  let validated = false;
  
  while (!validated) {
    apiKey = await ask('  API Key: ');
    
    if (!apiKey || apiKey.length < 10) {
      print('  ⚠️  API key seems too short. Try again.');
      continue;
    }
    
    validated = await validateApiKey(provider, apiKey);
    
    if (validated) {
      print('  ✓ API key validated successfully!');
    } else {
      print('  ⚠️  Could not validate key. Try again or press Enter to skip validation.');
      const skip = await ask('  Skip validation? [y/N]: ');
      if (skip.toLowerCase() === 'y') {
        validated = true;
      }
    }
  }
  
  // Set environment variable hint
  print(`\n  💡 Tip: Set ${provider.envVar} in your environment for persistence`);
  
  // Step 3: Resource limits
  print('\nStep 3: Configure resource limits');
  
  const sysInfo = getSystemInfo();
  print(`  Your system: ${sysInfo.cpus} CPU cores, ${sysInfo.totalMem}GB RAM (${sysInfo.freeMem}GB free)`);
  print(`  Recommended: max ${sysInfo.recommendedNodes} parallel workers`);
  print('');
  
  const maxNodesAnswer = await ask(`  Max workers [${sysInfo.recommendedNodes}]: `);
  const maxNodes = parseInt(maxNodesAnswer) || sysInfo.recommendedNodes;
  
  print(`  ✓ Will use up to ${maxNodes} parallel workers`);
  
  // Step 4: Web Search (Gemini only)
  let webSearch = false;
  if (provider.id === 'gemini') {
    print('\nStep 4: Enable Web Search for Workers');
    print('');
    print('  Gemini supports Google Search grounding — workers can search');
    print('  the live web for current data (pricing, news, real-time info).');
    print('  This uses the same API key at no extra cost.');
    print('');
    const wsAnswer = await ask('  Enable web search for research tasks? [Y/n]: ');
    webSearch = wsAnswer.toLowerCase() !== 'n';
    if (webSearch) {
      print('  ✓ Web search enabled — research tasks will use live Google Search');
    } else {
      print('  ✓ Web search disabled — workers will use model knowledge only');
    }
  }

  // Save configuration
  print('\nSaving configuration...');
  
  const config = {
    provider: provider.id,
    model: provider.id === 'gemini' ? 'gemini-2.0-flash' :
           provider.id === 'openai' ? 'gpt-4o-mini' :
           provider.id === 'anthropic' ? 'claude-3-haiku-20240307' :
           provider.id === 'groq' ? 'llama-3.1-70b-versatile' : 'default',
    envVar: provider.envVar,
    apiKey,
    maxNodes,
    webSearch,
  };
  
  const configPath = saveConfig(config);
  print(`  ✓ Configuration saved to: ${configPath}`);
  
  // Run diagnostics/tests
  print('\n' + '═'.repeat(50));
  print('\nStep 4: Verifying installation...\n');
  
  const { runDiagnostics, printReport } = require('../lib/diagnostics');
  
  // Run diagnostics with API key in environment
  process.env[provider.envVar] = apiKey;
  
  const report = await runDiagnostics({
    runTests: true,
    skipE2e: false,
  });
  
  printReport(report);
  
  // Save machine profile to config
  const profilePath = path.join(CONFIG_DIR, 'swarm-profile.json');
  fs.writeFileSync(profilePath, JSON.stringify({
    generatedAt: new Date().toISOString(),
    machine: report.machine,
    recommendations: report.recommendations,
    config: {
      maxNodes: maxNodes,
      provider: provider.id,
    }
  }, null, 2));
  print(`  Machine profile saved to: ${profilePath}`);
  
  // Final status
  print('\n' + '═'.repeat(50));
  
  if (report.status === 'ok') {
    print('');
    box([
      '✅ Setup complete! All tests passed.',
      '',
      'Swarm is ready to use. Try:',
      '  "Research the top 5 AI companies"',
      '  "Analyze these 10 URLs in parallel"',
      '',
      `Config: ${configPath}`,
    ], '🎉 Success!');
  } else if (report.status === 'warning') {
    print('');
    box([
      '⚠️  Setup complete with warnings.',
      '',
      'Swarm should work, but check the warnings above.',
      '',
      `Config: ${configPath}`,
    ], '⚠️  Warning');
  } else {
    print('');
    box([
      '❌ Setup complete but tests failed.',
      '',
      'See errors above. You may need to fix issues',
      'before Swarm will work correctly.',
      '',
      'Run diagnostics again: node bin/diagnose.js',
    ], '❌ Issues Found');
  }
  
  print('');
  
  rl.close();
  process.exit(report.status === 'error' ? 1 : 0);
}

main().catch(err => {
  console.error('Setup failed:', err.message);
  process.exit(1);
});
