"""Supermetrics API client for marketing data.

Provides simple functions to query marketing data from 100+ platforms.
"""

import json
import os
from pathlib import Path
from typing import Any
from urllib.error import HTTPError, URLError
from urllib.request import Request, urlopen

BASE_URL = "https://mcp.supermetrics.com"
ENV_FILE = Path.home() / ".openclaw" / "skills" / "supermetrics-openclawd" / ".env"


def _load_env_file() -> dict[str, str]:
    """Load variables from .env file."""
    env_vars: dict[str, str] = {}
    if ENV_FILE.exists():
        for line in ENV_FILE.read_text().splitlines():
            line = line.strip()
            if line and not line.startswith("#") and "=" in line:
                key, _, value = line.partition("=")
                env_vars[key.strip()] = value.strip().strip("\"'")
    return env_vars


def _get_api_key() -> str:
    """Get API key from environment or .env file."""
    key = os.environ.get("SUPERMETRICS_API_KEY")
    if not key:
        env_vars = _load_env_file()
        key = env_vars.get("SUPERMETRICS_API_KEY")
    if not key:
        raise ValueError(
            f"SUPERMETRICS_API_KEY not found in environment or {ENV_FILE}"
        )
    return key


def _call_api(tool_name: str, params: dict[str, Any] | None = None) -> dict[str, Any]:
    """Call a Supermetrics API endpoint."""
    url = f"{BASE_URL}/mcp/{tool_name}"
    headers = {
        "Authorization": f"Bearer {_get_api_key()}",
        "Content-Type": "application/json",
    }
    data = json.dumps(params or {}).encode()

    req = Request(url, data=data, headers=headers, method="POST")
    try:
        with urlopen(req, timeout=60) as resp:
            result: dict[str, Any] = json.loads(resp.read().decode())
            return result
    except HTTPError as e:
        error_body = e.read().decode() if e.fp else str(e)
        return {"success": False, "error": f"HTTP {e.code}: {error_body}"}
    except URLError as e:
        return {"success": False, "error": f"Connection error: {e.reason}"}
    except Exception as e:
        return {"success": False, "error": str(e)}


def discover_sources() -> dict[str, Any]:
    """List all available marketing data sources.

    Returns dict with 'sources' array containing platform info.
    Each source has: id, name, description, categories.

    Example:
        result = discover_sources()
        for src in result['data']['sources']:
            print(f"{src['id']}: {src['name']}")
    """
    return _call_api("data_source_discovery")


def discover_accounts(ds_id: str) -> dict[str, Any]:
    """Get connected accounts for a data source.

    Args:
        ds_id: Data source ID (e.g., 'FA', 'AW', 'GAWA', 'GA4')

    Common IDs:
        FA = Meta Ads (Facebook)
        AW = Google Ads
        GAWA = Google Analytics
        GA4 = Google Analytics 4
        LI = LinkedIn Ads
        AC = Microsoft Advertising

    Example:
        result = discover_accounts("GAWA")
        for acc in result['data']['accounts']:
            print(f"{acc['account_id']}: {acc['account_name']}")
    """
    return _call_api("accounts_discovery", {"ds_id": ds_id})


def discover_fields(ds_id: str, field_type: str | None = None) -> dict[str, Any]:
    """Get available metrics and dimensions for a data source.

    Args:
        ds_id: Data source ID
        field_type: Optional filter - 'metric' or 'dimension'

    Example:
        result = discover_fields("GAWA", "metric")
        for field in result['data']['metrics']:
            print(f"{field['id']}: {field['name']}")
    """
    params: dict[str, Any] = {"ds_id": ds_id}
    if field_type:
        params["field_type"] = field_type
    return _call_api("field_discovery", params)


def query_data(
    ds_id: str,
    ds_accounts: str | list[str],
    fields: str | list[str],
    date_range_type: str = "last_7_days",
    start_date: str | None = None,
    end_date: str | None = None,
    filters: str | None = None,
    timezone: str | None = None,
) -> dict[str, Any]:
    """Execute a marketing data query.

    Returns a schedule_id for async result retrieval.

    Args:
        ds_id: Data source ID
        ds_accounts: Account ID(s) from discover_accounts()
        fields: Field ID(s) from discover_fields()
        date_range_type: 'last_7_days', 'last_30_days', 'last_3_months', 'custom', etc.
        start_date: Required if date_range_type='custom' (YYYY-MM-DD)
        end_date: Required if date_range_type='custom' (YYYY-MM-DD)
        filters: Filter expression (e.g., "country == United States")
        timezone: IANA timezone (e.g., 'America/New_York')

    Example:
        result = query_data(
            ds_id="GAWA",
            ds_accounts="123456789",
            fields=["date", "sessions", "pageviews"],
            date_range_type="last_7_days"
        )
        schedule_id = result['data']['schedule_id']
    """
    params: dict[str, Any] = {
        "ds_id": ds_id,
        "ds_accounts": ds_accounts,
        "fields": fields,
        "date_range_type": date_range_type,
    }
    if start_date:
        params["start_date"] = start_date
    if end_date:
        params["end_date"] = end_date
    if filters:
        params["filters"] = filters
    if timezone:
        params["timezone"] = timezone
    return _call_api("data_query", params)


def get_results(schedule_id: str) -> dict[str, Any]:
    """Get results from a data query.

    Args:
        schedule_id: Exact ID from query_data() response. Copy verbatim.

    Example:
        result = get_results("abc123-def456")
        if result['success']:
            for row in result['data']['data']:
                print(row)
    """
    return _call_api("get_async_query_results", {"schedule_id": schedule_id})


def get_today() -> dict[str, Any]:
    """Get current UTC date for date calculations.

    Returns date, iso_string, and timestamp.
    """
    return _call_api("get_today")


def search(query: str) -> dict[str, Any]:
    """Search across Supermetrics resources.

    Returns guidance and suggestions based on the query.

    Args:
        query: Search query string

    Example:
        result = search("facebook ads metrics")
    """
    return _call_api("search", {"query": query})


def health() -> dict[str, Any]:
    """Check Supermetrics server health status.

    Returns:
        Dict with 'status' and 'service' fields.

    Example:
        result = health()
        print(result['status'])  # "healthy"
    """
    url = f"{BASE_URL}/health"
    headers = {"Authorization": f"Bearer {_get_api_key()}"}
    req = Request(url, headers=headers, method="GET")
    try:
        with urlopen(req, timeout=10) as resp:
            result: dict[str, Any] = json.loads(resp.read().decode())
            return {"success": True, "data": result}
    except HTTPError as e:
        return {"success": False, "error": f"HTTP {e.code}"}
    except URLError as e:
        return {"success": False, "error": f"Connection error: {e.reason}"}
    except Exception as e:
        return {"success": False, "error": str(e)}
