---
name: supermetrics
description: "Official Supermetrics skill. Query marketing data from 100+ platforms including Google Analytics, Meta Ads, Google Ads, and LinkedIn. Requires API key."
version: 1.0.1
triggers:
  - marketing data
  - supermetrics
  - analytics
  - ads performance
  - campaign metrics
  - google analytics
  - meta ads
  - facebook ads
  - google ads
  - linkedin ads
  - marketing report
author: supermetrics
tags: [marketing, analytics, supermetrics, api, data]
requires:
  env:
    - SUPERMETRICS_API_KEY
---

# Supermetrics Marketing Data

Query marketing data from 100+ platforms including Google Analytics, Meta Ads, Google Ads, and LinkedIn.

## Usage

Import the helper module:

```python
from supermetrics import (
    discover_sources,
    discover_accounts,
    discover_fields,
    query_data,
    get_results,
    get_today,
    search,
    health,
)
```

## Functions

### discover_sources()

List all available marketing platforms.

```python
result = discover_sources()
for src in result['data']['sources']:
    print(f"{src['id']}: {src['name']}")
```

### discover_accounts(ds_id)

Get connected accounts for a data source.

**Common data source IDs:**
| ID | Platform |
|----|----------|
| FA | Meta Ads (Facebook) |
| AW | Google Ads |
| GAWA | Google Analytics |
| GA4 | Google Analytics 4 |
| LI | LinkedIn Ads |
| AC | Microsoft Advertising (Bing) |

```python
result = discover_accounts("GAWA")
for acc in result['data']['accounts']:
    print(f"{acc['account_id']}: {acc['account_name']}")
```

### discover_fields(ds_id, field_type=None)

Get available metrics and dimensions.

```python
# Get all fields
result = discover_fields("GAWA")

# Get only metrics
result = discover_fields("GAWA", "metric")

# Get only dimensions
result = discover_fields("GAWA", "dimension")
```

### query_data(...)

Execute a marketing data query. Returns schedule_id for async retrieval.

```python
result = query_data(
    ds_id="GAWA",
    ds_accounts="123456789",
    fields=["date", "sessions", "pageviews", "users"],
    date_range_type="last_7_days"
)
schedule_id = result['data']['schedule_id']
```

**Parameters:**
- `ds_id` (required): Data source ID
- `ds_accounts` (required): Account ID(s) from discover_accounts()
- `fields` (required): Field ID(s) from discover_fields()
- `date_range_type`: `last_7_days`, `last_30_days`, `last_3_months`, `custom`
- `start_date`, `end_date`: For custom date range (YYYY-MM-DD)
- `filters`: Filter expression (e.g., `"country == United States"`)
- `timezone`: IANA timezone (e.g., `"America/New_York"`)

**Filter operators:**
- `==`, `!=` - equals, not equals
- `>`, `>=`, `<`, `<=` - comparisons
- `=@`, `!@` - contains, does not contain
- `=~`, `!~` - regex match

### get_results(schedule_id)

Retrieve query results.

```python
result = get_results(schedule_id)
for row in result['data']['data']:
    print(row)
```

### get_today()

Get current UTC date for date calculations.

```python
result = get_today()
print(result['data']['date'])  # "2026-02-03"
```

### search(query)

Search across Supermetrics resources for guidance and suggestions.

```python
result = search("facebook ads metrics")
print(result['data'])
```

### health()

Check Supermetrics server health status.

```python
result = health()
print(result['data']['status'])  # "healthy"
```

## Workflow Example

```python
from supermetrics import (
    discover_accounts,
    discover_fields,
    query_data,
    get_results,
)

# 1. Find accounts
accounts = discover_accounts("GAWA")
account_id = accounts['data']['accounts'][0]['account_id']

# 2. See available fields
fields = discover_fields("GAWA", "metric")
print([f['id'] for f in fields['data']['metrics'][:5]])

# 3. Query data
query = query_data(
    ds_id="GAWA",
    ds_accounts=account_id,
    fields=["date", "sessions", "users", "pageviews"],
    date_range_type="last_7_days"
)

# 4. Get results
data = get_results(query['data']['schedule_id'])
for row in data['data']['data']:
    print(row)
```

## Response Format

All functions return:

```python
{"success": True, "data": {...}}  # Success
{"success": False, "error": "..."}  # Error
```
