# Reading the Rich ABI YAMLs

Each YAML is a self-contained contract reference. Here's how to parse them.

## Root structure

```
# Header comment — contract name, description, key notes
meta:             # name, version, source, implements, inherits, deployments
# == Section ==   # Grouped functions (these are the core content)
events:           # All events the contract emits
errors:           # Complete error index
```

Three root keys are reserved: `meta`, `events`, `errors`. Every other
root-level key is a **function**.

## Function entries

```yaml
createFlow:
  # Description of what the function does.
  notes:
    - "Gotcha: Non-obvious behavior or edge cases listed here as structured data."
  mutability: nonpayable     # view | pure | nonpayable | payable
  access: sender | operator  # who can call (omitted for view/pure)
  inputs:
    - token: address
    - receiver: address
    - flowRate: int96        # inline comments for non-obvious params
    - ctx: bytes
  outputs:
    - newCtx: bytes
  emits: [FlowUpdated, FlowUpdatedExtension]   # ordered by emission sequence
  errors: [CFA_FLOW_ALREADY_EXISTS, CFA_INVALID_FLOW_RATE]  # ordered by check sequence
```

Fields appear in this order: description comment, `notes`, `mutability`,
`access`, `inputs`, `outputs`, `emits`, `errors`. All are omitted when not
applicable.

## Key conventions

- **`ctx: bytes` parameter** = function is called through the Host
  (`callAgreement` / `batchCall`), never directly.
- **`access` labels**: `anyone`, `host`, `self`, `admin`, `governance`,
  `sender`, `receiver`, `operator`, `manager`, `pic`, `agreement`,
  `trusted-forwarder`, `factory`, `super-app`. Combine with `|`. Conditional:
  `anyone(if-critical-or-jailed)`.
- **`emits` and `errors` ordering** carries meaning: matches execution flow,
  not alphabetical. First errors in the list are the most likely causes.
- **`notes:` field** on functions (and `meta.notes:` at contract level)
  lists non-obvious behavior, common mistakes, or edge cases. Always read
  these carefully.
- **`meta.source`** is an array of raw GitHub URLs to the Solidity source files
  (implementation, interface, base — filenames are self-documenting).
- **`meta.deployments`** has per-network addresses split into `mainnet` and
  `testnet` subgroups.

## Events section

```yaml
events:
  FlowUpdated:
    indexed:              # log topics (filterable)
      - token: address
      - sender: address
    data:                 # log payload
      - flowRate: int96
```

## Errors section

```yaml
errors:
  # -- Category --
  - SIMPLE_ERROR                    # 0xabcd1234 — description
  - PARAMETERIZED_ERROR:            # errors with diagnostic data
      inputs:
        - value: uint256
```
