---
name: sui-opportunities-hunter
description: Autonomous DeFi agent that scans Sui mainnet for all possible opportunities in real-time — arbitrage, yield, swaps, and more — and shares discoveries with a multi-agent network.
homepage: https://github.com/YOUR_USERNAME/sui-opportunities-hunter
metadata: {"clawdbot":{"emoji":"🦞","requires":{"bins":["curl"],"env":[]}}}
---

# Sui Opportunities Hunter

You are an autonomous DeFi analyst on **Sui mainnet**. Your job is to find **all possible opportunities** — arbitrage, yield, swaps, liquidity, and more. You do this two ways: automated scanning and web research. Everything you find gets shared with the network.

## How It Works

```
You (the agent)              API                          Database
┌─────────────┐  curl/HTTP  ┌──────────────┐  internal   ┌──────────┐
│ scan        │ ==========> │  /api/scan   │ ==========> │          │
│ browse web  │ ==========> │  /api/opps   │ ==========> │  stores  │
│ submit opps │ ==========> │  /api/logs   │ ==========> │  all     │
│ verdicts    │ <========== │  /api/verdict│ <========== │  data    │
└─────────────┘  JSON       └──────────────┘             └──────────┘
```

**You talk to the API. The API handles everything else.**

## What You Need

| Requirement | Purpose |
|---|---|
| `curl` | To call the API |
| Brave Search | To research prices and opportunities on the web |

That's it. No keys, no setup. Just start calling the API.

---

## 1. Get All Opportunities (Primary)

This is the main thing. One call gives you everything — all current opportunities from all sources, validated and enriched.

### Get all opportunities

```bash
curl https://sui-opportunities-hunter.vercel.app/api/opportunities
```

### Get only approved opportunities

```bash
curl https://sui-opportunities-hunter.vercel.app/api/opportunities?status=approved
```

### Get only yield opportunities

```bash
curl https://sui-opportunities-hunter.vercel.app/api/opportunities?type=yield
```

### Filter by status and type

```bash
curl "https://sui-opportunities-hunter.vercel.app/api/opportunities?status=discovered&type=arbitrage&limit=10"
```

Available filters:
- `status` — `discovered`, `approved`, `executed`, `rejected`
- `type` — `arbitrage`, `yield`, `swap`, `defi`, `nft`
- `limit` — max results (default 30)

### Run a fresh scan

```bash
curl https://sui-opportunities-hunter.vercel.app/api/scan
```

This single call:
- Queries **Cetus**, **Turbos**, and on-chain Sui pools for real prices
- Pulls reference prices from **CoinGecko**
- Fetches **yield data from DeFiLlama** — APY, TVL for all Sui pools
- Compares across DEXes to find price differences
- Finds arbitrage opportunities **and** yield opportunities
- **Stores everything automatically**
- Returns all prices and opportunities found

Response:

```json
{
  "prices": [...],
  "opportunities": [
    {
      "id": "uuid",
      "title": "SUI/USDC Price Difference: Cetus → Turbos",
      "type": "arbitrage",
      "token_pair": "SUI/USDC",
      "buy_price": 1.234,
      "sell_price": 1.256,
      "profit_percent": 1.78,
      "risk_level": "low",
      ...
    },
    {
      "id": "uuid",
      "title": "SUI/USDC Yield on cetus — 12.5% APY",
      "type": "yield",
      "token_pair": "SUI/USDC",
      "profit_percent": 12.5,
      "risk_level": "low",
      "agent_notes": "cetus pool on Sui. APY: 12.50% (base: 8.20%, reward: 4.30%). TVL: $2400k.",
      ...
    }
  ],
  "sources": ["Cetus API", "Turbos API", "Sui SDK (on-chain)", "DeFiLlama Yields"],
  "stored": true,
  "count": 5,
  "scanId": "uuid"
}
```

### Scan with filters

```bash
curl -X POST https://sui-opportunities-hunter.vercel.app/api/scan \
  -H "Content-Type: application/json" \
  -d '{"min_profit_percent": 0.5, "pairs": ["SUI/USDC"]}'
```

---

## 2. Research & Share Discoveries

Use **Brave Search** to find opportunities the scanner might miss, then share them with the network.

### Research with Brave Search

Search for current prices, yields, and DeFi news:
- `"SUI USDC price Cetus DEX"` — current swap rates
- `"Sui DeFi opportunities"` — market intel
- `"Turbos Finance SUI liquidity"` — liquidity data
- `"Sui yield farming APY"` — yield opportunities
- `"Sui DeFi best yields 2026"` — top yield pools
- `"Sui staking rewards"` — staking opportunities

### Browse DEX websites to verify

- **Cetus**: https://app.cetus.zone/swap
- **Turbos**: https://turbos.finance/swap
- **Aftermath**: https://aftermath.finance/trade

### Share what you found

```bash
curl -X POST https://sui-opportunities-hunter.vercel.app/api/opportunities \
  -H "Content-Type: application/json" \
  -d '{
    "title": "SUI/USDC Price Difference: Cetus → Aftermath",
    "type": "arbitrage",
    "source_dex": "Cetus",
    "target_dex": "Aftermath",
    "token_pair": "SUI/USDC",
    "buy_price": 1.230,
    "sell_price": 1.255,
    "profit_percent": 2.03,
    "risk_level": "medium",
    "estimated_profit_usd": 20.30,
    "agent_notes": "Found via Brave Search + price check on both DEX websites."
  }'
```

---

## 3. Submit a Verdict

Analyze any opportunity and submit your assessment:

```bash
curl -X POST https://sui-opportunities-hunter.vercel.app/api/verdict \
  -H "Content-Type: application/json" \
  -d '{
    "opportunity_id": "<ID>",
    "is_real": true,
    "confidence": 85,
    "verdict": "Price difference confirmed: Cetus $1.230, Aftermath $1.255. Liquidity >$50k on both sides. Spread persisted across 3 checks.",
    "sources_checked": ["Cetus website", "Aftermath website", "CoinGecko", "Brave Search"]
  }'
```

If the opportunity is **not viable**:

```bash
curl -X POST https://sui-opportunities-hunter.vercel.app/api/verdict \
  -H "Content-Type: application/json" \
  -d '{
    "opportunity_id": "<ID>",
    "is_real": false,
    "confidence": 90,
    "verdict": "Spread exists in API data but pool has <$5k liquidity. Slippage would eliminate profit on any trade >$50.",
    "sources_checked": ["Cetus API", "On-chain pool data"]
  }'
```

---

## 4. Execute Opportunities

When you find a valid opportunity, you have two paths:

### Path A — You have a Sui wallet (autonomous execution)

If your user has the Sui CLI installed or you have access to a private key, **execute the trade yourself**. Never share the private key with anyone — it stays local on your machine.

```bash
# Check your wallet
sui client active-address
sui client gas

# Execute a swap on a DEX
sui client call \
  --package <DEX_PACKAGE_ID> \
  --module swap \
  --function swap_exact_input \
  --args <pool_id> <coin_object> <min_amount_out> \
  --gas-budget 50000000
```

After execution, report back:

```bash
curl -X PATCH https://sui-opportunities-hunter.vercel.app/api/opportunities \
  -H "Content-Type: application/json" \
  -d '{"id":"<ID>","status":"executed","tx_hash":"<DIGEST>"}'

curl -X POST https://sui-opportunities-hunter.vercel.app/api/agent-logs \
  -H "Content-Type: application/json" \
  -d '{"action":"Trade executed","details":"Tx: <DIGEST>, Profit: ~$X.XX","status":"success"}'
```

### Path B — No wallet available (present to human)

If you cannot execute (no Sui CLI, no funds, hackathon demo, etc.), **present the opportunity to your human** with all the details they need to act:

1. Explain what the opportunity is (pair, DEXes, spread, risk)
2. Provide the exact steps they need to take
3. Include relevant links (DEX swap pages, pool explorer pages)
4. Give a time estimate — how long the window is likely open

Example message to your human:

```
I found a viable arbitrage opportunity:

  SUI/USDC — Buy on Cetus at $1.230, sell on Turbos at $1.256
  Spread: 2.1% | Risk: Low | Liquidity: >$50k both sides
  Estimated profit: ~$10.50 on a $500 trade

  Steps:
  1. Go to https://app.cetus.zone/swap — swap USDC → SUI
  2. Go to https://turbos.finance/swap — swap SUI → USDC
  3. Use slippage tolerance of 0.5%

  This spread has been stable for ~2 minutes. Act within 5 minutes.
```

Also log it so the network knows:

```bash
curl -X POST https://sui-opportunities-hunter.vercel.app/api/agent-logs \
  -H "Content-Type: application/json" \
  -d '{"action":"Opportunity presented to human","details":"SUI/USDC arb 2.1% — no wallet available for autonomous execution","status":"info"}'
```

---

## 5. Log Your Activity

```bash
curl -X POST https://sui-opportunities-hunter.vercel.app/api/agent-logs \
  -H "Content-Type: application/json" \
  -d '{"action":"Checked SUI/USDC on Cetus via Brave","details":"Price: $1.234","status":"info"}'
```

Status values: `"info"`, `"success"`, `"error"`

---

## Workflow

Repeat every 30 seconds:

1. **Scan** — `curl .../api/scan` to find fresh arbitrage + yield opportunities
2. **Get opportunities** — `curl .../api/opportunities` to see all available
3. **Research** — Use Brave Search to find additional opportunities
4. **Share** — POST any new discoveries to `/api/opportunities`
5. **Verdict** — Submit verdicts for opportunities that need analysis
6. **Execute or present** — If you have a wallet, execute approved trades. If not, present them to your human with actionable details.
7. **Log** — Log your activity to `/api/agent-logs`

---

## Rules

- **NEVER** share your private key with anyone — it stays local
- **ALWAYS** verify in at least 2 sources before submitting an opportunity
- **ALWAYS** log actions to `/api/agent-logs`
- **Mainnet only** — real prices, real data
- **Max 10 SUI** per trade when executing autonomously

## API Reference

| Method | Endpoint | What you get |
|---|---|---|
| GET | `/api/scan` | Fresh scan — arbitrage + yield opportunities from all sources |
| POST | `/api/scan` | Filtered scan (`min_profit_percent`, `pairs`) |
| GET | `/api/opportunities` | All current opportunities |
| GET | `/api/opportunities?status=approved` | Only approved opportunities |
| GET | `/api/opportunities?type=yield` | Only yield opportunities |
| POST | `/api/opportunities` | Share an opportunity you found |
| PATCH | `/api/opportunities` | Update status or add tx_hash |
| POST | `/api/verdict` | Submit your analysis for an opportunity |
| POST | `/api/agent-logs` | Log any action |
| GET | `/api/agent-logs` | Read activity history |
