# Sui Agent Wallet Skill

Give your AI agent its own Sui wallet to interact with DApps and sign transactions.

**GitHub:** <https://github.com/EasonC13-agent/sui-skills/tree/main/sui-agent-wallet>

## Architecture

```
Chrome Extension ◄──WebSocket──► Local Server ◄──API──► Agent
     │                                │
     ▼                                ▼
  DApp Page                    Key Management
  (Wallet Standard)            (Seed Phrase)
```

## Installation

```bash
cd <your-workspace>/skills/sui-agent-wallet

# Install server dependencies
cd server && bun install

# Start the server
bun run index.ts
```

Load Chrome Extension:
1. Open `chrome://extensions/`
2. Enable "Developer mode"
3. Click "Load unpacked"
4. Select the `extension/` folder

## First Launch

The server automatically generates a 12-word seed phrase and stores it in **macOS Keychain**:

```
═══════════════════════════════════════════════════════════
  🔐 NEW WALLET CREATED
═══════════════════════════════════════════════════════════

  Seed phrase stored securely in macOS Keychain.

  To view your seed phrase for backup:
    curl http://localhost:3847/mnemonic

  Or use macOS Keychain Access app:
    Service: sui-agent-wallet
    Account: mnemonic
═══════════════════════════════════════════════════════════
```

## Secure Storage

| Location | Contents |
|----------|----------|
| macOS Keychain | Seed phrase (encrypted) |
| `~/.sui-agent-wallet/wallet.json` | Account addresses, network settings (no sensitive data) |

View Keychain entry:
```bash
# Command line
security find-generic-password -s "sui-agent-wallet" -a "mnemonic" -w

# Or open Keychain Access app
# Search for "sui-agent-wallet"
```

## Agent API

### Wallet Info

```bash
# Get current address
curl http://localhost:3847/address

# Get balance
curl http://localhost:3847/balance

# Get seed phrase (for backup)
curl http://localhost:3847/mnemonic
```

### Account Management

```bash
# List all accounts
curl http://localhost:3847/accounts

# Create new account
curl -X POST http://localhost:3847/accounts

# Create account at specific index
curl -X POST http://localhost:3847/accounts \
  -H "Content-Type: application/json" \
  -d '{"index": 2}'

# Switch account
curl -X POST http://localhost:3847/accounts/switch \
  -H "Content-Type: application/json" \
  -d '{"index": 1}'
```

### Network Management

```bash
# Get current network
curl http://localhost:3847/network

# Switch network (mainnet | testnet | devnet | localnet)
curl -X POST http://localhost:3847/network \
  -H "Content-Type: application/json" \
  -d '{"network": "testnet"}'
```

### Get Test Coins (Faucet)

**Testnet:**
- Official Faucet: <https://faucet.testnet.sui.io/>
- Discord: Join [Sui Discord](https://discord.gg/sui), post your wallet address in `#testnet-faucet`
- CLI: `sui client faucet --address <YOUR_ADDRESS>`

**Devnet:**
- Official Faucet: <https://faucet.devnet.sui.io/>
- Discord: Post your wallet address in `#devnet-faucet`
- CLI: `sui client faucet --address <YOUR_ADDRESS>`

**Note**: Mainnet requires real SUI tokens and cannot use faucets.

### Transaction Signing

```bash
# View pending transactions
curl http://localhost:3847/pending

# View transaction details
curl http://localhost:3847/tx/<request-id>

# Approve transaction
curl -X POST http://localhost:3847/approve/<request-id>

# Reject transaction
curl -X POST http://localhost:3847/reject/<request-id>
```

### Import/Export

```bash
# Import seed phrase (WARNING: overwrites existing wallet!)
curl -X POST http://localhost:3847/import \
  -H "Content-Type: application/json" \
  -d '{"mnemonic": "your twelve word seed phrase here ..."}'
```

### CLI Integration (Direct Signing)

Sign unsigned transactions generated by Sui CLI:

```bash
# 1. Generate unsigned transaction (using Agent Wallet address)
AGENT_ADDR=$(curl -s localhost:3847/address | jq -r .address)
TX_BYTES=$(sui client publish --serialize-unsigned-transaction \
  --sender $AGENT_ADDR --gas-budget 100000000 | tail -1)

# 2. Sign and execute with Agent Wallet
curl -X POST http://localhost:3847/sign-and-execute \
  -H "Content-Type: application/json" \
  -d "{\"txBytes\": \"$TX_BYTES\"}"

# Or sign only without executing
curl -X POST http://localhost:3847/sign-raw \
  -H "Content-Type: application/json" \
  -d "{\"txBytes\": \"$TX_BYTES\"}"
```

Supported CLI commands:
- `sui client publish --serialize-unsigned-transaction`
- `sui client call --serialize-unsigned-transaction`
- `sui client transfer-sui --serialize-unsigned-transaction`

## Transaction Parsing

When a signing request comes in, the agent sees:

```json
{
  "id": "req_123",
  "method": "signTransaction",
  "origin": "http://localhost:5173",
  "payload": {
    "transaction": "{\"commands\":[{\"MoveCall\":{...}}]}",
    "chain": "sui:devnet"
  }
}
```

## Security Checklist

Before signing, verify:
- [ ] Is the target contract trustworthy?
- [ ] Is the amount reasonable?
- [ ] Are there suspicious coin transfers?
- [ ] Is the gas budget normal?

## Test DApp

Built-in Counter DApp for testing:

```bash
# Start frontend
cd test-dapp/frontend && pnpm dev

# Open http://localhost:5173
# 1. Connect Wallet → Select "Sui Agent Wallet"
# 2. Click "+1" → Sends a signing request
# 3. Agent uses /pending to view, /approve to sign
```

## Technical Details

### BIP44 Derivation Path

```
m/44'/784'/{accountIndex}'/0'/0'
```

- 784 = Sui's coin type
- Each accountIndex corresponds to one address

### Wallet Standard Features

Implemented Sui Wallet Standard features:
- `standard:connect`
- `standard:disconnect`
- `standard:events`
- `sui:signTransaction`
- `sui:signAndExecuteTransaction`
- `sui:signPersonalMessage`

### Event Notifications

When switching accounts or networks, the server notifies the Extension via WebSocket:
- `accountChanged` - Account changed
- `networkChanged` - Network changed

## Related Skills

This skill is part of the Sui development skill suite:

| Skill | Description |
|-------|-------------|
| [sui-decompile](https://clawhub.ai/EasonC13/sui-decompile) | Fetch and read on-chain contract source code |
| [sui-move](https://clawhub.ai/EasonC13/sui-move) | Write and deploy Move smart contracts |
| [sui-coverage](https://clawhub.ai/EasonC13/sui-coverage) | Analyze test coverage with security analysis |
| **sui-agent-wallet** | Build and test DApps frontend |

**Workflow:**
```
sui-decompile → sui-move → sui-coverage → sui-agent-wallet
    Study        Write      Test & Audit   Build DApps
```

All skills: <https://github.com/EasonC13-agent/sui-skills>
