#!/usr/bin/env python3
# OpenClaw Optimizer - Real-time Dashboard (290 lines)

import os
import sys
import time
import json
import threading
import curses

class OptimizerDashboard:
    def __init__(self, config_path='~/.clawdbot/optimizer/config.json'):
        self.config_path = os.path.expanduser(config_path)
        self.state = {
            'daily_budget': 0.0,
            'current_spend': 0.0,
            'tasks_executed': 0,
            'model_distribution': {},
            'circuit_breakers': {}
        }
        self.watch_mode = False
        self.update_interval = 5  # seconds

    def load_config(self):
        try:
            with open(self.config_path, 'r') as f:
                return json.load(f)
        except FileNotFoundError:
            return {}

    def fetch_live_metrics(self):
        # Simulate fetching real-time metrics
        # In production, replace with actual API/file read
        return {
            'daily_budget': 10.00,
            'current_spend': round(self.state.get('current_spend', 0.0), 2),
            'tasks_executed': self.state.get('tasks_executed', 0),
            'model_distribution': {
                'haiku': 70,
                'sonnet': 25,
                'opus': 5
            },
            'circuit_breakers': {
                'browser': False,
                'cost': False
            }
        }

    def render_dashboard(self, stdscr):
        curses.curs_set(0)
        stdscr.clear()

        metrics = self.fetch_live_metrics()

        # Budget Section
        stdscr.addstr(2, 2, f"💰 Daily Budget: ${metrics['daily_budget']:.2f}", curses.A_BOLD)
        stdscr.addstr(3, 2, f"💵 Current Spend: ${metrics['current_spend']:.2f}")
        spend_percentage = (metrics['current_spend'] / metrics['daily_budget']) * 100
        stdscr.addstr(4, 2, f"📊 Spend %: {spend_percentage:.1f}%")

        # Tasks Section
        stdscr.addstr(6, 2, f"🚀 Tasks Executed: {metrics['tasks_executed']}", curses.A_BOLD)

        # Model Distribution
        stdscr.addstr(8, 2, "🤖 Model Distribution:", curses.A_BOLD)
        y_pos = 9
        for model, percentage in metrics['model_distribution'].items():
            stdscr.addstr(y_pos, 4, f"{model.capitalize()}: {percentage}%")
            y_pos += 1

        # Circuit Breakers
        stdscr.addstr(13, 2, "🔒 Circuit Breakers:", curses.A_BOLD)
        y_pos = 14
        for breaker, status in metrics['circuit_breakers'].items():
            color = curses.color_pair(2) if status else curses.color_pair(1)
            stdscr.addstr(y_pos, 4, f"{breaker.capitalize()}: {'ACTIVE' if status else 'INACTIVE'}", color)
            y_pos += 1

        stdscr.refresh()

    def watch(self):
        self.watch_mode = True
        try:
            curses.wrapper(self.render_dashboard)
        except KeyboardInterrupt:
            print("\nStopped watching dashboard.")

def main():
    dashboard = OptimizerDashboard()
    
    if len(sys.argv) > 1 and sys.argv[1] == 'watch':
        dashboard.watch()
    else:
        # One-time display
        curses.wrapper(dashboard.render_dashboard)

if __name__ == '__main__':
    main()