const { encode } = require('tiktoken');

class SubagentContextCompactor {
  constructor(options = {}) {
    this.tokenThreshold = options.tokenThreshold || 50000;
    this.summaryModel = options.summaryModel || 'claude-3-5-haiku';
    this.preserveLastN = options.preserveLastN || 5;
    this.logger = options.logger || console;
    
    // Tracking for cross-session context
    this.sessionTokens = new Map();
    this.criticalPatterns = [
      'task objective',
      'error message',
      'code reference',
      'user constraint',
      'decision rationale'
    ];
  }

  // Count tokens in a message
  countTokens(message) {
    try {
      const encoder = encode('cl100k_base');
      return encoder.encode(JSON.stringify(message)).length;
    } catch (error) {
      this.logger.warn('Token counting failed, using estimate', error);
      return Math.ceil(message.length / 4); // Rough estimate
    }
  }

  // Track tokens across main and subagent sessions
  trackSessionTokens(sessionKey, message) {
    const currentTokens = this.sessionTokens.get(sessionKey) || 0;
    const messageTokens = this.countTokens(message);
    const totalTokens = currentTokens + messageTokens;
    
    this.sessionTokens.set(sessionKey, totalTokens);
    return totalTokens;
  }

  // Determine if compaction is needed
  shouldCompact(sessionKey) {
    const totalTokens = this.sessionTokens.get(sessionKey) || 0;
    return totalTokens >= this.tokenThreshold;
  }

  // Identify critical information to preserve
  extractCriticalContext(messages) {
    return messages.filter(msg => 
      this.criticalPatterns.some(pattern => 
        JSON.stringify(msg).toLowerCase().includes(pattern)
      )
    ).slice(-this.preserveLastN);
  }

  // Summarize context using a cost-effective model
  async compactContext(messages, sessionKey) {
    try {
      // Preserve critical recent context
      const criticalContext = this.extractCriticalContext(messages);
      
      // Use Haiku for cost-effective summarization
      const summary = await this.summarizeWithHaiku(messages);
      
      // Merge critical context with summary
      const compactedContext = [
        ...criticalContext,
        { role: 'system', content: summary }
      ];
      
      // Reset token tracking for this session
      this.sessionTokens.set(sessionKey, this.countTokens(compactedContext));
      
      // Log compaction event
      this.logger.info('Context Compacted', {
        sessionKey,
        originalTokens: this.countTokens(messages),
        compactedTokens: this.countTokens(compactedContext)
      });
      
      return compactedContext;
    } catch (error) {
      this.logger.error('Context Compaction Failed', error);
      // Fallback: return last N messages if compaction fails
      return messages.slice(-this.preserveLastN);
    }
  }

  // Placeholder for Haiku summarization (to be implemented with actual API call)
  async summarizeWithHaiku(messages) {
    const fullContext = messages.map(m => `${m.role}: ${m.content}`).join('\n\n');
    
    // Basic summarization strategy
    return `CONTEXT SUMMARY:\n
    - Original message count: ${messages.length}
    - Key themes detected: ${this.detectKeyThemes(fullContext)}
    - Context preserved: Most recent critical exchanges
    
    Recommendation: Refer to preserved critical context for precise details.`;
  }

  // Basic theme detection
  detectKeyThemes(context) {
    const themes = this.criticalPatterns.filter(pattern => 
      context.toLowerCase().includes(pattern)
    );
    return themes.length > 0 ? themes.join(', ') : 'General task context';
  }

  // Integration with sessions_spawn workflow
  async prepareForSubagent(fullContext, sessionKey) {
    if (this.shouldCompact(sessionKey)) {
      return this.compactContext(fullContext, sessionKey);
    }
    return fullContext;
  }
}

module.exports = { SubagentContextCompactor };