// OpenClaw Optimizer - Scheduler Component
// Automation wrapper with preflight/postflight (169 lines)
class OptimizerScheduler {
  constructor(config = {}) {
    this.maxRetries = config.maxRetries || 3;
    this.timeoutMs = config.timeoutMs || 300000; // 5 minutes default
    this.hooks = {
      preflight: config.preflight || [],
      postflight: config.postflight || []
    };
  }

  // Execute task with robust scheduling
  async execute(task, context = {}) {
    let attempt = 0;
    const startTime = Date.now();

    while (attempt < this.maxRetries) {
      try {
        // Run preflight hooks
        for (let hook of this.hooks.preflight) {
          await hook(context);
        }

        // Execute main task with timeout
        const result = await this.runWithTimeout(task, context);

        // Run postflight hooks
        for (let hook of this.hooks.postflight) {
          await hook({ context, result });
        }

        return result;
      } catch (error) {
        attempt++;
        
        // Log retry attempt
        console.log(`Task retry ${attempt}/${this.maxRetries}:`, error);
        
        // Exponential backoff
        const backoffMs = Math.pow(2, attempt) * 1000;
        await new Promise(resolve => setTimeout(resolve, backoffMs));
      }

      // Check overall timeout
      if (Date.now() - startTime > this.timeoutMs) {
        throw new Error('Task execution timed out');
      }
    }

    throw new Error(`Task failed after ${this.maxRetries} attempts`);
  }

  // Run task with configurable timeout
  async runWithTimeout(task, context) {
    return new Promise((resolve, reject) => {
      const taskPromise = typeof task === 'function' 
        ? task(context) 
        : Promise.resolve(task);

      const timeoutId = setTimeout(() => {
        reject(new Error('Task execution timed out'));
      }, this.timeoutMs);

      taskPromise
        .then((result) => {
          clearTimeout(timeoutId);
          resolve(result);
        })
        .catch((error) => {
          clearTimeout(timeoutId);
          reject(error);
        });
    });
  }

  // Add dynamic hooks
  addPreflight(hook) {
    this.hooks.preflight.push(hook);
  }

  addPostflight(hook) {
    this.hooks.postflight.push(hook);
  }
}

module.exports = { OptimizerScheduler };