// OpenClaw Optimizer - Router Component
// Task classification & model selection (179 lines)
class TaskRouter {
  constructor(config = {}) {
    this.models = {
      'haiku': 'anthropic/claude-3-5-haiku-latest',
      'sonnet': 'anthropic/claude-sonnet-4-5',
      'opus': 'anthropic/claude-opus-4-5'
    };
    this.defaultModel = config.defaultModel || this.models.haiku;
    this.complexityThresholds = {
      simple: ['lookup', 'calculation', 'short-response'],
      complex: ['analysis', 'code-generation', 'strategy', 'multi-step-reasoning']
    };
  }

  // Classify task complexity
  classifyTask(task) {
    const taskDescription = task.toLowerCase();
    
    // Check complex patterns first
    for (let pattern of this.complexityThresholds.complex) {
      if (taskDescription.includes(pattern)) {
        return this.models.sonnet; // Escalate to Sonnet
      }
    }

    // Check simple patterns
    for (let pattern of this.complexityThresholds.simple) {
      if (taskDescription.includes(pattern)) {
        return this.models.haiku; // Use Haiku
      }
    }

    // Default routing
    return this.defaultModel;
  }

  // Model selection with cost prediction
  selectModel(task, options = {}) {
    const selectedModel = this.classifyTask(task);
    
    return {
      model: selectedModel,
      estimatedCost: this.predictCost(selectedModel, task),
      rationale: `Routed based on task complexity: ${task}`
    };
  }

  // Predict approximate cost based on model and task
  predictCost(model, task) {
    const costMap = {
      [this.models.haiku]: 0.00025,
      [this.models.sonnet]: 0.003,
      [this.models.opus]: 0.015
    };
    
    const baseRate = costMap[model] || costMap[this.models.haiku];
    const taskLengthFactor = Math.min(task.length / 1000, 5); // Cap at 5x
    
    return baseRate * taskLengthFactor;
  }
}

module.exports = { TaskRouter };