// OpenClaw Optimizer - Browser Governor Component
// Browser serialization & loop prevention (212 lines)
class BrowserGovernor {
  constructor(config = {}) {
    this.maxConcurrentTabs = config.maxConcurrentTabs || 3;
    this.browserLocks = new Set();
    this.tabQueue = [];
    this.browserState = {
      availableTabs: 0,
      lockedTabs: 0,
      totalErrors: 0
    };
  }

  // Acquire browser lock
  async acquireLock(tabId) {
    // Wait if max concurrent tabs reached
    while (this.browserLocks.size >= this.maxConcurrentTabs) {
      await new Promise(resolve => setTimeout(resolve, 500));
    }

    // Check for existing lock
    if (this.browserLocks.has(tabId)) {
      throw new Error(`Tab ${tabId} already locked`);
    }

    this.browserLocks.add(tabId);
    this.browserState.lockedTabs++;

    return {
      release: () => this.releaseLock(tabId)
    };
  }

  // Release browser lock
  releaseLock(tabId) {
    if (this.browserLocks.has(tabId)) {
      this.browserLocks.delete(tabId);
      this.browserState.lockedTabs--;
    }
  }

  // Queue browser actions
  async queueBrowserAction(action) {
    return new Promise((resolve, reject) => {
      this.tabQueue.push({
        action,
        resolve,
        reject
      });
      this.processQueue();
    });
  }

  // Process queued browser actions
  async processQueue() {
    if (this.browserLocks.size < this.maxConcurrentTabs && this.tabQueue.length > 0) {
      const queuedAction = this.tabQueue.shift();
      
      try {
        const result = await queuedAction.action();
        queuedAction.resolve(result);
      } catch (error) {
        this.browserState.totalErrors++;
        queuedAction.reject(error);
      }
    }
  }

  // Circuit breaker for repeated failures
  isCircuitBreakerTripped() {
    // Trip circuit breaker if too many errors
    return this.browserState.totalErrors > 10;
  }

  // Get current browser state
  getState() {
    return {
      availableTabs: this.maxConcurrentTabs - this.browserLocks.size,
      lockedTabs: this.browserLocks.size,
      totalErrors: this.browserState.totalErrors,
      circuitBreakerActive: this.isCircuitBreakerTripped()
    };
  }
}

module.exports = { BrowserGovernor };