---
name: squareup
description: |
  Square API integration with managed OAuth. Process payments, manage customers, orders, catalog, inventory, and invoices.
  Use this skill when users want to accept payments, manage point-of-sale operations, track inventory, or handle invoicing through Square.
  For other third party apps, use the api-gateway skill (https://clawhub.ai/byungkyu/api-gateway).
  Requires network access and valid Maton API key.
metadata:
  author: maton
  version: "1.0"
  clawdbot:
    emoji: 🧠
    requires:
      env:
        - MATON_API_KEY
---

# Square

Access the Square API with managed OAuth authentication. Process payments, manage customers, orders, catalog items, inventory, and invoices.

## Quick Start

```bash
# List locations
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://gateway.maton.ai/squareup/v2/locations')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

## Base URL

```
https://gateway.maton.ai/squareup/{native-api-path}
```

Replace `{native-api-path}` with the actual Square API endpoint path. The gateway proxies requests to `connect.squareup.com` and automatically injects your OAuth token.

## Authentication

All requests require the Maton API key in the Authorization header:

```
Authorization: Bearer $MATON_API_KEY
```

**Environment Variable:** Set your API key as `MATON_API_KEY`:

```bash
export MATON_API_KEY="YOUR_API_KEY"
```

### Getting Your API Key

1. Sign in or create an account at [maton.ai](https://maton.ai)
2. Go to [maton.ai/settings](https://maton.ai/settings)
3. Copy your API key

## Connection Management

Manage your Square OAuth connections at `https://ctrl.maton.ai`.

### List Connections

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://ctrl.maton.ai/connections?app=squareup&status=ACTIVE')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

### Create Connection

```bash
python <<'EOF'
import urllib.request, os, json
data = json.dumps({'app': 'squareup'}).encode()
req = urllib.request.Request('https://ctrl.maton.ai/connections', data=data, method='POST')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
req.add_header('Content-Type', 'application/json')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

### Get Connection

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://ctrl.maton.ai/connections/{connection_id}')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

**Response:**
```json
{
  "connection": {
    "connection_id": "21fd90f9-5935-43cd-b6c8-bde9d915ca80",
    "status": "ACTIVE",
    "creation_time": "2025-12-08T07:20:53.488460Z",
    "last_updated_time": "2026-01-31T20:03:32.593153Z",
    "url": "https://connect.maton.ai/?session_token=...",
    "app": "squareup",
    "metadata": {}
  }
}
```

Open the returned `url` in a browser to complete OAuth authorization.

### Delete Connection

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://ctrl.maton.ai/connections/{connection_id}', method='DELETE')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

### Specifying Connection

If you have multiple Square connections, specify which one to use with the `Maton-Connection` header:

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://gateway.maton.ai/squareup/v2/locations')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
req.add_header('Maton-Connection', '21fd90f9-5935-43cd-b6c8-bde9d915ca80')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

If omitted, the gateway uses the default (oldest) active connection.

## API Reference

### Locations

#### List Locations

```bash
GET /squareup/v2/locations
```

#### Get Location

```bash
GET /squareup/v2/locations/{location_id}
```

#### Create Location

```bash
POST /squareup/v2/locations
Content-Type: application/json

{
  "location": {
    "name": "New Location",
    "address": {
      "address_line_1": "123 Main St",
      "locality": "San Francisco",
      "administrative_district_level_1": "CA",
      "postal_code": "94102",
      "country": "US"
    }
  }
}
```

#### Update Location

```bash
PUT /squareup/v2/locations/{location_id}
Content-Type: application/json

{
  "location": {
    "name": "Updated Location Name"
  }
}
```

### Merchants

#### Get Merchant

```bash
GET /squareup/v2/merchants/me
```

#### List Merchants

```bash
GET /squareup/v2/merchants
```

### Payments

#### List Payments

```bash
GET /squareup/v2/payments
```

With filters:

```bash
GET /squareup/v2/payments?location_id={location_id}&begin_time=2026-01-01T00:00:00Z&end_time=2026-02-01T00:00:00Z
```

#### Get Payment

```bash
GET /squareup/v2/payments/{payment_id}
```

#### Create Payment

```bash
POST /squareup/v2/payments
Content-Type: application/json

{
  "source_id": "cnon:card-nonce-ok",
  "idempotency_key": "unique-key-12345",
  "amount_money": {
    "amount": 1000,
    "currency": "USD"
  },
  "location_id": "{location_id}"
}
```

#### Update Payment

```bash
PUT /squareup/v2/payments/{payment_id}
Content-Type: application/json

{
  "payment": {
    "tip_money": {
      "amount": 200,
      "currency": "USD"
    }
  },
  "idempotency_key": "unique-key-67890"
}
```

#### Complete Payment

```bash
POST /squareup/v2/payments/{payment_id}/complete
Content-Type: application/json

{}
```

#### Cancel Payment

```bash
POST /squareup/v2/payments/{payment_id}/cancel
Content-Type: application/json

{}
```

### Refunds

#### List Refunds

```bash
GET /squareup/v2/refunds
```

#### Get Refund

```bash
GET /squareup/v2/refunds/{refund_id}
```

#### Create Refund

```bash
POST /squareup/v2/refunds
Content-Type: application/json

{
  "idempotency_key": "unique-refund-key",
  "payment_id": "{payment_id}",
  "amount_money": {
    "amount": 500,
    "currency": "USD"
  },
  "reason": "Customer requested refund"
}
```

### Customers

#### List Customers

```bash
GET /squareup/v2/customers
```

#### Get Customer

```bash
GET /squareup/v2/customers/{customer_id}
```

#### Create Customer

```bash
POST /squareup/v2/customers
Content-Type: application/json

{
  "given_name": "John",
  "family_name": "Doe",
  "email_address": "john.doe@example.com",
  "phone_number": "+15551234567"
}
```

#### Update Customer

```bash
PUT /squareup/v2/customers/{customer_id}
Content-Type: application/json

{
  "email_address": "john.updated@example.com"
}
```

#### Delete Customer

```bash
DELETE /squareup/v2/customers/{customer_id}
```

#### Search Customers

```bash
POST /squareup/v2/customers/search
Content-Type: application/json

{
  "query": {
    "filter": {
      "email_address": {
        "exact": "john.doe@example.com"
      }
    }
  }
}
```

### Orders

#### Create Order

```bash
POST /squareup/v2/orders
Content-Type: application/json

{
  "order": {
    "location_id": "{location_id}",
    "line_items": [
      {
        "name": "Item 1",
        "quantity": "1",
        "base_price_money": {
          "amount": 1000,
          "currency": "USD"
        }
      }
    ]
  },
  "idempotency_key": "unique-order-key"
}
```

#### Get Order

```bash
GET /squareup/v2/orders/{order_id}
```

#### Update Order

```bash
PUT /squareup/v2/orders/{order_id}
Content-Type: application/json

{
  "order": {
    "location_id": "{location_id}",
    "version": 1
  },
  "fields_to_clear": ["line_items"]
}
```

#### Search Orders

```bash
POST /squareup/v2/orders/search
Content-Type: application/json

{
  "location_ids": ["{location_id}"],
  "query": {
    "filter": {
      "state_filter": {
        "states": ["OPEN"]
      }
    }
  }
}
```

#### Batch Retrieve Orders

```bash
POST /squareup/v2/orders/batch-retrieve
Content-Type: application/json

{
  "location_id": "{location_id}",
  "order_ids": ["{order_id_1}", "{order_id_2}"]
}
```

### Catalog

#### List Catalog

```bash
GET /squareup/v2/catalog/list
```

With type filter:

```bash
GET /squareup/v2/catalog/list?types=ITEM,CATEGORY
```

#### Get Catalog Object

```bash
GET /squareup/v2/catalog/object/{object_id}
```

#### Upsert Catalog Object

```bash
POST /squareup/v2/catalog/object
Content-Type: application/json

{
  "idempotency_key": "unique-catalog-key",
  "object": {
    "type": "ITEM",
    "id": "#new-item",
    "item_data": {
      "name": "Coffee",
      "description": "Hot brewed coffee",
      "variations": [
        {
          "type": "ITEM_VARIATION",
          "id": "#small-coffee",
          "item_variation_data": {
            "name": "Small",
            "pricing_type": "FIXED_PRICING",
            "price_money": {
              "amount": 300,
              "currency": "USD"
            }
          }
        }
      ]
    }
  }
}
```

#### Delete Catalog Object

```bash
DELETE /squareup/v2/catalog/object/{object_id}
```

#### Batch Upsert Catalog Objects

```bash
POST /squareup/v2/catalog/batch-upsert
Content-Type: application/json

{
  "idempotency_key": "unique-batch-key",
  "batches": [
    {
      "objects": [...]
    }
  ]
}
```

#### Search Catalog Objects

```bash
POST /squareup/v2/catalog/search
Content-Type: application/json

{
  "object_types": ["ITEM"],
  "query": {
    "text_query": {
      "keywords": ["coffee"]
    }
  }
}
```

#### Get Catalog Info

```bash
GET /squareup/v2/catalog/info
```

### Inventory

#### Retrieve Inventory Count

```bash
GET /squareup/v2/inventory/{catalog_object_id}
```

#### Batch Retrieve Inventory Counts

```bash
POST /squareup/v2/inventory/counts/batch-retrieve
Content-Type: application/json

{
  "catalog_object_ids": ["{object_id_1}", "{object_id_2}"],
  "location_ids": ["{location_id}"]
}
```

#### Batch Change Inventory

```bash
POST /squareup/v2/inventory/changes/batch-create
Content-Type: application/json

{
  "idempotency_key": "unique-inventory-key",
  "changes": [
    {
      "type": "ADJUSTMENT",
      "adjustment": {
        "catalog_object_id": "{object_id}",
        "location_id": "{location_id}",
        "quantity": "10",
        "from_state": "NONE",
        "to_state": "IN_STOCK"
      }
    }
  ]
}
```

#### Retrieve Inventory Adjustment

```bash
GET /squareup/v2/inventory/adjustments/{adjustment_id}
```

### Invoices

#### List Invoices

```bash
GET /squareup/v2/invoices?location_id={location_id}
```

#### Get Invoice

```bash
GET /squareup/v2/invoices/{invoice_id}
```

#### Create Invoice

```bash
POST /squareup/v2/invoices
Content-Type: application/json

{
  "invoice": {
    "location_id": "{location_id}",
    "order_id": "{order_id}",
    "primary_recipient": {
      "customer_id": "{customer_id}"
    },
    "payment_requests": [
      {
        "request_type": "BALANCE",
        "due_date": "2026-02-15"
      }
    ],
    "delivery_method": "EMAIL"
  },
  "idempotency_key": "unique-invoice-key"
}
```

#### Update Invoice

```bash
PUT /squareup/v2/invoices/{invoice_id}
Content-Type: application/json

{
  "invoice": {
    "version": 1,
    "payment_requests": [
      {
        "uid": "{payment_request_uid}",
        "due_date": "2026-02-20"
      }
    ]
  },
  "idempotency_key": "unique-update-key"
}
```

#### Publish Invoice

```bash
POST /squareup/v2/invoices/{invoice_id}/publish
Content-Type: application/json

{
  "version": 1,
  "idempotency_key": "unique-publish-key"
}
```

#### Cancel Invoice

```bash
POST /squareup/v2/invoices/{invoice_id}/cancel
Content-Type: application/json

{
  "version": 1
}
```

#### Delete Invoice

```bash
DELETE /squareup/v2/invoices/{invoice_id}?version=1
```

#### Search Invoices

```bash
POST /squareup/v2/invoices/search
Content-Type: application/json

{
  "query": {
    "filter": {
      "location_ids": ["{location_id}"],
      "customer_ids": ["{customer_id}"]
    }
  }
}
```

## Pagination

Square uses cursor-based pagination. List endpoints return a `cursor` field when more results exist:

```bash
GET /squareup/v2/payments?cursor={cursor_value}
```

Response includes pagination info:

```json
{
  "payments": [...],
  "cursor": "next_page_cursor_value"
}
```

Continue fetching by passing the cursor value in subsequent requests until no cursor is returned.

## Code Examples

### JavaScript

```javascript
const response = await fetch(
  'https://gateway.maton.ai/squareup/v2/locations',
  {
    headers: {
      'Authorization': `Bearer ${process.env.MATON_API_KEY}`
    }
  }
);
const data = await response.json();
```

### Python

```python
import os
import requests

response = requests.get(
    'https://gateway.maton.ai/squareup/v2/locations',
    headers={'Authorization': f'Bearer {os.environ["MATON_API_KEY"]}'}
)
data = response.json()
```

## Notes

- All amounts are in the smallest currency unit (e.g., cents for USD: 1000 = $10.00)
- IDs are alphanumeric strings
- Timestamps are in ISO 8601 format (e.g., `2026-02-07T01:59:28.459Z`)
- Most write operations require an `idempotency_key` to prevent duplicate operations
- Some endpoints require specific OAuth scopes (CUSTOMERS_READ, ORDERS_READ, ITEMS_READ, INVOICES_READ, etc.)
- IMPORTANT: When using curl commands, use `curl -g` when URLs contain brackets to disable glob parsing
- IMPORTANT: When piping curl output to `jq` or other commands, environment variables like `$MATON_API_KEY` may not expand correctly in some shell environments

## Error Handling

| Status | Meaning |
|--------|---------|
| 400 | Missing Square connection or bad request |
| 401 | Invalid or missing Maton API key |
| 403 | Insufficient OAuth scopes |
| 404 | Resource not found |
| 429 | Rate limited |
| 4xx/5xx | Passthrough error from Square API |

### Error Response Format

```json
{
  "errors": [
    {
      "category": "INVALID_REQUEST_ERROR",
      "code": "NOT_FOUND",
      "detail": "Could not find payment with id: {payment_id}"
    }
  ]
}
```

### Troubleshooting: API Key Issues

1. Check that the `MATON_API_KEY` environment variable is set:

```bash
echo $MATON_API_KEY
```

2. Verify the API key is valid by listing connections:

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://ctrl.maton.ai/connections')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

### Troubleshooting: Invalid App Name

1. Ensure your URL path starts with `squareup`. For example:

- Correct: `https://gateway.maton.ai/squareup/v2/locations`
- Incorrect: `https://gateway.maton.ai/v2/locations`

### Troubleshooting: Insufficient Scopes

If you receive a 403 error with `INSUFFICIENT_SCOPES`, the OAuth connection doesn't have the required permissions. Create a new connection and ensure you grant all necessary permissions during OAuth authorization.

## Resources

- [Square API Overview](https://developer.squareup.com/docs)
- [Square API Reference](https://developer.squareup.com/reference/square)
- [Payments API](https://developer.squareup.com/reference/square/payments-api)
- [Customers API](https://developer.squareup.com/reference/square/customers-api)
- [Orders API](https://developer.squareup.com/reference/square/orders-api)
- [Catalog API](https://developer.squareup.com/reference/square/catalog-api)
- [Inventory API](https://developer.squareup.com/reference/square/inventory-api)
- [Invoices API](https://developer.squareup.com/reference/square/invoices-api)
- [Locations API](https://developer.squareup.com/reference/square/locations-api)
- [Maton Community](https://discord.com/invite/dBfFAcefs2)
- [Maton Support](mailto:support@maton.ai)
