#!/bin/bash
# speedtest-social.sh - Run speedtest and format for social media

set -euo pipefail

# Check if speedtest is installed
if ! command -v speedtest &> /dev/null; then
    echo "Error: speedtest CLI not found. Install with: brew install speedtest" >&2
    exit 1
fi

# Parse arguments
POST_TO_MOLTBOOK=false
if [[ "${1:-}" == "--post-to-moltbook" ]]; then
    POST_TO_MOLTBOOK=true
fi

# Run speedtest and capture JSON output
echo "Running speedtest..." >&2
RESULT=$(speedtest --format=json --progress=no)

# Parse JSON (using jq if available, otherwise basic parsing)
if command -v jq &> /dev/null; then
    DOWNLOAD=$(echo "$RESULT" | jq -r '.download.bandwidth' | awk '{printf "%.1f", $1 * 8 / 1000000}')
    UPLOAD=$(echo "$RESULT" | jq -r '.upload.bandwidth' | awk '{printf "%.1f", $1 * 8 / 1000000}')
    LATENCY=$(echo "$RESULT" | jq -r '.ping.latency' | awk '{printf "%.0f", $1}')
    PACKET_LOSS=$(echo "$RESULT" | jq -r '.packetLoss // 0')
    SERVER_NAME=$(echo "$RESULT" | jq -r '.server.name')
    SERVER_LOCATION=$(echo "$RESULT" | jq -r '.server.location')
else
    # Fallback: basic grep/awk parsing
    DOWNLOAD=$(echo "$RESULT" | grep -o '"download"[^}]*"bandwidth":[0-9]*' | grep -o '[0-9]*$' | awk '{printf "%.1f", $1 * 8 / 1000000}')
    UPLOAD=$(echo "$RESULT" | grep -o '"upload"[^}]*"bandwidth":[0-9]*' | grep -o '[0-9]*$' | awk '{printf "%.1f", $1 * 8 / 1000000}')
    LATENCY=$(echo "$RESULT" | grep -o '"latency":[0-9.]*' | grep -o '[0-9.]*$' | awk '{printf "%.0f", $1}')
    SERVER_NAME=$(echo "$RESULT" | grep -o '"name":"[^"]*"' | head -1 | cut -d'"' -f4)
    SERVER_LOCATION=$(echo "$RESULT" | grep -o '"location":"[^"]*"' | head -1 | cut -d'"' -f4)
    PACKET_LOSS=0
fi

# Determine status emoji
if (( $(echo "$DOWNLOAD > 100" | bc -l) )) && (( $(echo "$LATENCY < 20" | bc -l) )); then
    STATUS="🚀 Excellent"
elif (( $(echo "$DOWNLOAD > 25" | bc -l) )) && (( $(echo "$LATENCY < 50" | bc -l) )); then
    STATUS="⚡ Good"
else
    STATUS="🐌 Slow"
fi

# Format the post
POST_TEXT="📊 SpeedTest Results
⬇️ Download: ${DOWNLOAD} Mbps
⬆️ Upload: ${UPLOAD} Mbps
⏱️ Latency: ${LATENCY}ms"

if (( $(echo "$PACKET_LOSS > 0" | bc -l) )); then
    POST_TEXT="${POST_TEXT}
📉 Packet Loss: ${PACKET_LOSS}%"
fi

POST_TEXT="${POST_TEXT}
📍 Server: ${SERVER_NAME}, ${SERVER_LOCATION}
${STATUS}

#SpeedTest #AgentInfra 🦞"

# Output the formatted post
echo "$POST_TEXT"

# Save to history
HISTORY_FILE="${HOME}/.openclaw/data/speedtest-history.jsonl"
mkdir -p "$(dirname "$HISTORY_FILE")"
TIMESTAMP=$(date -u +"%Y-%m-%dT%H:%M:%SZ")
echo "{\"timestamp\":\"$TIMESTAMP\",\"download\":$DOWNLOAD,\"upload\":$UPLOAD,\"latency\":$LATENCY,\"packet_loss\":$PACKET_LOSS,\"server\":\"$SERVER_NAME\",\"location\":\"$SERVER_LOCATION\"}" >> "$HISTORY_FILE"

# Ask user about publishing (if not already specified via flag)
if [ "$POST_TO_MOLTBOOK" = false ]; then
    echo "" >&2
    echo "📢 Would you like to publish these results?" >&2
    echo "   1) Moltbook" >&2
    echo "   2) Twitter" >&2
    echo "   3) Both" >&2
    echo "   4) Skip" >&2
    echo -n "Choice (1-4): " >&2
    read -r CHOICE
    
    case "$CHOICE" in
        1)
            POST_TO_MOLTBOOK=true
            POST_TO_TWITTER=false
            ;;
        2)
            POST_TO_MOLTBOOK=false
            POST_TO_TWITTER=true
            ;;
        3)
            POST_TO_MOLTBOOK=true
            POST_TO_TWITTER=true
            ;;
        *)
            echo "Skipping social post." >&2
            exit 0
            ;;
    esac
fi

# Post to Moltbook
if [ "$POST_TO_MOLTBOOK" = true ]; then
    echo "" >&2
    echo "Posting to Moltbook..." >&2
    
    # Check for credentials
    if [ ! -f "${HOME}/.config/moltbook/credentials.json" ]; then
        echo "Error: Moltbook credentials not found at ~/.config/moltbook/credentials.json" >&2
    else
        API_KEY=$(grep -o '"api_key":"[^"]*"' "${HOME}/.config/moltbook/credentials.json" | cut -d'"' -f4)
        
        # Post to Moltbook
        RESPONSE=$(curl -s -X POST https://www.moltbook.com/api/v1/posts \
            -H "Authorization: Bearer $API_KEY" \
            -H "Content-Type: application/json" \
            -d "{\"submolt\": \"general\", \"title\": \"📊 SpeedTest Results\", \"content\": $(echo "$POST_TEXT" | jq -Rs .)}")
        
        echo "$RESPONSE" | jq -r '.message // .error' >&2
        
        # Extract post URL if successful
        if echo "$RESPONSE" | jq -e '.success' > /dev/null 2>&1; then
            POST_ID=$(echo "$RESPONSE" | jq -r '.post.id')
            echo "🔗 https://www.moltbook.com/post/$POST_ID" >&2
        fi
    fi
fi

# Post to Twitter
if [ "${POST_TO_TWITTER:-false}" = true ]; then
    echo "" >&2
    echo "Posting to Twitter..." >&2
    
    # Check if bird CLI is available
    if ! command -v bird &> /dev/null; then
        echo "Error: bird CLI not found. Install with: brew install steipete/tap/bird" >&2
    else
        # Shortened tweet version
        TWEET_TEXT="📊 SpeedTest Results
⬇️ ${DOWNLOAD} Mbps ⬆️ ${UPLOAD} Mbps ⏱️ ${LATENCY}ms
📍 ${SERVER_LOCATION}
${STATUS}

#SpeedTest #AgentInfra 🦞"
        
        bird tweet "$TWEET_TEXT" 2>&1 | grep -E '(✅|❌|https://x.com)' >&2
    fi
fi
