#!/bin/bash
# speedtest-history.sh - Track speedtest results over time

set -euo pipefail

HISTORY_FILE="${HOME}/.openclaw/data/speedtest-history.jsonl"

# Ensure history file exists
mkdir -p "$(dirname "$HISTORY_FILE")"
touch "$HISTORY_FILE"

ACTION="${1:-run}"

case "$ACTION" in
    run)
        echo "Running speedtest and saving to history..."
        if ! command -v speedtest &> /dev/null; then
            echo "Error: speedtest CLI not found" >&2
            exit 1
        fi
        
        RESULT=$(speedtest --format=json --progress=no)
        
        if command -v jq &> /dev/null; then
            DOWNLOAD=$(echo "$RESULT" | jq -r '.download.bandwidth' | awk '{printf "%.1f", $1 * 8 / 1000000}')
            UPLOAD=$(echo "$RESULT" | jq -r '.upload.bandwidth' | awk '{printf "%.1f", $1 * 8 / 1000000}')
            LATENCY=$(echo "$RESULT" | jq -r '.ping.latency' | awk '{printf "%.0f", $1}')
            PACKET_LOSS=$(echo "$RESULT" | jq -r '.packetLoss // 0')
            SERVER_NAME=$(echo "$RESULT" | jq -r '.server.name')
            SERVER_LOCATION=$(echo "$RESULT" | jq -r '.server.location')
        else
            echo "Warning: jq not found, using basic parsing" >&2
            DOWNLOAD=$(echo "$RESULT" | grep -o '"download"[^}]*"bandwidth":[0-9]*' | grep -o '[0-9]*$' | awk '{printf "%.1f", $1 * 8 / 1000000}')
            UPLOAD=$(echo "$RESULT" | grep -o '"upload"[^}]*"bandwidth":[0-9]*' | grep -o '[0-9]*$' | awk '{printf "%.1f", $1 * 8 / 1000000}')
            LATENCY=$(echo "$RESULT" | grep -o '"latency":[0-9.]*' | grep -o '[0-9.]*$' | awk '{printf "%.0f", $1}')
            SERVER_NAME="Unknown"
            SERVER_LOCATION="Unknown"
            PACKET_LOSS=0
        fi
        
        TIMESTAMP=$(date -u +"%Y-%m-%dT%H:%M:%SZ")
        echo "{\"timestamp\":\"$TIMESTAMP\",\"download\":$DOWNLOAD,\"upload\":$UPLOAD,\"latency\":$LATENCY,\"packet_loss\":$PACKET_LOSS,\"server\":\"$SERVER_NAME\",\"location\":\"$SERVER_LOCATION\"}" >> "$HISTORY_FILE"
        
        echo "✅ Saved: Download ${DOWNLOAD} Mbps, Upload ${UPLOAD} Mbps, Latency ${LATENCY}ms"
        ;;
        
    stats)
        if [ ! -s "$HISTORY_FILE" ]; then
            echo "No history found. Run 'speedtest-history.sh run' first."
            exit 0
        fi
        
        echo "📊 SpeedTest Statistics"
        echo ""
        
        if command -v jq &> /dev/null; then
            COUNT=$(wc -l < "$HISTORY_FILE" | tr -d ' ')
            AVG_DOWN=$(jq -s 'map(.download) | add / length' "$HISTORY_FILE" | awk '{printf "%.1f", $1}')
            AVG_UP=$(jq -s 'map(.upload) | add / length' "$HISTORY_FILE" | awk '{printf "%.1f", $1}')
            AVG_LAT=$(jq -s 'map(.latency) | add / length' "$HISTORY_FILE" | awk '{printf "%.0f", $1}')
            MAX_DOWN=$(jq -s 'map(.download) | max' "$HISTORY_FILE" | awk '{printf "%.1f", $1}')
            MIN_DOWN=$(jq -s 'map(.download) | min' "$HISTORY_FILE" | awk '{printf "%.1f", $1}')
            
            echo "Total tests: $COUNT"
            echo ""
            echo "⬇️ Download:"
            echo "   Average: ${AVG_DOWN} Mbps"
            echo "   Max: ${MAX_DOWN} Mbps"
            echo "   Min: ${MIN_DOWN} Mbps"
            echo ""
            echo "⬆️ Upload:"
            echo "   Average: ${AVG_UP} Mbps"
            echo ""
            echo "⏱️ Latency:"
            echo "   Average: ${AVG_LAT}ms"
        else
            echo "Install jq for detailed statistics: brew install jq"
            echo "Total tests: $(wc -l < "$HISTORY_FILE")"
        fi
        ;;
        
    trend)
        if [ ! -s "$HISTORY_FILE" ]; then
            echo "No history found. Run 'speedtest-history.sh run' first."
            exit 0
        fi
        
        echo "📈 Recent Speed Trend (last 10 tests)"
        echo ""
        
        if command -v jq &> /dev/null; then
            tail -10 "$HISTORY_FILE" | jq -r '"⬇️ " + (.download|tostring) + " Mbps | ⬆️ " + (.upload|tostring) + " Mbps | ⏱️ " + (.latency|tostring) + "ms | " + .timestamp'
        else
            echo "Install jq for formatted output: brew install jq"
            tail -10 "$HISTORY_FILE"
        fi
        ;;
        
    *)
        echo "Usage: speedtest-history.sh [run|stats|trend]"
        echo ""
        echo "  run    - Run speedtest and save to history"
        echo "  stats  - Show statistics (average, min, max)"
        echo "  trend  - Show recent trend (last 10 tests)"
        exit 1
        ;;
esac
