# SpecVibe Reference: Backend Development

This guide outlines the standards for building robust, scalable, and maintainable backend services.

## 1. Architecture: Layered & Decoupled

A layered architecture separates concerns, making the codebase easier to understand, test, and maintain. Follow this structure:

1.  **Controller/Router Layer**: The entry point for HTTP requests. Its only job is to parse the request (headers, body, params), call the appropriate service, and send the response. **No business logic here.**
2.  **Service Layer**: Contains the core business logic. It orchestrates calls to the data access layer and other services. It knows *what* to do, but not *how* the data is stored.
3.  **Data Access Layer (DAL/Repository)**: Responsible for all communication with the database. This is the only layer that should contain database-specific code (e.g., Prisma, SQL queries).

```
Request -> Controller -> Service -> Data Access Layer -> Database
```

## 2. API Implementation

- **Follow the Contract**: Your API implementation MUST strictly adhere to the `openapi.yaml` specification. All request parameters, body structures, and response formats must match the contract.
- **Middleware for Common Tasks**: Use middleware for cross-cutting concerns like authentication, logging, and error handling. This keeps your endpoint handlers clean and focused on their specific task.

## 3. Database & ORM

- **Use an ORM**: Use a modern ORM like Prisma or Drizzle. This provides type safety and abstracts away much of the boilerplate SQL.
- **Migrations are Mandatory**: NEVER change the database schema manually in production. All schema changes MUST be managed through migration files generated by the ORM (`prisma migrate dev`). These files must be committed to version control.
- **Optimize Queries**: 
    - **Indexing**: Add indexes to columns that are frequently used in `WHERE` clauses, `JOIN`s, or `ORDER BY` operations. Use compound indexes for queries that filter on multiple columns. This is the single most important performance optimization.
    - **N+1 Problem**: Be aware of the N+1 query problem when fetching related data in a loop. Use the ORM's features for eager loading (e.g., Prisma's `include` or `select`) to fetch all necessary data in a constant number of queries.
    - **Connection Pooling**: Ensure your database connection pool is properly configured to handle the expected concurrent traffic.

## 4. Error Handling

Refer to `references/07-error-handling.md` for detailed guidelines. The key principle is to handle errors gracefully and provide meaningful feedback to the client.

## 5. Configuration

- **Environment Variables**: All configuration, especially secrets (API keys, database URLs, JWT secrets), MUST be loaded from environment variables. DO NOT hardcode them.
- **Use a `.env` file** for local development, and ensure a `.env.example` file is committed to the repository to document required variables.
