#!/usr/bin/env bash
#
# test_upgrade.sh — Tests for upgrade.sh
#
# Verifies:
#   - SPACESUIT sections are updated with new base content
#   - User content outside markers is PRESERVED
#   - Version marker is updated
#   - Files without markers get markers added (migration case)

set -euo pipefail

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
REPO_DIR="$(cd "$SCRIPT_DIR/.." && pwd)"
INSTALL_SCRIPT="$REPO_DIR/scripts/install.sh"
UPGRADE_SCRIPT="$REPO_DIR/scripts/upgrade.sh"

TMPDIR_BASE="$(mktemp -d)"
trap 'rm -rf "$TMPDIR_BASE"' EXIT

ERRORS=0

fail() {
  echo "    FAIL: $1"
  ERRORS=$((ERRORS + 1))
}

pass() {
  echo "    ok: $1"
}

echo "  Test: upgrade.sh"

# --- Setup: Create an installed workspace ---
WORKSPACE="$TMPDIR_BASE/ws1"
mkdir -p "$WORKSPACE"
bash "$INSTALL_SCRIPT" "$WORKSPACE" > /dev/null 2>&1

# --- Test 1: User content outside markers is preserved after upgrade ---
echo ""
echo "  [1] User content outside markers is preserved"

USER_CONTENT="## My Custom Section
This is user-added content that should survive upgrades.
<!-- CANARY: upgrade-test-preserve -->"

echo "$USER_CONTENT" >> "$WORKSPACE/AGENTS.md"

bash "$UPGRADE_SCRIPT" "$WORKSPACE" > /dev/null 2>&1

if grep -qF "CANARY: upgrade-test-preserve" "$WORKSPACE/AGENTS.md"; then
  pass "User content preserved in AGENTS.md"
else
  fail "User content lost in AGENTS.md after upgrade"
fi

if grep -qF "My Custom Section" "$WORKSPACE/AGENTS.md"; then
  pass "User section heading preserved"
else
  fail "User section heading lost after upgrade"
fi

# --- Test 2: Base content is updated when changed ---
echo ""
echo "  [2] SPACESUIT sections are updated with new base content"

# Create a modified base dir to simulate upstream update
FAKE_SPACESUIT="$TMPDIR_BASE/fake-spacesuit"
cp -r "$REPO_DIR" "$FAKE_SPACESUIT"

# Modify the base content
echo "# UPDATED AGENTS.md — v999" > "$FAKE_SPACESUIT/base/AGENTS.md"
echo "This is the new framework content." >> "$FAKE_SPACESUIT/base/AGENTS.md"
echo "<!-- UPGRADE_CANARY: new-content -->" >> "$FAKE_SPACESUIT/base/AGENTS.md"

# Run upgrade using the fake spacesuit
bash "$FAKE_SPACESUIT/scripts/upgrade.sh" "$WORKSPACE" > /dev/null 2>&1

if grep -qF "UPGRADE_CANARY: new-content" "$WORKSPACE/AGENTS.md"; then
  pass "New base content applied"
else
  fail "New base content NOT applied after upgrade"
fi

# Also verify user content still there
if grep -qF "CANARY: upgrade-test-preserve" "$WORKSPACE/AGENTS.md"; then
  pass "User content still preserved after base update"
else
  fail "User content lost after base update"
fi

# --- Test 3: Version marker is updated ---
echo ""
echo "  [3] Version marker is updated"

echo "99.99.99" > "$FAKE_SPACESUIT/VERSION"
bash "$FAKE_SPACESUIT/scripts/upgrade.sh" "$WORKSPACE" > /dev/null 2>&1

INSTALLED_VERSION="$(cat "$WORKSPACE/.spacesuit-version")"
if [[ "$INSTALLED_VERSION" == "99.99.99" ]]; then
  pass ".spacesuit-version updated to 99.99.99"
else
  fail ".spacesuit-version is '$INSTALLED_VERSION', expected '99.99.99'"
fi

# --- Test 4: Files without markers get markers added (migration) ---
echo ""
echo "  [4] Files without markers get markers added"

WORKSPACE2="$TMPDIR_BASE/ws2"
mkdir -p "$WORKSPACE2"

# Create a bare AGENTS.md without any markers (simulates pre-spacesuit file)
cat > "$WORKSPACE2/AGENTS.md" << 'EOF'
# My Old AGENTS.md

This file existed before spacesuit was installed.
It has no markers at all.

## My Custom Rules

- Do this
- Do that
EOF

bash "$UPGRADE_SCRIPT" "$WORKSPACE2" > /dev/null 2>&1

if grep -qF "<!-- SPACESUIT:BEGIN AGENTS -->" "$WORKSPACE2/AGENTS.md"; then
  pass "SPACESUIT:BEGIN marker added to unmarked file"
else
  fail "SPACESUIT:BEGIN marker NOT added to unmarked file"
fi

if grep -qF "<!-- SPACESUIT:END -->" "$WORKSPACE2/AGENTS.md"; then
  pass "SPACESUIT:END marker added to unmarked file"
else
  fail "SPACESUIT:END marker NOT added to unmarked file"
fi

# Original content should still be there
if grep -qF "My Old AGENTS.md" "$WORKSPACE2/AGENTS.md"; then
  pass "Original content preserved after marker addition"
else
  fail "Original content lost after marker addition"
fi

if grep -qF "My Custom Rules" "$WORKSPACE2/AGENTS.md"; then
  pass "Custom rules preserved after marker addition"
else
  fail "Custom rules lost after marker addition"
fi

# --- Test 5: Upgrade is idempotent ---
echo ""
echo "  [5] Upgrade is idempotent (running twice produces same result)"

WORKSPACE3="$TMPDIR_BASE/ws3"
mkdir -p "$WORKSPACE3"
bash "$INSTALL_SCRIPT" "$WORKSPACE3" > /dev/null 2>&1

bash "$UPGRADE_SCRIPT" "$WORKSPACE3" > /dev/null 2>&1
cp "$WORKSPACE3/AGENTS.md" "$TMPDIR_BASE/agents_after_first.md"

bash "$UPGRADE_SCRIPT" "$WORKSPACE3" > /dev/null 2>&1
cp "$WORKSPACE3/AGENTS.md" "$TMPDIR_BASE/agents_after_second.md"

if diff -q "$TMPDIR_BASE/agents_after_first.md" "$TMPDIR_BASE/agents_after_second.md" > /dev/null 2>&1; then
  pass "AGENTS.md identical after two upgrades"
else
  fail "AGENTS.md changed after second upgrade (not idempotent)"
fi

# --- Summary ---
echo ""
if [[ $ERRORS -gt 0 ]]; then
  echo "  ❌ test_upgrade.sh: $ERRORS failure(s)"
  exit 1
else
  echo "  ✅ test_upgrade.sh: all checks passed"
  exit 0
fi
