#!/usr/bin/env bash
#
# test_markers.sh — Marker integrity tests
#
# Verifies:
#   - All files in base/ exist and are non-empty
#   - All template files with base references have matching SPACESUIT:BEGIN/END markers
#   - VERSION file exists and is valid semver
#   - SKILL.md has required ClawHub fields

set -euo pipefail

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
REPO_DIR="$(cd "$SCRIPT_DIR/.." && pwd)"

ERRORS=0

fail() {
  echo "    FAIL: $1"
  ERRORS=$((ERRORS + 1))
}

pass() {
  echo "    ok: $1"
}

echo "  Test: marker integrity"

# --- Test 1: All base files exist and are non-empty ---
echo ""
echo "  [1] All base/ files exist and are non-empty"

if [[ ! -d "$REPO_DIR/base" ]]; then
  fail "base/ directory does not exist"
else
  file_count=0
  for f in "$REPO_DIR/base"/*.md; do
    if [[ ! -f "$f" ]]; then
      continue
    fi
    file_count=$((file_count + 1))
    fname="$(basename "$f")"

    if [[ -s "$f" ]]; then
      pass "base/$fname is non-empty ($(wc -l < "$f") lines)"
    else
      fail "base/$fname is empty"
    fi
  done

  if [[ $file_count -eq 0 ]]; then
    fail "No .md files found in base/"
  else
    pass "$file_count base files found"
  fi
fi

# --- Test 2: Template files have matching markers ---
echo ""
echo "  [2] Template files have matching SPACESUIT markers"

for tmpl in "$REPO_DIR/templates"/*.md; do
  if [[ ! -f "$tmpl" ]]; then
    continue
  fi
  tmpl_name="$(basename "$tmpl")"

  # Check if template references any SPACESUIT_BASE_ placeholder
  if grep -q '{{SPACESUIT_BASE_' "$tmpl"; then
    # Extract section name from placeholder
    section_names=$(grep -o '{{SPACESUIT_BASE_[A-Z_]*}}' "$tmpl" | sed 's/{{SPACESUIT_BASE_//;s/}}//')

    for section in $section_names; do
      if grep -qF "<!-- SPACESUIT:BEGIN $section -->" "$tmpl"; then
        pass "$tmpl_name has BEGIN marker for $section"
      else
        fail "$tmpl_name missing BEGIN marker for $section"
      fi

      if grep -qF "<!-- SPACESUIT:END -->" "$tmpl"; then
        pass "$tmpl_name has END marker"
      else
        fail "$tmpl_name missing END marker"
      fi
    done
  fi
done

# --- Test 3: Marker pairs are balanced ---
echo ""
echo "  [3] Marker pairs are balanced in templates"

for tmpl in "$REPO_DIR/templates"/*.md; do
  if [[ ! -f "$tmpl" ]]; then
    continue
  fi
  tmpl_name="$(basename "$tmpl")"

  begin_count=$(grep -c 'SPACESUIT:BEGIN' "$tmpl" 2>/dev/null || true)
  begin_count=${begin_count:-0}
  begin_count=$(echo "$begin_count" | tr -d '[:space:]')
  end_count=$(grep -c 'SPACESUIT:END' "$tmpl" 2>/dev/null || true)
  end_count=${end_count:-0}
  end_count=$(echo "$end_count" | tr -d '[:space:]')

  if [[ "$begin_count" -eq "$end_count" ]]; then
    if [[ "$begin_count" -gt 0 ]]; then
      pass "$tmpl_name has balanced markers ($begin_count pairs)"
    fi
  else
    fail "$tmpl_name has unbalanced markers (BEGIN=$begin_count, END=$end_count)"
  fi
done

# --- Test 4: VERSION file exists and is valid semver ---
echo ""
echo "  [4] VERSION file is valid semver"

VERSION_FILE="$REPO_DIR/VERSION"
if [[ ! -f "$VERSION_FILE" ]]; then
  fail "VERSION file does not exist"
else
  VERSION_CONTENT="$(cat "$VERSION_FILE" | tr -d '[:space:]')"
  # Match semver: MAJOR.MINOR.PATCH (optional -prerelease and +buildmeta)
  if [[ "$VERSION_CONTENT" =~ ^[0-9]+\.[0-9]+\.[0-9]+(-[a-zA-Z0-9.]+)?(\+[a-zA-Z0-9.]+)?$ ]]; then
    pass "VERSION is valid semver: $VERSION_CONTENT"
  else
    fail "VERSION is not valid semver: '$VERSION_CONTENT'"
  fi
fi

# --- Test 5: SKILL.md has required ClawHub fields ---
echo ""
echo "  [5] SKILL.md has required ClawHub fields"

SKILL_FILE="$REPO_DIR/SKILL.md"
if [[ ! -f "$SKILL_FILE" ]]; then
  fail "SKILL.md does not exist"
else
  REQUIRED_FIELDS=("Name" "Version" "Author" "License" "Category" "Description")

  for field in "${REQUIRED_FIELDS[@]}"; do
    if grep -qi "$field" "$SKILL_FILE"; then
      pass "SKILL.md contains '$field'"
    else
      fail "SKILL.md missing required field '$field'"
    fi
  done
fi

# --- Test 6: Each base file has a corresponding template ---
echo ""
echo "  [6] Base files have corresponding templates"

for base_file in "$REPO_DIR/base"/*.md; do
  if [[ ! -f "$base_file" ]]; then
    continue
  fi
  fname="$(basename "$base_file")"
  tmpl_path="$REPO_DIR/templates/${fname}"

  if [[ -f "$tmpl_path" ]]; then
    pass "base/$fname has template templates/$fname"
  else
    fail "base/$fname has no corresponding template (expected templates/$fname)"
  fi
done

# --- Summary ---
echo ""
if [[ $ERRORS -gt 0 ]]; then
  echo "  ❌ test_markers.sh: $ERRORS failure(s)"
  exit 1
else
  echo "  ✅ test_markers.sh: all checks passed"
  exit 0
fi
