#!/usr/bin/env bash
#
# test_install.sh — Tests for install.sh
#
# Verifies:
#   - All expected files are created
#   - Files contain SPACESUIT:BEGIN and SPACESUIT:END markers
#   - Base content is present between markers
#   - User placeholder sections exist outside markers
#   - Running install again does NOT overwrite existing files

set -euo pipefail

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
REPO_DIR="$(cd "$SCRIPT_DIR/.." && pwd)"
INSTALL_SCRIPT="$REPO_DIR/scripts/install.sh"

TMPDIR_BASE="$(mktemp -d)"
trap 'rm -rf "$TMPDIR_BASE"' EXIT

ERRORS=0

fail() {
  echo "    FAIL: $1"
  ERRORS=$((ERRORS + 1))
}

pass() {
  echo "    ok: $1"
}

echo "  Test: install.sh"

# --- Test 1: Fresh install creates all expected files ---
echo ""
echo "  [1] Fresh install creates all expected files"

WORKSPACE="$TMPDIR_BASE/ws1"
mkdir -p "$WORKSPACE"

# Run install (suppress output)
bash "$INSTALL_SCRIPT" "$WORKSPACE" > /dev/null 2>&1

# Expected files
EXPECTED_FILES=(
  "AGENTS.md"
  "SOUL.md"
  "TOOLS.md"
  "HEARTBEAT.md"
  "SECURITY.md"
  "MEMORY.md"
  "IDENTITY.md"
  "USER.md"
  "Makefile"
  ".spacesuit-version"
)

for f in "${EXPECTED_FILES[@]}"; do
  if [[ -f "$WORKSPACE/$f" ]]; then
    pass "$f exists"
  else
    fail "$f was not created"
  fi
done

# Expected directories
EXPECTED_DIRS=(
  "memory"
  "handoff/pending"
  "handoff/completed"
  "decisions"
  "agents"
  "people"
  "scripts"
  "state"
)

for d in "${EXPECTED_DIRS[@]}"; do
  if [[ -d "$WORKSPACE/$d" ]]; then
    pass "directory $d exists"
  else
    fail "directory $d was not created"
  fi
done

# --- Test 2: Marker files have SPACESUIT:BEGIN and SPACESUIT:END ---
echo ""
echo "  [2] Installed files contain SPACESUIT markers"

MARKER_FILES=("AGENTS.md" "SOUL.md" "TOOLS.md" "HEARTBEAT.md" "SECURITY.md" "MEMORY.md")

for f in "${MARKER_FILES[@]}"; do
  target="$WORKSPACE/$f"
  if [[ ! -f "$target" ]]; then
    fail "$f missing — can't check markers"
    continue
  fi

  section_name="${f%.md}"
  section_name="${section_name^^}"  # uppercase

  if grep -qF "<!-- SPACESUIT:BEGIN $section_name -->" "$target"; then
    pass "$f has SPACESUIT:BEGIN $section_name"
  else
    fail "$f missing SPACESUIT:BEGIN $section_name marker"
  fi

  if grep -qF "<!-- SPACESUIT:END -->" "$target"; then
    pass "$f has SPACESUIT:END"
  else
    fail "$f missing SPACESUIT:END marker"
  fi
done

# --- Test 3: Base content is present between markers ---
echo ""
echo "  [3] Base content is present between markers"

for f in "${MARKER_FILES[@]}"; do
  target="$WORKSPACE/$f"
  base_file="$REPO_DIR/base/$f"

  if [[ ! -f "$base_file" ]]; then
    continue
  fi

  # Get first non-empty line from base content
  first_line="$(grep -m1 '.' "$base_file")"

  if grep -qF "$first_line" "$target"; then
    pass "$f contains base content"
  else
    fail "$f does not contain expected base content"
  fi
done

# --- Test 4: User placeholder sections exist outside markers ---
echo ""
echo "  [4] User customization sections exist"

# AGENTS.md should have user sections below the markers
if grep -q "YOUR CUSTOMIZATIONS BELOW" "$WORKSPACE/AGENTS.md"; then
  pass "AGENTS.md has user customization section"
else
  fail "AGENTS.md missing user customization section"
fi

# --- Test 5: Idempotency — running install again does NOT overwrite ---
echo ""
echo "  [5] Re-running install does not overwrite existing files"

# Add a canary string to AGENTS.md
echo "<!-- CANARY: user-added content -->" >> "$WORKSPACE/AGENTS.md"

# Run install again
bash "$INSTALL_SCRIPT" "$WORKSPACE" > /dev/null 2>&1

if grep -qF "CANARY: user-added content" "$WORKSPACE/AGENTS.md"; then
  pass "AGENTS.md preserved after re-install"
else
  fail "AGENTS.md was overwritten by re-install"
fi

# --- Test 6: Version file is written ---
echo ""
echo "  [6] Version tracking"

EXPECTED_VERSION="$(cat "$REPO_DIR/VERSION")"
INSTALLED_VERSION="$(cat "$WORKSPACE/.spacesuit-version")"

if [[ "$INSTALLED_VERSION" == "$EXPECTED_VERSION" ]]; then
  pass ".spacesuit-version matches VERSION ($EXPECTED_VERSION)"
else
  fail ".spacesuit-version ($INSTALLED_VERSION) != VERSION ($EXPECTED_VERSION)"
fi

# --- Summary ---
echo ""
if [[ $ERRORS -gt 0 ]]; then
  echo "  ❌ test_install.sh: $ERRORS failure(s)"
  exit 1
else
  echo "  ✅ test_install.sh: all checks passed"
  exit 0
fi
