#!/usr/bin/env bash
#
# spacesuit upgrade.sh — Section-based merge
#
# Replaces content between <!-- SPACESUIT:BEGIN ... --> and <!-- SPACESUIT:END -->
# markers with the latest base content. Preserves everything outside markers.
#
# If no markers exist in a root file, prepends the base content with markers at top.
#
# Usage: ./scripts/upgrade.sh [--workspace /path/to/workspace] [--dry-run]

set -euo pipefail

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
SPACESUIT_DIR="$(cd "$SCRIPT_DIR/.." && pwd)"
BASE_DIR="$SPACESUIT_DIR/base"
VERSION="$(cat "$SPACESUIT_DIR/VERSION")"

DRY_RUN=false
WORKSPACE=""

# Parse args
while [[ $# -gt 0 ]]; do
  case "$1" in
    --dry-run) DRY_RUN=true; shift ;;
    --workspace) WORKSPACE="$2"; shift 2 ;;
    *) WORKSPACE="$1"; shift ;;
  esac
done

# Default workspace is two levels up (skills/spacesuit/ → workspace root)
WORKSPACE="${WORKSPACE:-$(cd "$SPACESUIT_DIR/../.." && pwd)}"

echo "🔄 Spacesuit Upgrade v${VERSION}"
echo "   Workspace: $WORKSPACE"
if $DRY_RUN; then
  echo "   Mode: DRY RUN (no changes will be made)"
fi
echo ""

# Track changes
CHANGED=0
SKIPPED=0
ADDED=0

# Map: section name → base file
declare -A SECTION_MAP=(
  ["AGENTS"]="AGENTS.md"
  ["SOUL"]="SOUL.md"
  ["TOOLS"]="TOOLS.md"
  ["HEARTBEAT"]="HEARTBEAT.md"
  ["SECURITY"]="SECURITY.md"
  ["MEMORY"]="MEMORY.md"
)

# Map: section name → target file in workspace
declare -A TARGET_MAP=(
  ["AGENTS"]="AGENTS.md"
  ["SOUL"]="SOUL.md"
  ["TOOLS"]="TOOLS.md"
  ["HEARTBEAT"]="HEARTBEAT.md"
  ["SECURITY"]="SECURITY.md"
  ["MEMORY"]="MEMORY.md"
)

upgrade_section() {
  local section="$1"
  local base_file="$BASE_DIR/${SECTION_MAP[$section]}"
  local target_file="$WORKSPACE/${TARGET_MAP[$section]}"

  if [[ ! -f "$base_file" ]]; then
    echo "  ⚠️  Base file for $section not found — skipping"
    SKIPPED=$((SKIPPED + 1))
    return 0
  fi

  if [[ ! -f "$target_file" ]]; then
    echo "  ⚠️  Target ${TARGET_MAP[$section]} not found — skipping (run install.sh first)"
    SKIPPED=$((SKIPPED + 1))
    return 0
  fi

  local begin_marker="<!-- SPACESUIT:BEGIN ${section} -->"
  local end_marker="<!-- SPACESUIT:END -->"
  local base_content
  base_content="$(cat "$base_file")"

  # Check if markers exist in target
  if grep -qF "$begin_marker" "$target_file"; then
    # Markers exist — replace content between them
    local tmp_file
    tmp_file="$(mktemp)"

    # Replace content between markers using a temp file for the replacement content
    local replacement_file
    replacement_file="$(mktemp)"
    echo "$base_content" > "$replacement_file"

    awk -v begin="$begin_marker" -v end="$end_marker" -v repfile="$replacement_file" '
      BEGIN { replacing = 0; printed_replacement = 0 }
      $0 == begin {
        print $0
        replacing = 1
        next
      }
      $0 == end {
        if (!printed_replacement) {
          while ((getline line < repfile) > 0) print line
          close(repfile)
          printed_replacement = 1
        }
        replacing = 0
        print $0
        next
      }
      !replacing { print $0 }
    ' "$target_file" > "$tmp_file"

    rm -f "$replacement_file"

    # Check if anything changed
    if diff -q "$target_file" "$tmp_file" > /dev/null 2>&1; then
      echo "  ✅ ${TARGET_MAP[$section]} — already up to date"
      rm -f "$tmp_file"
      return 0
    fi

    # Show diff
    echo "  📝 ${TARGET_MAP[$section]} — changes detected:"
    diff --unified=3 "$target_file" "$tmp_file" | head -40 || true
    echo ""

    if ! $DRY_RUN; then
      cp "$tmp_file" "$target_file"
      echo "  ✅ ${TARGET_MAP[$section]} — updated"
    else
      echo "  🔍 ${TARGET_MAP[$section]} — would update (dry run)"
    fi

    rm -f "$tmp_file"
    CHANGED=$((CHANGED + 1))
  else
    # No markers — prepend base content with markers at top
    echo "  📌 ${TARGET_MAP[$section]} — no markers found, prepending framework section"

    local tmp_file
    tmp_file="$(mktemp)"

    {
      echo "$begin_marker"
      echo "$base_content"
      echo "$end_marker"
      echo ""
      cat "$target_file"
    } > "$tmp_file"

    if ! $DRY_RUN; then
      cp "$tmp_file" "$target_file"
      echo "  ✅ ${TARGET_MAP[$section]} — markers added + content prepended"
    else
      echo "  🔍 ${TARGET_MAP[$section]} — would prepend (dry run)"
    fi

    rm -f "$tmp_file"
    ADDED=$((ADDED + 1))
  fi
}

echo "📄 Upgrading workspace files..."
echo ""

for section in "${!SECTION_MAP[@]}"; do
  upgrade_section "$section"
done

# Update version tracker
if ! $DRY_RUN; then
  echo "$VERSION" > "$WORKSPACE/.spacesuit-version"
fi

echo ""
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
echo "  Changed: $CHANGED"
echo "  Added:   $ADDED"
echo "  Skipped: $SKIPPED"
echo "  Version: $VERSION"
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"

if $DRY_RUN; then
  echo ""
  echo "🔍 Dry run complete. No files were modified."
  echo "   Run without --dry-run to apply changes."
fi
