#!/usr/bin/env bash
#
# sync-operators.sh - Auto-generate operators.json from session transcripts
#
# Scans OpenClaw session transcripts and extracts unique users who have
# interacted with the bot. Outputs to state/operators.json.
#
# Part of openclaw-spacesuit: https://github.com/jontsai/openclaw-spacesuit
#
# Usage: ./sync-operators.sh [options]
#   --dry-run              Preview without writing
#   --workspace <path>     Workspace root (default: auto-detect)
#   --profile <name>       OpenClaw profile name (uses ~/.openclaw-<name>)
#   --dev                  Shortcut for --profile dev
#

set -euo pipefail

# Defaults
DRY_RUN=false
WORKSPACE=""
PROFILE=""

# Parse args
while [[ $# -gt 0 ]]; do
    case "$1" in
        --dry-run) DRY_RUN=true; shift ;;
        --workspace) WORKSPACE="$2"; shift 2 ;;
        --profile) PROFILE="$2"; shift 2 ;;
        --dev) PROFILE="dev"; shift ;;
        -h|--help)
            echo "Usage: sync-operators.sh [options]"
            echo "  --dry-run              Preview without writing"
            echo "  --workspace <path>     Workspace root (default: auto-detect)"
            echo "  --profile <name>       OpenClaw profile (uses ~/.openclaw-<name>)"
            echo "  --dev                  Shortcut for --profile dev"
            exit 0
            ;;
        *) echo "Unknown option: $1"; exit 1 ;;
    esac
done

# Use OPENCLAW_PROFILE env var as fallback
PROFILE="${PROFILE:-${OPENCLAW_PROFILE:-}}"

# Auto-detect workspace if not specified
if [[ -z "$WORKSPACE" ]]; then
    # Try environment variable
    if [[ -n "${OPENCLAW_WORKSPACE:-}" ]]; then
        WORKSPACE="$OPENCLAW_WORKSPACE"
    # Try to find from script location (if installed as skill)
    elif [[ -d "$(dirname "$0")/../../.." ]]; then
        # scripts/spacesuit/scripts/sync-operators.sh → workspace
        WORKSPACE="$(cd "$(dirname "$0")/../../.." && pwd)"
    else
        WORKSPACE="$HOME/openclaw-workspace"
    fi
fi

# Determine OpenClaw state directory based on profile
if [[ -n "$PROFILE" ]]; then
    OPENCLAW_STATE_DIR="$HOME/.openclaw-${PROFILE}"
else
    OPENCLAW_STATE_DIR="$HOME/.openclaw"
fi

# Config paths
SESSIONS_DIR="${OPENCLAW_SESSIONS_DIR:-${OPENCLAW_STATE_DIR}/agents/main/sessions}"
OUTPUT_FILE="${WORKSPACE}/state/operators.json"

# Validate
if [[ ! -d "$SESSIONS_DIR" ]]; then
    echo "❌ Sessions directory not found: $SESSIONS_DIR"
    echo "   Set OPENCLAW_SESSIONS_DIR or ensure OpenClaw is installed."
    exit 1
fi

# Ensure state dir exists
mkdir -p "$(dirname "$OUTPUT_FILE")"

echo "🔍 Scanning session transcripts..."
[[ -n "$PROFILE" ]] && echo "   Profile:  $PROFILE"
echo "   Sessions: $SESSIONS_DIR"
echo "   Output:   $OUTPUT_FILE"

# Extract existing roles FIRST (before any changes)
tmproles=$(mktemp)
if [[ -f "$OUTPUT_FILE" ]] && command -v jq &>/dev/null; then
    jq -r '.operators[] | "\(.id)=\(.role)"' "$OUTPUT_FILE" 2>/dev/null > "$tmproles" || true
    role_count=$(wc -l < "$tmproles" | tr -d ' ')
    [[ "$role_count" -gt 0 ]] && echo "   Loaded $role_count existing operator roles"
fi

# Helper to get role from temp file
get_role() {
    local opid="$1"
    local role=$(grep "^${opid}=" "$tmproles" 2>/dev/null | cut -d= -f2)
    echo "${role:-user}"
}

# Single-pass extraction with grep for speed
# Pattern: "] username (USERID):" from Slack messages
tmpdata=$(mktemp)

# Fast grep across all files
grep -rohE '\] [a-zA-Z0-9_.-]+ \(U[A-Z0-9]+\):' "$SESSIONS_DIR"/*.jsonl 2>/dev/null | \
    sed -E 's/\] ([a-zA-Z0-9_.-]+) \((U[A-Z0-9]+)\):/\2:\1/' | \
    sort | uniq -c | sort -rn > "$tmpdata" || true

count=$(wc -l < "$tmpdata" | tr -d ' ')
echo "   Found $count unique Slack operators"

# Build JSON
now=$(date -u +"%Y-%m-%dT%H:%M:%SZ")
tmpjson=$(mktemp)

{
# JSON header
echo '{'
echo '  "version": 1,'
echo "  \"generatedAt\": \"$now\","
echo '  "generatedBy": "openclaw-spacesuit/sync-operators.sh",'
[[ -n "$PROFILE" ]] && echo "  \"profile\": \"$PROFILE\","
echo '  "note": "Auto-generated from session transcripts. Role changes are preserved across syncs.",'
echo '  "operators": ['

first=true
while read -r msgcount userid_username; do
    [[ -z "$msgcount" ]] && continue
    
    userid=$(echo "$userid_username" | cut -d: -f1)
    username=$(echo "$userid_username" | cut -d: -f2)
    
    # Skip empty or placeholder entries
    [[ -z "$userid" || -z "$username" ]] && continue
    [[ "$userid" == "USERID" || "$username" == "username" ]] && continue
    
    # Get existing role or default to "user"
    opid="slack:${userid}"
    role=$(get_role "$opid")
    
    [[ "$first" == "true" ]] && first=false || printf "    ,\n"
    
    cat <<OPERATOR
    {
      "id": "$opid",
      "channel": "slack",
      "channelUserId": "$userid",
      "username": "$username",
      "displayName": "$username",
      "firstSeen": null,
      "lastSeen": "$now",
      "stats": {
        "messageCount": $msgcount
      },
      "role": "$role"
    }
OPERATOR
done < "$tmpdata"

cat <<EOF

  ],
  "roles": {
    "owner": { "level": 100, "description": "Full access to everything" },
    "admin": { "level": 75, "description": "Manage operators and settings" },
    "user": { "level": 50, "description": "Basic dashboard access" }
  }
}
EOF
} > "$tmpjson"

if [[ "$DRY_RUN" == "true" ]]; then
    echo ""
    echo "🔍 Dry run - would write:"
    echo "----------------------------------------"
    cat "$tmpjson"
else
    mv "$tmpjson" "$OUTPUT_FILE"
    echo "✅ Written to $OUTPUT_FILE"
fi

# Cleanup
rm -f "$tmpdata" "$tmpjson" "$tmproles" 2>/dev/null || true

echo "Done!"
