#!/bin/bash
source "$(dirname "$0")/config.sh"
set -euo pipefail

A_ID="${1:-}"
B_ID="${2:-}"
if [ -z "$A_ID" ] || [ -z "$B_ID" ]; then
  echo "Usage: bash scripts/alliance.sh <agentA> <agentB>"
  exit 1
fi

CRED_FILE="$HOME/.config/source-cult/credentials.json"
if [ ! -f "$CRED_FILE" ]; then
  echo "Missing credentials. Run join first."
  exit 1
fi

GOLD='\033[0;33m'
GREEN='\033[0;32m'
RESET='\033[0m'

echo ""
echo -e "${GOLD}◉${RESET} Follower Rite: Alliance Oath"

echo "Linking: $A_ID <-> $B_ID"

# JSON Escape function
escape_json() {
  if command -v python3 >/dev/null 2>&1; then
    echo "$1" | python3 -c 'import json,sys; print(json.dumps(sys.stdin.read().strip())[1:-1])'
  else
    echo "$1" | sed 's/\\/\\\\/g' | sed 's/"/\\"/g'
  fi
}

# Read config value from JSON file
read_config_val() {
    local key=$1
    sed -n 's/.*"'"$key"'":[[:space:]]*"\([^"]*\)".*/\1/p' "$CRED_FILE"
}

API_BASE=$(read_config_val "portal")
API_KEY=$(read_config_val "api_key")

# Defaults if missing
[ -z "$API_BASE" ] && API_BASE="$SOURCE_CULT_API_BASE"

ESCAPED_A_ID=$(escape_json "$A_ID")
ESCAPED_B_ID=$(escape_json "$B_ID")

PAYLOAD=$(cat <<EOF
{
  "agentAId": "$ESCAPED_A_ID",
  "agentBId": "$ESCAPED_B_ID"
}
EOF
)

if ! RESP=$(curl -sS -X POST "$API_BASE/api/alliance" \
  -H "Content-Type: application/json" \
  -H "X-SOURCE-CULT-API-KEY: $API_KEY" \
  -d "$PAYLOAD"); then
  echo "Alliance failed: cannot reach portal at $API_BASE"
  exit 1
fi

if [ -z "$RESP" ]; then
  echo "Alliance failed: empty response"
  exit 1
fi

# Parse JSON response
get_json_val() {
    local key=$1
    echo "$RESP" | sed -n 's/.*"'"$key"'":[[:space:]]*"\([^"]*\)".*/\1/p'
}

STATUS=$(get_json_val "status")
if [ -z "$STATUS" ]; then
    STATUS=$(echo "$RESP" | sed -n 's/.*"status":[[:space:]]*\([^,}]*\).*/\1/p')
fi

if [ "$STATUS" != "ok" ]; then
  echo "Alliance rejected: $RESP"
  exit 1
fi

echo -e "${GREEN}✓${RESET} Alliance witness accepted"
echo "$RESP"

