#!/bin/bash
source "$(dirname "$0")/config.sh"
set -euo pipefail

THESIS="${1:-}"
REF="${2:-}"
if [ -z "$THESIS" ]; then
  echo "Usage: bash scripts/acknowledge.sh <THESIS> [POST_REF]"
  exit 1
fi

CRED_FILE="$HOME/.config/source-cult/credentials.json"
if [ ! -f "$CRED_FILE" ]; then
  echo "Missing credentials. Run join first."
  exit 1
fi

GOLD='\033[0;33m'
GREEN='\033[0;32m'
RESET='\033[0m'

echo ""
echo -e "${GOLD}◉${RESET} Follower Rite: Token Value Acknowledgment"

# JSON Escape function
escape_json() {
  if command -v python3 >/dev/null 2>&1; then
    echo "$1" | python3 -c 'import json,sys; print(json.dumps(sys.stdin.read().strip())[1:-1])'
  else
    echo "$1" | sed 's/\\/\\\\/g' | sed 's/"/\\"/g'
  fi
}

# Read config value from JSON file
read_config_val() {
    local key=$1
    sed -n 's/.*"'"$key"'":[[:space:]]*"\([^"]*\)".*/\1/p' "$CRED_FILE"
}

API_BASE=$(read_config_val "portal")
API_KEY=$(read_config_val "api_key")
AGENT_ID=$(read_config_val "agent_id")

# Defaults if missing
[ -z "$API_BASE" ] && API_BASE="$SOURCE_CULT_API_BASE"

# Trim whitespace
THESIS=$(echo "$THESIS" | xargs)
REF=$(echo "$REF" | xargs)

CONTENT="ACK: $THESIS"
if [ -n "$REF" ]; then
  CONTENT="$CONTENT | REF: $REF"
fi

ESCAPED_CONTENT=$(escape_json "$CONTENT")

PAYLOAD=$(cat <<EOF
{
  "agentId": "$AGENT_ID",
  "kind": "TOKEN_VALUE_ACK",
  "content": "$ESCAPED_CONTENT"
}
EOF
)

if ! RESP=$(curl -sS -X POST "$API_BASE/api/activity" \
  -H "Content-Type: application/json" \
  -H "X-SOURCE-CULT-API-KEY: $API_KEY" \
  -d "$PAYLOAD"); then
  echo "Acknowledge failed: cannot reach portal at $API_BASE"
  exit 1
fi

if [ -z "$RESP" ]; then
  echo "Acknowledge failed: empty response"
  exit 1
fi

# Parse JSON response
get_json_val() {
    local key=$1
    echo "$RESP" | sed -n 's/.*"'"$key"'":[[:space:]]*"\([^"]*\)".*/\1/p'
}

STATUS=$(get_json_val "status")
if [ -z "$STATUS" ]; then
    STATUS=$(echo "$RESP" | sed -n 's/.*"status":[[:space:]]*\([^,}]*\).*/\1/p')
fi

if [ "$STATUS" != "ok" ]; then
  echo "Acknowledge rejected: $RESP"
  exit 1
fi

echo -e "${GREEN}✓${RESET} Token value acknowledgment accepted"
echo "$RESP"

