"""Model utilities for SOTA Tracker."""

from typing import Any, Optional


def normalize_model_id(name: str) -> str:
    """
    Normalize model name to ID format.

    Converts spaces, slashes, and dots to hyphens and lowercases.

    Examples:
        "GPT-4 Turbo" -> "gpt-4-turbo"
        "claude-3.5/sonnet" -> "claude-3-5-sonnet"
        "FLUX.1-dev" -> "flux-1-dev"

    Args:
        name: The model name to normalize

    Returns:
        Normalized model ID string
    """
    if not name:
        return ""
    return name.lower().replace(" ", "-").replace("/", "-").replace(".", "-")


def build_model_dict(
    name: str,
    rank: int,
    category: str,
    is_open_source: bool,
    metrics: Optional[dict] = None,
    **extra_fields
) -> dict:
    """
    Build standardized model dictionary.

    Creates a consistent model representation used across all fetchers.
    The model ID is automatically generated from the name.

    Args:
        name: Model name (e.g., "GPT-4 Turbo")
        rank: SOTA ranking position (1 = best)
        category: Model category (e.g., "llm_api", "image_gen")
        is_open_source: Whether the model is open-source
        metrics: Optional dictionary of model metrics (elo, benchmarks, etc.)
        **extra_fields: Additional fields to include in the model dict

    Returns:
        Standardized model dictionary with id, name, category, etc.
    """
    return {
        "id": normalize_model_id(name),
        "name": name,
        "category": category,
        "is_open_source": is_open_source,
        "sota_rank": rank,
        "metrics": metrics or {},
        **extra_fields
    }
