"""Hardware profile utilities for personalized model recommendations."""

import json
import re
import socket
from pathlib import Path
from typing import Any, Optional

# Path to hardware profiles config
DATA_DIR = Path(__file__).parent.parent / "data"
HARDWARE_PROFILES_PATH = DATA_DIR / "hardware_profiles.json"
VRAM_ESTIMATES_PATH = DATA_DIR / "vram_estimates.json"

# Default VRAM estimates for concurrent workloads (in GB)
# These are conservative estimates based on common model requirements as of Jan 2026
# Users can override these via vram_estimates.json
DEFAULT_VRAM_ESTIMATES = {
    "image_gen": 24,      # FLUX.2-dev, Qwen-Image (24GB for quality)
    "video_gen": 24,      # HunyuanVideo, Wan2.6 (24GB typical)
    "image_edit": 18,     # FLUX.1-Kontext (18GB)
    "stable_diffusion": 12,  # SD 3.5 Large (12GB)
    "comfyui": 16,        # ComfyUI with typical workflow
    "blender": 8,         # Blender GPU rendering
    "gaming": 12,         # Modern games at high settings
    "desktop": 2,         # Normal desktop compositing
    "none": 0
}

# Default profile template
DEFAULT_PROFILE = {
    "gpu": "Unknown",
    "vram_gb": 8,
    "ram_gb": 16,
    "cpu_threads": 4,
    "preferences": {
        "uncensored": False,
        "local_first": True,
        "cost_sensitive": True
    }
}


def load_hardware_profiles() -> dict:
    """Load all hardware profiles from config file."""
    if not HARDWARE_PROFILES_PATH.exists():
        return {"current": None, "profiles": {}}

    with open(HARDWARE_PROFILES_PATH, "r") as f:
        return json.load(f)


def save_hardware_profiles(profiles: dict) -> None:
    """Save hardware profiles to config file."""
    DATA_DIR.mkdir(parents=True, exist_ok=True)
    with open(HARDWARE_PROFILES_PATH, "w") as f:
        json.dump(profiles, f, indent=2)


def get_current_profile() -> Optional[dict]:
    """Get the currently active hardware profile."""
    profiles = load_hardware_profiles()
    current_name = profiles.get("current")

    if not current_name:
        return None

    return profiles.get("profiles", {}).get(current_name)


def get_profile_with_defaults() -> dict:
    """Get current profile, falling back to defaults if not configured."""
    profile = get_current_profile()
    if profile:
        # Merge with defaults to ensure all keys exist
        result = DEFAULT_PROFILE.copy()
        result.update(profile)
        # Handle preferences - ensure it's a valid dict before unpacking
        if "preferences" in profile and isinstance(profile["preferences"], dict):
            result["preferences"] = {**DEFAULT_PROFILE["preferences"], **profile["preferences"]}
        else:
            result["preferences"] = DEFAULT_PROFILE["preferences"].copy()
        return result
    return DEFAULT_PROFILE.copy()


def configure_profile(
    profile_name: Optional[str] = None,
    vram_gb: Optional[int] = None,
    gpu: Optional[str] = None,
    ram_gb: Optional[int] = None,
    cpu_threads: Optional[int] = None,
    uncensored_preference: Optional[bool] = None,
    local_first: Optional[bool] = None,
    cost_sensitive: Optional[bool] = None
) -> dict:
    """Configure or update a hardware profile."""
    profiles = load_hardware_profiles()

    # Use hostname if no profile name provided
    if not profile_name:
        profile_name = socket.gethostname()

    # Get existing profile or create new one
    existing = profiles.get("profiles", {}).get(profile_name, DEFAULT_PROFILE.copy())

    # Update fields if provided (with validation)
    if vram_gb is not None:
        if not isinstance(vram_gb, int) or vram_gb <= 0:
            raise ValueError(f"vram_gb must be a positive integer, got {vram_gb}")
        existing["vram_gb"] = vram_gb
    if gpu is not None:
        existing["gpu"] = str(gpu)
    if ram_gb is not None:
        if not isinstance(ram_gb, int) or ram_gb <= 0:
            raise ValueError(f"ram_gb must be a positive integer, got {ram_gb}")
        existing["ram_gb"] = ram_gb
    if cpu_threads is not None:
        if not isinstance(cpu_threads, int) or cpu_threads <= 0:
            raise ValueError(f"cpu_threads must be a positive integer, got {cpu_threads}")
        existing["cpu_threads"] = cpu_threads

    # Update preferences
    if "preferences" not in existing:
        existing["preferences"] = DEFAULT_PROFILE["preferences"].copy()
    if uncensored_preference is not None:
        existing["preferences"]["uncensored"] = uncensored_preference
    if local_first is not None:
        existing["preferences"]["local_first"] = local_first
    if cost_sensitive is not None:
        existing["preferences"]["cost_sensitive"] = cost_sensitive

    # Save profile
    if "profiles" not in profiles:
        profiles["profiles"] = {}
    profiles["profiles"][profile_name] = existing
    profiles["current"] = profile_name

    save_hardware_profiles(profiles)

    return {"profile_name": profile_name, **existing}


def parse_vram_string(vram_str: str) -> Optional[int]:
    """Parse VRAM string like '16GB' or '24 GB GGUF' to integer GB."""
    if not vram_str:
        return None

    # Extract number followed by GB
    match = re.search(r'(\d+)\s*(?:GB|gb)', str(vram_str))
    if match:
        return int(match.group(1))

    # Try plain integer
    try:
        return int(vram_str)
    except (ValueError, TypeError):
        return None


def vram_fits(model_vram: Any, available_vram_gb: int) -> bool:
    """Check if a model fits in available VRAM."""
    if model_vram is None:
        # If no VRAM info, assume it fits (be permissive)
        return True

    # Validate available_vram_gb parameter
    if available_vram_gb is None or not isinstance(available_vram_gb, (int, float)):
        return True  # Be permissive on invalid input

    required = parse_vram_string(str(model_vram))
    if required is None:
        return True

    return required <= int(available_vram_gb)


def get_available_vram(concurrent_usage_gb: int = 0) -> int:
    """Calculate available VRAM based on profile and concurrent usage."""
    profile = get_profile_with_defaults()
    total_vram = profile.get("vram_gb", 8)
    # Ensure concurrent_usage_gb is valid
    if concurrent_usage_gb is None or not isinstance(concurrent_usage_gb, (int, float)):
        concurrent_usage_gb = 0
    available = total_vram - int(concurrent_usage_gb)
    return max(0, available)


def load_vram_estimates() -> dict:
    """Load VRAM estimates from config file, falling back to defaults."""
    if not VRAM_ESTIMATES_PATH.exists():
        return DEFAULT_VRAM_ESTIMATES.copy()

    try:
        with open(VRAM_ESTIMATES_PATH, "r") as f:
            user_estimates = json.load(f)
            # Merge with defaults - user overrides take precedence
            result = DEFAULT_VRAM_ESTIMATES.copy()
            result.update(user_estimates)
            return result
    except (json.JSONDecodeError, IOError):
        return DEFAULT_VRAM_ESTIMATES.copy()


def save_vram_estimates(estimates: dict) -> None:
    """Save custom VRAM estimates to config file."""
    DATA_DIR.mkdir(parents=True, exist_ok=True)
    with open(VRAM_ESTIMATES_PATH, "w") as f:
        json.dump(estimates, f, indent=2)


def get_concurrent_vram_estimate(workload: str) -> int:
    """
    Estimate VRAM usage for common concurrent workloads.

    Loads from vram_estimates.json if available, otherwise uses defaults.
    Users can customize estimates by creating/editing data/vram_estimates.json.
    """
    if not workload:
        return 0

    estimates = load_vram_estimates()
    return estimates.get(workload.lower(), 0)
