#!/usr/bin/env python3
"""
SOTA Tracker Update Script for OpenCode Agents

This script updates your agents.md file with the latest SOTA data.
Designed to be run via cron, systemd timer, or manually.

Usage:
    python update_agents_md.py              # Update with full SOTA data
    python update_agents_md.py --minimal   # Lightweight version (top 5 models)
    python update_agents_md.py --categories llm_local,image_gen  # Specific categories
    python update_agents_md.py --refresh    # Force refresh from sources first

Integration with OpenCode:
    Add to your agent automation:
    - Daily cron job: @daily python ~/Apps/sota-tracker-mcp/update_agents_md.py
    - Or run before important tasks
"""

import argparse
import json
import sys
from datetime import datetime
from pathlib import Path
from typing import Optional

# Add repo to path for imports
REPO_DIR = Path(__file__).parent
sys.path.insert(0, str(REPO_DIR))

from utils.db import get_db_context
from server import _query_sota_impl, _get_forbidden_impl, _check_freshness_impl

# Default agents.md path
DEFAULT_AGENTS_MD = Path.home() / ".config" / "opencode" / "agents.md"


def get_current_sota(category: str = "llm_local", limit: int = 5) -> dict:
    """Get current SOTA models for a category."""
    result = _query_sota_impl(category, open_source_only=True)
    return {
        "category": category,
        "result": result,
        "limit": limit
    }


def update_agents_md(
    agents_path: Optional[Path] = None,
    categories: Optional[list] = None,
    limit: int = 5,
    minimal: bool = False,
    force_refresh: bool = False
) -> str:
    """
    Update agents.md with SOTA data.

    Args:
        agents_path: Path to agents.md (default: ~/.config/opencode/agents.md)
        categories: List of categories to include
        limit: Number of models per category
        minimal: Include only top model per category
        force_refresh: Refresh data from sources before updating

    Returns:
        Success/failure message
    """
    # Default categories
    if categories is None:
        categories = ["llm_local", "llm_api", "llm_coding", "image_gen", "video"]

    # Limit categories for minimal mode
    if minimal:
        categories = ["llm_local"]

    # Ensure agents.md exists
    if agents_path is None:
        agents_path = DEFAULT_AGENTS_MD
    else:
        agents_path = Path(agents_path).expanduser()  # Expand ~ and any user path

    agents_path.parent.mkdir(parents=True, exist_ok=True)

    # Check if force refresh is needed
    if force_refresh:
        print("Forcing data refresh from sources...")
        try:
            if REPO_DIR.joinpath("scrapers", "run_all.py").exists():
                import subprocess
                result = subprocess.run(
                    ["python", "scrapers/run_all.py"],
                    cwd=REPO_DIR,
                    capture_output=True,
                    text=True
                )
                if result.returncode != 0:
                    print(f"Warning: Refresh failed: {result.stderr}")
        except Exception as e:
            print(f"Warning: Could not refresh: {e}")

    # Build SOTA section
    timestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S")

    lines = [
        "# SOTA Models for Agents",
        "",
        f"<!-- Updated: {timestamp} -->",
        "",
        "Auto-generated from SOTA Tracker (https://github.com/romancircus/sota-tracker-mcp)",
        "",
        "## Summary",
        "",
    ]

    # Get forbidden models
    forbidden_result = _get_forbidden_impl()
    forbidden_count = forbidden_result.count("FORBIDDEN") if "FORBIDDEN" in forbidden_result else 0

    lines.extend([
        "- **Categories**: " + ", ".join(categories),
        f"- **Models per category**: {limit}",
        f"- **Forbidden models**: {forbidden_count}",
        "",
        "---",
        "",
        ""
    ])

    # Query each category
    for category in categories:
        result = _query_sota_impl(category, open_source_only=True)

        # Parse and limit results
        model_lines = []
        model_count = 0
        in_model = False

        for line in result.split('\n'):
            if line.startswith("**#"):
                # New model header
                in_model = True
                model_count += 1
                model_lines.append(line)
                if model_count >= limit:
                    break
            elif in_model:
                # Model details
                model_lines.append(line)

        # Format section
        lines.extend([
            f"## {category.upper()}",
            "",
            f"Top {min(limit, model_count)} open-source models:",
            ""
        ])

        lines.extend(model_lines)
        lines.extend([
            "",
            "",
            ""
        ])

    # Get forbidden section
    lines.extend([
        "## Forbidden (Do Not Use)",
        "",
        forbidden_result,
        "",
    ])

    # Get freshness check for top models
    lines.extend([
        "## Model Freshness",
        "",
        "Check if a model is current SOTA or outdated:",
        ""
    ])

    DB_PATH = REPO_DIR / "data" / "sota.db"
    with get_db_context(DB_PATH) as db:
        query = """
            SELECT name FROM models
            WHERE is_sota = 1 AND is_open_source = 1
            ORDER BY sota_rank_open ASC
            LIMIT 5
        """
        top_models = db.execute(query).fetchall()
        for model in top_models:
            name = model["name"]
            lines.append(f"- **{name}**: {_check_freshness_impl(name).split('\n')[0]}")

    lines.append("")

    # Write to agents.md
    content = "\n".join(lines)

    try:
        if agents_path.exists():
            # Backup existing
            backup = agents_path.with_suffix(f".md.backup")
            agents_path.replace(backup)

        agents_path.write_text(content)
        return f"✅ Updated {agents_path} with {len(categories)} categories"

    except PermissionError as e:
        return f"❌ Permission denied: {e}"
    except Exception as e:
        return f"❌ Error writing file: {e}"


def main():
    """Main entry point."""
    parser = argparse.ArgumentParser(
        description="Update agents.md with SOTA model data"
    )

    parser.add_argument(
        "--agents-path",
        type=str,
        help=f"Path to agents.md (default: {DEFAULT_AGENTS_MD})"
    )

    parser.add_argument(
        "--categories",
        nargs="*",
        help="Categories to include (default: llm_local, llm_api, etc.)"
    )

    parser.add_argument(
        "--limit",
        "-l",
        type=int,
        default=5,
        help="Number of models per category (default: 5)"
    )

    parser.add_argument(
        "--minimal",
        action="store_true",
        help="Include only top model per category"
    )

    parser.add_argument(
        "--refresh",
        action="store_true",
        help="Force refresh data from sources before updating"
    )

    args = parser.parse_args()

    result = update_agents_md(
        agents_path=args.agents_path,
        categories=args.categories,
        limit=args.limit,
        minimal=args.minimal,
        force_refresh=args.refresh
    )

    print(result)


if __name__ == "__main__":
    main()