"""Tests for SOTA Tracker MCP server tools."""

import sys
from pathlib import Path

# Add parent to path for imports
sys.path.insert(0, str(Path(__file__).parent.parent))

from server import (
    _query_sota_impl,
    _check_freshness_impl,
    _get_forbidden_impl,
    _compare_models_impl,
    _recent_releases_impl,
    _configure_hardware_impl,
    _query_sota_for_hardware_impl,
    _get_model_recommendation_impl,
)
from utils.classification import is_open_source
from utils.hardware import (
    parse_vram_string,
    vram_fits,
    get_concurrent_vram_estimate,
)


class TestQuerySota:
    """Tests for query_sota tool."""

    def test_valid_category_video(self):
        """Should return SOTA models for video category."""
        result = _query_sota_impl("video", True)
        assert "SOTA" in result
        assert "video" in result.lower()

    def test_valid_category_llm_api(self):
        """Should return SOTA models for llm_api category."""
        result = _query_sota_impl("llm_api", False)
        assert "SOTA" in result or "Models" in result

    def test_invalid_category(self):
        """Should return error for invalid category."""
        result = _query_sota_impl("invalid_category")
        assert "Invalid category" in result

    def test_open_source_only_note(self):
        """Should show note when filtering open-source."""
        result = _query_sota_impl("llm_local", True)
        assert "open-source" in result.lower()


class TestCheckFreshness:
    """Tests for check_freshness tool."""

    def test_forbidden_model(self):
        """Should identify forbidden models."""
        result = _check_freshness_impl("FLUX.1-dev")
        assert "OUTDATED" in result
        assert "FLUX.2" in result

    def test_unknown_model(self):
        """Should handle unknown models."""
        result = _check_freshness_impl("NonExistentModel12345")
        assert "UNKNOWN" in result


class TestGetForbidden:
    """Tests for get_forbidden tool."""

    def test_returns_list(self):
        """Should return forbidden models list."""
        result = _get_forbidden_impl()
        assert "Forbidden" in result
        assert "FLUX.1-dev" in result or "Redux" in result


class TestRecentReleases:
    """Tests for recent_releases tool."""

    def test_valid_days(self):
        """Should accept valid days parameter."""
        result = _recent_releases_impl(30, True)
        assert "Error" not in result

    def test_invalid_days_negative(self):
        """Should reject negative days."""
        result = _recent_releases_impl(-5, True)
        assert "Error" in result

    def test_invalid_days_too_large(self):
        """Should reject days > 365."""
        result = _recent_releases_impl(500, True)
        assert "Error" in result

    def test_invalid_days_zero(self):
        """Should reject zero days."""
        result = _recent_releases_impl(0, True)
        assert "Error" in result


class TestIsOpenSource:
    """Tests for is_open_source utility."""

    def test_closed_source_gpt(self):
        """Should identify GPT as closed-source."""
        assert is_open_source("GPT-4") is False
        assert is_open_source("gpt-4o") is False
        assert is_open_source("ChatGPT-latest") is False

    def test_closed_source_claude(self):
        """Should identify Claude as closed-source."""
        assert is_open_source("Claude Opus 4.5") is False
        assert is_open_source("claude-3-sonnet") is False

    def test_closed_source_gemini(self):
        """Should identify Gemini as closed-source."""
        assert is_open_source("Gemini 2.0 Pro") is False

    def test_open_source_llama(self):
        """Should identify Llama as open-source."""
        assert is_open_source("Llama 3.3-70B") is True
        assert is_open_source("llama-3.3-70b-instruct") is True

    def test_open_source_qwen(self):
        """Should identify Qwen as open-source."""
        assert is_open_source("Qwen 3 (Alibaba)") is True
        assert is_open_source("Qwen2.5-72B") is True

    def test_open_source_deepseek(self):
        """Should identify DeepSeek as open-source."""
        assert is_open_source("DeepSeek V3") is True


class TestCompareModels:
    """Tests for compare_models tool."""

    def test_both_unknown(self):
        """Should handle both unknown models."""
        result = _compare_models_impl("Unknown1", "Unknown2")
        assert "Neither" in result or "Not in database" in result


class TestVramParsing:
    """Tests for VRAM string parsing."""

    def test_parse_simple_gb(self):
        """Should parse simple GB values."""
        assert parse_vram_string("16GB") == 16
        assert parse_vram_string("32GB") == 32

    def test_parse_gb_with_space(self):
        """Should parse GB with space."""
        assert parse_vram_string("16 GB") == 16
        assert parse_vram_string("24 GB GGUF") == 24

    def test_parse_lowercase(self):
        """Should handle lowercase."""
        assert parse_vram_string("16gb") == 16

    def test_parse_plain_integer(self):
        """Should parse plain integers."""
        assert parse_vram_string("16") == 16

    def test_parse_none(self):
        """Should handle None."""
        assert parse_vram_string(None) is None
        assert parse_vram_string("") is None


class TestVramFits:
    """Tests for VRAM fitting logic."""

    def test_model_fits(self):
        """Should return True if model fits."""
        assert vram_fits("16GB", 32) is True
        assert vram_fits("8GB", 8) is True

    def test_model_does_not_fit(self):
        """Should return False if model doesn't fit."""
        assert vram_fits("32GB", 16) is False
        assert vram_fits("24GB", 8) is False

    def test_none_model_fits(self):
        """Should return True for None (be permissive)."""
        assert vram_fits(None, 8) is True


class TestConcurrentVramEstimate:
    """Tests for concurrent workload VRAM estimates."""

    def test_known_workloads(self):
        """Should return estimates for known workloads."""
        assert get_concurrent_vram_estimate("image_gen") == 24
        assert get_concurrent_vram_estimate("video_gen") == 24
        assert get_concurrent_vram_estimate("gaming") == 12
        assert get_concurrent_vram_estimate("none") == 0

    def test_unknown_workload(self):
        """Should return 0 for unknown workloads."""
        assert get_concurrent_vram_estimate("unknown_task") == 0

    def test_case_insensitive(self):
        """Should be case-insensitive."""
        assert get_concurrent_vram_estimate("IMAGE_GEN") == 24
        assert get_concurrent_vram_estimate("Image_Gen") == 24


class TestConfigureHardware:
    """Tests for configure_hardware tool."""

    def test_configure_with_vram(self):
        """Should configure hardware with VRAM."""
        result = _configure_hardware_impl(
            profile_name="test-profile",
            vram_gb=32,
            gpu="RTX 5090"
        )
        assert "32 GB" in result
        assert "RTX 5090" in result

    def test_configure_preferences(self):
        """Should configure preferences."""
        result = _configure_hardware_impl(
            profile_name="test-profile",
            uncensored_preference=True
        )
        assert "Uncensored" in result
        assert "Yes" in result


class TestQuerySotaForHardware:
    """Tests for query_sota_for_hardware tool."""

    def test_filters_by_vram(self):
        """Should filter models by available VRAM."""
        # First configure hardware with limited VRAM
        _configure_hardware_impl(profile_name="test", vram_gb=8)

        # Query with concurrent workload
        result = _query_sota_for_hardware_impl("llm_local", concurrent_vram_gb=0)
        assert "SOTA" in result or "Models" in result

    def test_invalid_category(self):
        """Should reject invalid category."""
        result = _query_sota_for_hardware_impl("invalid_cat")
        assert "Invalid category" in result

    def test_concurrent_workload_estimate(self):
        """Should use workload estimate."""
        _configure_hardware_impl(profile_name="test", vram_gb=32)
        result = _query_sota_for_hardware_impl(
            "llm_local",
            concurrent_workload="image_gen"
        )
        # Should show 8GB available (32 - 24 for image_gen)
        assert "8GB" in result or "Available" in result


class TestGetModelRecommendation:
    """Tests for get_model_recommendation tool."""

    def test_chat_task(self):
        """Should recommend model for chat task."""
        _configure_hardware_impl(profile_name="test", vram_gb=32)
        result = _get_model_recommendation_impl("chat")
        assert "Recommended" in result

    def test_invalid_task(self):
        """Should reject invalid task."""
        result = _get_model_recommendation_impl("invalid_task")
        assert "Unknown task" in result

    def test_concurrent_workload(self):
        """Should consider concurrent workload."""
        _configure_hardware_impl(profile_name="test", vram_gb=32)
        result = _get_model_recommendation_impl("chat", concurrent_workload="image_gen")
        # Should recommend smaller model due to concurrent usage
        assert "Recommended" in result


# Run with: python -m pytest tests/test_server.py -v
