"""Tests for REST API endpoints."""

import pytest
from fastapi.testclient import TestClient

from rest_api import app

client = TestClient(app)


class TestHealthEndpoint:
    """Tests for health check endpoint."""

    def test_health_returns_ok(self):
        """Health endpoint should return status ok."""
        response = client.get("/health")
        assert response.status_code == 200
        assert response.json()["status"] == "ok"


class TestRootEndpoint:
    """Tests for root endpoint."""

    def test_root_returns_api_info(self):
        """Root endpoint should return API info."""
        response = client.get("/")
        assert response.status_code == 200
        data = response.json()
        assert data["service"] == "SOTA Tracker API"
        assert "endpoints" in data


class TestModelsEndpoint:
    """Tests for /api/v1/models endpoint."""

    def test_valid_category(self):
        """Should return models for valid category."""
        response = client.get("/api/v1/models", params={"category": "llm_api"})
        assert response.status_code == 200
        data = response.json()
        assert data["category"] == "llm_api"
        assert "result" in data

    def test_invalid_category(self):
        """Should return 400 for invalid category."""
        response = client.get("/api/v1/models", params={"category": "invalid_cat"})
        assert response.status_code == 400

    def test_open_source_filter(self):
        """Should accept open_source_only parameter."""
        response = client.get(
            "/api/v1/models",
            params={"category": "llm_api", "open_source_only": "false"}
        )
        assert response.status_code == 200
        assert response.json()["open_source_only"] is False


class TestFreshnessEndpoint:
    """Tests for /api/v1/models/{model}/freshness endpoint."""

    def test_known_model(self):
        """Should return status for known forbidden model."""
        response = client.get("/api/v1/models/FLUX.1-dev/freshness")
        assert response.status_code == 200
        data = response.json()
        assert data["model"] == "FLUX.1-dev"
        assert "OUTDATED" in data["status"]

    def test_unknown_model(self):
        """Should return unknown status for unknown model."""
        response = client.get("/api/v1/models/nonexistent-model-xyz/freshness")
        assert response.status_code == 200
        assert "UNKNOWN" in response.json()["status"]


class TestForbiddenEndpoint:
    """Tests for /api/v1/forbidden endpoint."""

    def test_returns_forbidden_list(self):
        """Should return list of forbidden models."""
        response = client.get("/api/v1/forbidden")
        assert response.status_code == 200
        data = response.json()
        assert "result" in data
        assert "Forbidden Models" in data["result"]


class TestRecentEndpoint:
    """Tests for /api/v1/recent endpoint."""

    def test_default_days(self):
        """Should accept default days parameter."""
        response = client.get("/api/v1/recent")
        assert response.status_code == 200
        assert response.json()["days"] == 30

    def test_custom_days(self):
        """Should accept custom days parameter."""
        response = client.get("/api/v1/recent", params={"days": 7})
        assert response.status_code == 200
        assert response.json()["days"] == 7

    def test_invalid_days(self):
        """Should reject invalid days parameter."""
        response = client.get("/api/v1/recent", params={"days": 0})
        assert response.status_code == 422  # Validation error

        response = client.get("/api/v1/recent", params={"days": 500})
        assert response.status_code == 422


class TestCompareEndpoint:
    """Tests for /api/v1/compare endpoint."""

    def test_compare_two_models(self):
        """Should compare two models."""
        response = client.get(
            "/api/v1/compare",
            params={"model_a": "Claude Opus 4.5", "model_b": "GPT-4.5"}
        )
        assert response.status_code == 200
        data = response.json()
        assert data["model_a"] == "Claude Opus 4.5"
        assert data["model_b"] == "GPT-4.5"

    def test_missing_model_param(self):
        """Should require both model parameters."""
        response = client.get("/api/v1/compare", params={"model_a": "Test"})
        assert response.status_code == 422


class TestRecommendEndpoint:
    """Tests for /api/v1/recommend endpoint."""

    def test_valid_task(self):
        """Should return recommendation for valid task."""
        response = client.get("/api/v1/recommend", params={"task": "chat"})
        assert response.status_code == 200
        assert response.json()["task"] == "chat"

    def test_invalid_task(self):
        """Should handle invalid task gracefully."""
        response = client.get("/api/v1/recommend", params={"task": "invalid_task"})
        assert response.status_code == 200
        assert "Unknown task" in response.json()["result"]


# Run with: python -m pytest tests/test_rest_api.py -v
