#!/usr/bin/env python3
"""
SOTA Tracker MCP Server

Provides real-time SOTA (State of the Art) AI model information to Claude,
preventing suggestions of outdated models.

Features:
- Defaults to open-source/local models (but aware of closed-source)
- Filter by open-source only
- 11 categories including coding LLMs

Tools:
- query_sota(category, open_source_only): Get current SOTA models
- check_freshness(model): Check if a model is current or outdated
- get_forbidden(): List all forbidden/outdated models
- compare_models(a, b): Compare two models side-by-side
- recent_releases(days): Get models released in past N days

Usage:
    # Development
    mcp dev server.py

    # Install globally
    claude mcp add --transport stdio --scope user sota-tracker \
        -- python ~/Desktop/applications/sota-tracker-mcp/server.py
"""

import json
import sqlite3
from datetime import datetime, timedelta
from pathlib import Path
from typing import Optional

from fastmcp import FastMCP

from utils.db import get_db as _get_db, get_db_context as _get_db_context
from utils.hardware import (
    get_profile_with_defaults,
    configure_profile,
    vram_fits,
    parse_vram_string,
    get_available_vram,
    get_concurrent_vram_estimate
)

# Smart caching - refresh data if stale (not fetched today)
try:
    from fetchers.cache_manager import get_cache_manager
    CACHE_ENABLED = True
except ImportError:
    CACHE_ENABLED = False

# Project paths
PROJECT_DIR = Path(__file__).parent
DATA_DIR = PROJECT_DIR / "data"
DB_PATH = DATA_DIR / "sota.db"
FORBIDDEN_PATH = DATA_DIR / "forbidden.json"

# Initialize MCP server
mcp = FastMCP(
    name="sota-tracker",
    instructions="""
    SOTA Tracker provides current State-of-the-Art AI model information,
    personalized to the user's hardware capabilities and preferences.

    DEFAULT BEHAVIOR: Shows open-source/local models first (but is aware of closed-source).

    MANDATORY PROTOCOL:
    Before suggesting ANY AI model, you MUST:
    1. Call query_sota(category) to get current SOTA (defaults to open-source)
    2. Call check_freshness(model_name) to verify model is current
    3. NEVER suggest models returned by get_forbidden()

    HARDWARE-AWARE RECOMMENDATIONS:
    For local models, prefer using the hardware-aware tools:
    - query_sota_for_hardware(category, concurrent_workload): Filters by user's VRAM
    - get_model_recommendation(task, concurrent_workload): Single best model for a task
    - configure_hardware(): Set up user's GPU specs and preferences

    The user may have preferences like:
    - uncensored: Prefer uncensored model variants (JOSIEFIED, abliterated)
    - local_first: Prefer local models over API-based ones
    - cost_sensitive: Prefer free/cheap options

    Categories:
    - image_gen, image_edit, video, video2audio (foley/SFX from video)
    - llm_local (general local LLMs), llm_api (cloud APIs), llm_coding (code-focused)
    - tts, stt, music, 3d, embeddings

    Use open_source_only=False to see all models including closed-source.
    """
)


def get_db():
    """Get database connection with row factory. Caller must close."""
    return _get_db(DB_PATH)


def get_db_context():
    """Get database connection as context manager (auto-closes)."""
    return _get_db_context(DB_PATH)


def load_forbidden():
    """Load forbidden models from JSON file."""
    if FORBIDDEN_PATH.exists():
        with open(FORBIDDEN_PATH) as f:
            return json.load(f)
    return {"models": [], "last_updated": None}


# ============================================================================
# CORE LOGIC (testable functions)
# ============================================================================

def _query_sota_impl(category: str, open_source_only: bool = True) -> str:
    """Implementation of query_sota."""

    valid_categories = [
        "image_gen", "image_edit", "video", "video2audio", "llm_local", "llm_api", "llm_coding",
        "tts", "stt", "music", "3d", "embeddings"
    ]

    if category not in valid_categories:
        return f"Invalid category '{category}'. Valid: {', '.join(valid_categories)}"

    # Smart cache: refresh if data is stale (not fetched today)
    refresh_note = ""
    if CACHE_ENABLED:
        try:
            cache_mgr = get_cache_manager(DB_PATH)
            result = cache_mgr.refresh_if_stale(category)
            if result["refreshed"]:
                refresh_note = f"\n[Data refreshed from {result['source']}, {result['models_updated']} models updated]\n"
            elif result["error"]:
                refresh_note = f"\n[Cache refresh failed: {result['error']}. Using cached data.]\n"
        except Exception as e:
            refresh_note = f"\n[Cache error: {e}. Using cached data.]\n"

    # Build query based on open_source_only flag
    # Get current month/year for header
    current_date = datetime.now().strftime("%B %Y")

    with get_db_context() as db:
        if open_source_only:
            rows = db.execute("""
                SELECT name, release_date, sota_rank_open as rank, metrics, source_url, is_open_source
                FROM models
                WHERE category = ? AND is_sota = 1 AND is_open_source = 1
                ORDER BY sota_rank_open ASC
            """, (category,)).fetchall()
            header = f"## SOTA Open-Source Models for {category.upper()} ({current_date})\n"
        else:
            rows = db.execute("""
                SELECT name, release_date, sota_rank as rank, metrics, source_url, is_open_source
                FROM models
                WHERE category = ? AND is_sota = 1
                ORDER BY sota_rank ASC
            """, (category,)).fetchall()
            header = f"## ALL SOTA Models for {category.upper()} ({current_date})\n"

    if not rows:
        if open_source_only:
            return f"No open-source SOTA models found for '{category}'. Try open_source_only=False to see closed-source options."
        return f"No SOTA models found for category '{category}'"

    result = [header]

    if refresh_note:
        result.append(refresh_note)

    if open_source_only:
        result.append("(Showing open-source only. Use open_source_only=False for all models.)\n")

    for row in rows:
        try:
            metrics = json.loads(row["metrics"]) if row["metrics"] else {}
        except json.JSONDecodeError:
            metrics = {}
        notes = metrics.get("notes", "")
        why_sota = metrics.get("why_sota", "")
        strengths = metrics.get("strengths", [])
        use_cases = metrics.get("use_cases", [])

        # Add badge for open/closed
        badge = "" if row["is_open_source"] else " [CLOSED]"

        result.append(
            f"**#{row['rank']} {row['name']}**{badge} (Released: {row['release_date']})"
        )
        if notes:
            result.append(f"   {notes}")
        if why_sota:
            result.append(f"   Why SOTA: {why_sota}")
        if strengths:
            result.append(f"   Strengths: {', '.join(strengths)}")
        if use_cases:
            result.append(f"   Best for: {', '.join(use_cases)}")

    return "\n".join(result)


def _check_freshness_impl(model_name: str) -> str:
    """Implementation of check_freshness."""
    forbidden = load_forbidden()
    for model in forbidden.get("models", []):
        if model["name"].lower() == model_name.lower():
            return (
                f"OUTDATED: {model['name']} - {model['reason']}\n"
                f"USE INSTEAD: {model['replacement']}"
            )

    with get_db_context() as db:
        row = db.execute("""
            SELECT name, category, is_sota, is_open_source, sota_rank, sota_rank_open, release_date
            FROM models
            WHERE LOWER(name) = LOWER(?)
        """, (model_name,)).fetchone()

        if row:
            open_badge = "open-source" if row["is_open_source"] else "closed-source"
            if row["is_sota"]:
                rank = row["sota_rank_open"] if row["is_open_source"] else row["sota_rank"]
                rank_type = "open-source" if row["is_open_source"] else "overall"
                return f"CURRENT: {row['name']} is #{rank} {rank_type} SOTA for {row['category']} ({open_badge}, Released: {row['release_date']})"
            else:
                # Find current SOTA for same category
                sota = db.execute("""
                    SELECT name FROM models
                    WHERE category = ? AND is_sota = 1 AND sota_rank = 1
                """, (row["category"],)).fetchone()

                replacement = sota["name"] if sota else "Check query_sota() for current options"
                return f"OUTDATED: {row['name']} is no longer SOTA.\nUSE INSTEAD: {replacement}"

    return f"UNKNOWN: Model '{model_name}' not in database. Use query_sota() to find current options."


def _get_forbidden_impl() -> str:
    """Implementation of get_forbidden."""
    forbidden = load_forbidden()
    models = forbidden.get("models", [])

    if not models:
        return "No forbidden models defined."

    result = [f"## Forbidden Models (Do Not Suggest)\n"]
    result.append(f"Last updated: {forbidden.get('last_updated', 'Unknown')}\n")

    by_category = {}
    for m in models:
        cat = m.get("category", "other")
        if cat not in by_category:
            by_category[cat] = []
        by_category[cat].append(m)

    for category, items in sorted(by_category.items()):
        result.append(f"\n### {category.upper()}")
        for m in items:
            result.append(f"- **{m['name']}**: {m['reason']}")
            result.append(f"  → Use: {m['replacement']}")

    return "\n".join(result)


def _compare_models_impl(model_a: str, model_b: str) -> str:
    """Implementation of compare_models."""

    def get_model_info(db, name: str) -> Optional[dict]:
        row = db.execute("""
            SELECT name, category, release_date, is_sota, is_open_source, sota_rank, sota_rank_open, metrics
            FROM models WHERE LOWER(name) = LOWER(?)
        """, (name,)).fetchone()
        if row:
            return dict(row)

        forbidden = load_forbidden()
        for m in forbidden.get("models", []):
            if m["name"].lower() == name.lower():
                return {
                    "name": m["name"],
                    "category": m.get("category", "unknown"),
                    "is_sota": False,
                    "sota_rank": None,
                    "status": "FORBIDDEN",
                    "reason": m["reason"],
                    "replacement": m["replacement"]
                }
        return None

    with get_db_context() as db:
        info_a = get_model_info(db, model_a)
        info_b = get_model_info(db, model_b)

    if not info_a and not info_b:
        return f"Neither '{model_a}' nor '{model_b}' found in database."

    result = ["## Model Comparison\n"]

    def format_model(info: Optional[dict], name: str) -> str:
        if not info:
            return f"**{name}**: Not in database"

        lines = [f"**{info['name']}**"]
        lines.append(f"- Category: {info.get('category', 'Unknown')}")

        # Open source status
        if 'is_open_source' in info:
            lines.append(f"- License: {'Open-source' if info['is_open_source'] else 'Closed-source'}")

        if info.get("status") == "FORBIDDEN":
            lines.append(f"- Status: FORBIDDEN")
            lines.append(f"- Reason: {info.get('reason', 'Unknown')}")
            lines.append(f"- Use instead: {info.get('replacement', 'Unknown')}")
        elif info.get("is_sota"):
            rank = info.get('sota_rank_open') or info.get('sota_rank', '?')
            lines.append(f"- Status: SOTA #{rank}")
            lines.append(f"- Released: {info.get('release_date', 'Unknown')}")
        else:
            lines.append(f"- Status: Not SOTA")
            lines.append(f"- Released: {info.get('release_date', 'Unknown')}")

        return "\n".join(lines)

    result.append(format_model(info_a, model_a))
    result.append("\n---\n")
    result.append(format_model(info_b, model_b))

    result.append("\n### Recommendation")
    if info_a and info_a.get("status") == "FORBIDDEN":
        result.append(f"Use {info_b['name'] if info_b else model_b} (or its SOTA replacement)")
    elif info_b and info_b.get("status") == "FORBIDDEN":
        result.append(f"Use {info_a['name'] if info_a else model_a} (or its SOTA replacement)")
    elif info_a and info_a.get("is_sota") and (not info_b or not info_b.get("is_sota")):
        result.append(f"Use {info_a['name']} (currently SOTA)")
    elif info_b and info_b.get("is_sota") and (not info_a or not info_a.get("is_sota")):
        result.append(f"Use {info_b['name']} (currently SOTA)")
    elif info_a and info_b:
        # Both are SOTA, recommend based on open-source preference
        if info_a.get("is_open_source") and not info_b.get("is_open_source"):
            result.append(f"Use {info_a['name']} (open-source, preferred for local deployment)")
        elif info_b.get("is_open_source") and not info_a.get("is_open_source"):
            result.append(f"Use {info_b['name']} (open-source, preferred for local deployment)")
        else:
            result.append("Both models are comparable. Check query_sota() for latest rankings.")
    else:
        result.append("Both models are comparable. Check query_sota() for latest rankings.")

    return "\n".join(result)


def _recent_releases_impl(days: int = 30, open_source_only: bool = True) -> str:
    """Implementation of recent_releases."""
    # Input validation
    if not isinstance(days, int) or days < 1 or days > 365:
        return "Error: days must be an integer between 1 and 365"

    cutoff = (datetime.now() - timedelta(days=days)).strftime("%Y-%m-%d")

    with get_db_context() as db:
        if open_source_only:
            rows = db.execute("""
                SELECT name, category, release_date, is_sota, sota_rank_open as rank, is_open_source
                FROM models
                WHERE release_date >= ? AND is_open_source = 1
                ORDER BY release_date DESC
            """, (cutoff,)).fetchall()
        else:
            rows = db.execute("""
                SELECT name, category, release_date, is_sota, sota_rank as rank, is_open_source
                FROM models
                WHERE release_date >= ?
                ORDER BY release_date DESC
            """, (cutoff,)).fetchall()

    if not rows:
        return f"No models released in the past {days} days."

    header = f"## Models Released in Past {days} Days"
    if open_source_only:
        header += " (Open-Source Only)"
    result = [header + "\n"]

    for row in rows:
        sota_badge = f" [SOTA #{row['rank']}]" if row["is_sota"] and row["rank"] is not None else ""
        open_badge = "" if row["is_open_source"] else " [CLOSED]"
        result.append(f"- **{row['name']}** ({row['category']}) - {row['release_date']}{sota_badge}{open_badge}")

    return "\n".join(result)


# ============================================================================
# HARDWARE-AWARE IMPLEMENTATIONS
# ============================================================================

def _configure_hardware_impl(
    profile_name: Optional[str] = None,
    vram_gb: Optional[int] = None,
    gpu: Optional[str] = None,
    ram_gb: Optional[int] = None,
    cpu_threads: Optional[int] = None,
    uncensored_preference: Optional[bool] = None,
    local_first: Optional[bool] = None,
    cost_sensitive: Optional[bool] = None
) -> str:
    """Implementation of configure_hardware."""
    result = configure_profile(
        profile_name=profile_name,
        vram_gb=vram_gb,
        gpu=gpu,
        ram_gb=ram_gb,
        cpu_threads=cpu_threads,
        uncensored_preference=uncensored_preference,
        local_first=local_first,
        cost_sensitive=cost_sensitive
    )

    output = ["## Hardware Profile Configured\n"]
    output.append(f"**Profile:** {result['profile_name']}")
    output.append(f"**GPU:** {result.get('gpu', 'Unknown')}")
    output.append(f"**VRAM:** {result.get('vram_gb', 'Unknown')} GB")
    output.append(f"**RAM:** {result.get('ram_gb', 'Unknown')} GB")
    output.append(f"**CPU Threads:** {result.get('cpu_threads', 'Unknown')}")
    output.append("\n**Preferences:**")
    prefs = result.get('preferences', {})
    output.append(f"- Uncensored models: {'Yes' if prefs.get('uncensored') else 'No'}")
    output.append(f"- Local-first: {'Yes' if prefs.get('local_first') else 'No'}")
    output.append(f"- Cost-sensitive: {'Yes' if prefs.get('cost_sensitive') else 'No'}")

    return "\n".join(output)


def _query_sota_for_hardware_impl(
    category: str,
    concurrent_vram_gb: int = 0,
    concurrent_workload: Optional[str] = None
) -> str:
    """Implementation of query_sota_for_hardware."""
    valid_categories = [
        "image_gen", "image_edit", "video", "video2audio", "llm_local", "llm_api", "llm_coding",
        "tts", "stt", "music", "3d", "embeddings"
    ]

    if category not in valid_categories:
        return f"Invalid category '{category}'. Valid: {', '.join(valid_categories)}"

    # Get hardware profile
    profile = get_profile_with_defaults()
    total_vram = profile.get("vram_gb", 8)
    prefs = profile.get("preferences", {})

    # Calculate available VRAM
    if concurrent_workload:
        concurrent_vram_gb = get_concurrent_vram_estimate(concurrent_workload)
    available_vram = max(0, total_vram - concurrent_vram_gb)

    current_date = datetime.now().strftime("%B %Y")

    # Query SOTA models (open-source for local categories, all for API)
    with get_db_context() as db:
        if category in ["llm_api"]:
            rows = db.execute("""
                SELECT name, release_date, sota_rank as rank, metrics, source_url, is_open_source
                FROM models
                WHERE category = ? AND is_sota = 1
                ORDER BY sota_rank ASC
            """, (category,)).fetchall()
        else:
            rows = db.execute("""
                SELECT name, release_date, sota_rank_open as rank, metrics, source_url, is_open_source
                FROM models
                WHERE category = ? AND is_sota = 1 AND is_open_source = 1
                ORDER BY sota_rank_open ASC
            """, (category,)).fetchall()

    if not rows:
        return f"No SOTA models found for '{category}'"

    result = [f"## SOTA Models for {category.upper()} ({current_date})"]
    result.append(f"**Your Hardware:** {profile.get('gpu', 'Unknown')} ({total_vram}GB VRAM)")
    if concurrent_vram_gb > 0:
        workload_note = f" ({concurrent_workload})" if concurrent_workload else ""
        result.append(f"**Concurrent Usage:** {concurrent_vram_gb}GB{workload_note}")
    result.append(f"**Available VRAM:** {available_vram}GB")

    prefer_uncensored = prefs.get("uncensored", False)
    if prefer_uncensored:
        result.append("**Preference:** Uncensored models prioritized\n")
    else:
        result.append("")

    fits_models = []
    no_fit_models = []

    for row in rows:
        try:
            metrics = json.loads(row["metrics"]) if row["metrics"] else {}
        except json.JSONDecodeError:
            metrics = {}

        model_vram = metrics.get("vram_gb") or metrics.get("vram")
        model_fits = vram_fits(model_vram, available_vram)

        is_uncensored = metrics.get("is_uncensored", False)
        uncensored_variant = metrics.get("uncensored_variant")

        model_info = {
            "name": row["name"],
            "rank": row["rank"],
            "release_date": row["release_date"],
            "vram_gb": parse_vram_string(str(model_vram)) if model_vram else None,
            "fits": model_fits,
            "metrics": metrics,
            "is_uncensored": is_uncensored,
            "uncensored_variant": uncensored_variant
        }

        if model_fits:
            fits_models.append(model_info)
        else:
            no_fit_models.append(model_info)

    # If user prefers uncensored, sort to show uncensored models first
    if prefer_uncensored:
        fits_models.sort(key=lambda m: (not m["is_uncensored"], m["rank"]))

    if fits_models:
        result.append("### Models That Fit Your VRAM\n")
        for m in fits_models:
            vram_str = f" ({m['vram_gb']}GB)" if m['vram_gb'] else ""
            uncensored_badge = " [UNCENSORED]" if m["is_uncensored"] else ""
            result.append(f"**#{m['rank']} {m['name']}**{vram_str}{uncensored_badge}")

            # Show uncensored variant for censored models if user prefers uncensored
            if prefer_uncensored and not m["is_uncensored"] and m.get("uncensored_variant"):
                result.append(f"   → Uncensored alternative: {m['uncensored_variant']}")

            notes = m["metrics"].get("notes", "")
            if notes:
                result.append(f"   {notes}")

    if no_fit_models and category == "llm_local":
        result.append("\n### Models Too Large (need more free VRAM)\n")
        for m in no_fit_models[:3]:  # Show top 3 that don't fit
            vram_str = f" ({m['vram_gb']}GB)" if m['vram_gb'] else ""
            uncensored_badge = " [UNCENSORED]" if m.get("is_uncensored") else ""
            result.append(f"~~#{m['rank']} {m['name']}~~{vram_str}{uncensored_badge} - needs {m['vram_gb'] - available_vram}GB more")

    return "\n".join(result)


def _get_model_recommendation_impl(
    task: str,
    concurrent_workload: Optional[str] = None
) -> str:
    """Implementation of get_model_recommendation."""
    # Map tasks to categories and priorities
    task_mapping = {
        "chat": {"category": "llm_local", "priority": ["fast", "quality"]},
        "daily_chat": {"category": "llm_local", "priority": ["fast", "quality"]},
        "code": {"category": "llm_coding", "priority": ["quality"]},
        "coding": {"category": "llm_coding", "priority": ["quality"]},
        "reason": {"category": "llm_local", "priority": ["reasoning"]},
        "reasoning": {"category": "llm_local", "priority": ["reasoning"]},
        "creative": {"category": "llm_local", "priority": ["creative", "quality"]},
        "creative_writing": {"category": "llm_local", "priority": ["creative", "quality"]},
        "max_quality": {"category": "llm_local", "priority": ["quality"]},
        "image": {"category": "image_gen", "priority": ["quality"]},
        "image_gen": {"category": "image_gen", "priority": ["quality"]},
        "video": {"category": "video", "priority": ["quality"]},
    }

    task_lower = task.lower().replace(" ", "_")
    task_info = task_mapping.get(task_lower)

    if not task_info:
        valid_tasks = ", ".join(task_mapping.keys())
        return f"Unknown task '{task}'. Valid tasks: {valid_tasks}"

    category = task_info["category"]

    # Get hardware profile
    profile = get_profile_with_defaults()
    total_vram = profile.get("vram_gb", 8)
    prefs = profile.get("preferences", {})

    # Calculate available VRAM
    concurrent_vram = get_concurrent_vram_estimate(concurrent_workload) if concurrent_workload else 0
    available_vram = max(0, total_vram - concurrent_vram)

    # Query models
    with get_db_context() as db:
        rows = db.execute("""
            SELECT name, metrics, source_url
            FROM models
            WHERE category = ? AND is_sota = 1 AND is_open_source = 1
            ORDER BY sota_rank_open ASC
        """, (category,)).fetchall()

    if not rows:
        return f"No models found for task '{task}'"

    # Find best fitting model
    best_model = None
    for row in rows:
        try:
            metrics = json.loads(row["metrics"]) if row["metrics"] else {}
        except json.JSONDecodeError:
            metrics = {}

        model_vram = metrics.get("vram_gb") or metrics.get("vram")
        if vram_fits(model_vram, available_vram):
            best_model = {
                "name": row["name"],
                "metrics": metrics,
                "source_url": row["source_url"]
            }
            break

    if not best_model:
        return f"No models fit your available VRAM ({available_vram}GB) for task '{task}'. Try closing other GPU workloads."

    result = [f"## Recommended Model for {task.title()}\n"]
    result.append(f"**{best_model['name']}**")

    metrics = best_model["metrics"]
    if metrics.get("vram_gb"):
        result.append(f"VRAM: {metrics['vram_gb']}GB")
    if metrics.get("notes"):
        result.append(f"{metrics['notes']}")

    # Add uncensored variant if user prefers
    uncensored_variant = metrics.get("uncensored_variant")
    if prefs.get("uncensored") and uncensored_variant:
        result.append(f"\n**Uncensored Alternative:** {uncensored_variant}")

    # Add download/run commands for local models
    if category == "llm_local":
        result.append("\n### Quick Start")
        model_name_lower = best_model["name"].lower().replace(" ", "-")
        if "qwen" in model_name_lower:
            result.append(f"```bash")
            result.append(f"# Download from HuggingFace")
            result.append(f"huggingface-cli download <model-repo> --include '*Q4_K_M*'")
            result.append(f"")
            result.append(f"# Run with llama.cpp")
            result.append(f"llama-cli -m ~/models/gguf/<model>.gguf -cnv -ngl 99")
            result.append(f"```")

    return "\n".join(result)


# ============================================================================
# MCP TOOLS (wrappers for MCP decorator)
# ============================================================================

@mcp.tool()
def query_sota(category: str, open_source_only: bool = True) -> str:
    """
    Get current SOTA (State of the Art) models for a category.

    DEFAULTS TO OPEN-SOURCE ONLY. Set open_source_only=False to see closed-source models too.

    Args:
        category: One of: image_gen, image_edit, video, video2audio, llm_local, llm_api, llm_coding,
                  tts, stt, music, 3d, embeddings
        open_source_only: If True (default), only show open-source models.
                          If False, show all models including closed-source.

    Returns:
        Ranked list of current SOTA models with release dates and notes.
    """
    return _query_sota_impl(category, open_source_only)


@mcp.tool()
def check_freshness(model_name: str) -> str:
    """
    Check if a model is current SOTA or outdated.

    Args:
        model_name: Name of the model to check (e.g., "FLUX.1-dev", "Llama 2")

    Returns:
        Status: "CURRENT" with rank, or "OUTDATED" with replacement suggestion.
    """
    return _check_freshness_impl(model_name)


@mcp.tool()
def get_forbidden() -> str:
    """
    Get list of all forbidden/outdated models that should never be suggested.

    Returns:
        List of outdated models with reasons and replacements.
    """
    return _get_forbidden_impl()


@mcp.tool()
def compare_models(model_a: str, model_b: str) -> str:
    """
    Compare two models side-by-side.

    Args:
        model_a: First model name
        model_b: Second model name

    Returns:
        Side-by-side comparison of both models including open-source status.
    """
    return _compare_models_impl(model_a, model_b)


@mcp.tool()
def recent_releases(days: int = 30, open_source_only: bool = True) -> str:
    """
    Get models released in the past N days.

    Args:
        days: Number of days to look back (default: 30)
        open_source_only: If True (default), only show open-source models.

    Returns:
        List of recently released models.
    """
    return _recent_releases_impl(days, open_source_only)


@mcp.tool()
def refresh_data(category: str = None) -> str:
    """
    Force refresh SOTA data from external sources.

    By default, data auto-refreshes once per day on first query.
    Use this to force an immediate refresh.

    Args:
        category: Category to refresh, or None for all categories.

    Returns:
        Refresh status for each category.
    """
    if not CACHE_ENABLED:
        return "Cache system not available. Using static data."

    try:
        cache_mgr = get_cache_manager(DB_PATH)
        results = []

        if category:
            # Refresh single category
            result = cache_mgr.force_refresh(category)
            status = "OK" if result["refreshed"] else f"FAILED: {result['error']}"
            return f"Refresh {category}: {status} (source: {result['source']}, updated: {result['models_updated']} models)"
        else:
            # Refresh all categories
            categories = [
                "llm_local", "llm_api", "llm_coding",
                "image_gen", "video", "tts"
            ]
            for cat in categories:
                result = cache_mgr.force_refresh(cat)
                status = "OK" if result["refreshed"] else f"FAILED"
                results.append(f"  {cat}: {status} ({result['models_updated']} models)")

            return "## Refresh Results\n" + "\n".join(results)
    except Exception as e:
        return f"Refresh error: {e}"


@mcp.tool()
def cache_status() -> str:
    """
    Check the cache status for all categories.

    Shows when each category was last refreshed and from what source.

    Returns:
        Cache freshness status for all categories.
    """
    if not CACHE_ENABLED:
        return "Cache system not available. Using static data."

    try:
        cache_mgr = get_cache_manager(DB_PATH)
        statuses = cache_mgr.get_cache_status()

        if not statuses:
            return "No cache data yet. Categories will be refreshed on first query."

        result = ["## Cache Status\n"]
        for s in statuses:
            fresh = "FRESH" if cache_mgr.is_cache_fresh(s["category"]) else "STALE"
            result.append(
                f"- **{s['category']}**: {fresh} (last: {s['last_fetched']}, source: {s['fetch_source']})"
            )

        return "\n".join(result)
    except Exception as e:
        return f"Cache status error: {e}"


@mcp.tool()
def configure_hardware(
    profile_name: str = None,
    vram_gb: int = None,
    gpu: str = None,
    ram_gb: int = None,
    cpu_threads: int = None,
    uncensored_preference: bool = None,
    local_first: bool = None,
    cost_sensitive: bool = None
) -> str:
    """
    Configure or update your hardware profile for personalized model recommendations.

    Args:
        profile_name: Name for this profile (default: auto-detect hostname)
        vram_gb: GPU VRAM in gigabytes (e.g., 32 for RTX 5090)
        gpu: GPU model name (e.g., "RTX 5090", "RTX 4090")
        ram_gb: System RAM in gigabytes
        cpu_threads: Number of CPU threads available
        uncensored_preference: Prefer uncensored model variants when available
        local_first: Prefer local models over API-based ones
        cost_sensitive: Prefer free/cheap options over paid APIs

    Returns:
        Current profile configuration after updates.
    """
    return _configure_hardware_impl(
        profile_name=profile_name,
        vram_gb=vram_gb,
        gpu=gpu,
        ram_gb=ram_gb,
        cpu_threads=cpu_threads,
        uncensored_preference=uncensored_preference,
        local_first=local_first,
        cost_sensitive=cost_sensitive
    )


@mcp.tool()
def query_sota_for_hardware(
    category: str,
    concurrent_vram_gb: int = 0,
    concurrent_workload: str = None
) -> str:
    """
    Get SOTA models filtered by your hardware capabilities.

    Automatically considers your GPU VRAM and shows which models fit.
    Respects your preferences (uncensored, local-first, etc.).

    Args:
        category: Model category (llm_local, image_gen, video, etc.)
        concurrent_vram_gb: VRAM already in use by other workloads (in GB)
        concurrent_workload: Named workload (image_gen, video_gen, comfyui, gaming, etc.)
                            If provided, overrides concurrent_vram_gb with estimate.

    Returns:
        Models ranked by quality, showing which fit your available VRAM.
        Includes uncensored variants if you have that preference enabled.

    Example:
        # When running FLUX.2-dev for image gen (~24GB VRAM):
        query_sota_for_hardware("llm_local", concurrent_workload="image_gen")
        # Returns smaller models like Qwen3-8B that fit in remaining 8GB
    """
    return _query_sota_for_hardware_impl(
        category=category,
        concurrent_vram_gb=concurrent_vram_gb,
        concurrent_workload=concurrent_workload
    )


@mcp.tool()
def get_model_recommendation(
    task: str,
    concurrent_workload: str = None
) -> str:
    """
    Get a single best model recommendation for a specific task.

    Analyzes your hardware profile and current GPU usage to recommend
    the best model that will actually fit and run well.

    Args:
        task: What you want to do. Options:
              - chat, daily_chat: General conversation
              - code, coding: Code generation and assistance
              - reason, reasoning: Complex reasoning tasks
              - creative, creative_writing: Story writing, creative content
              - max_quality: Best possible quality (when GPU is free)
              - image, image_gen: Image generation
              - video: Video generation
        concurrent_workload: What else is running on GPU (image_gen, video_gen,
                            comfyui, gaming, none). Affects available VRAM.

    Returns:
        Single best model with VRAM requirements and quick-start commands.
        Includes uncensored alternative if you prefer those.

    Example:
        # Best chat model while running image generation:
        get_model_recommendation("chat", concurrent_workload="image_gen")
    """
    return _get_model_recommendation_impl(
        task=task,
        concurrent_workload=concurrent_workload
    )


# ============================================================================
# BEST IN CLASS TOOL
# ============================================================================

@mcp.tool()
def get_best_in_class(category: str = "llm_local") -> str:
    """
    Get the #1 model in each subcategory for easier decision-making.

    Instead of one ranked list mixing all model types, this shows
    the BEST model for each use case.

    For llm_local, shows:
    - Best overall (general use)
    - Best small (fits alongside image/video gen)
    - Best reasoning (math, logic, chain-of-thought)
    - Best uncensored (no restrictions)
    - Best uncensored small (no restrictions + concurrent use)
    - Best uncensored reasoning

    Args:
        category: Model category (default: llm_local)

    Returns:
        #1 model for each subcategory with VRAM requirements.
    """
    if category != "llm_local":
        return f"Best-in-class subcategories only implemented for llm_local. Use query_sota('{category}') instead."

    profile = get_profile_with_defaults()
    total_vram = profile.get("vram_gb", 32)
    prefs = profile.get("preferences", {})
    uncensored_pref = prefs.get("uncensored", False)

    with get_db_context() as db:
        rows = db.execute("""
            SELECT name, sota_rank_open as rank, metrics
            FROM models
            WHERE category = 'llm_local' AND is_sota = 1 AND is_open_source = 1
            ORDER BY sota_rank_open ASC
        """).fetchall()

    if not rows:
        return "No models found."

    # Categorize models
    best = {
        "overall": None,
        "small": None,  # <=8GB VRAM
        "reasoning": None,
        "uncensored": None,
        "uncensored_small": None,
        "uncensored_reasoning": None,
    }

    for row in rows:
        try:
            metrics = json.loads(row["metrics"]) if row["metrics"] else {}
        except:
            metrics = {}

        vram = metrics.get("vram_gb", 20)
        is_uncensored = metrics.get("is_uncensored", False)
        is_reasoning = "reasoning" in row["name"].lower() or "qwq" in row["name"].lower() or "r1" in row["name"].lower()
        is_small = vram <= 8

        # Best overall
        if not best["overall"]:
            best["overall"] = (row["name"], vram, row["rank"])

        # Best small
        if is_small and not best["small"]:
            best["small"] = (row["name"], vram, row["rank"])

        # Best reasoning
        if is_reasoning and not best["reasoning"]:
            best["reasoning"] = (row["name"], vram, row["rank"])

        # Best uncensored
        if is_uncensored and not best["uncensored"]:
            best["uncensored"] = (row["name"], vram, row["rank"])

        # Best uncensored small
        if is_uncensored and is_small and not best["uncensored_small"]:
            best["uncensored_small"] = (row["name"], vram, row["rank"])

        # Best uncensored reasoning
        if is_uncensored and is_reasoning and not best["uncensored_reasoning"]:
            best["uncensored_reasoning"] = (row["name"], vram, row["rank"])

    result = [f"## Best-in-Class for LLM_LOCAL"]
    result.append(f"**Your VRAM:** {total_vram}GB")
    if uncensored_pref:
        result.append("**Preference:** Uncensored models\n")
    result.append("")

    labels = {
        "overall": "Best Overall",
        "small": "Best Small (concurrent use)",
        "reasoning": "Best Reasoning",
        "uncensored": "Best Uncensored",
        "uncensored_small": "Best Uncensored + Small",
        "uncensored_reasoning": "Best Uncensored + Reasoning",
    }

    for key, label in labels.items():
        if best[key]:
            name, vram, rank = best[key]
            fits = "✅" if vram <= total_vram else "❌"
            result.append(f"**{label}:** {name} ({vram}GB) {fits}")
            result.append(f"   Overall rank: #{rank} (but #1 in this subcategory)")
        else:
            result.append(f"**{label}:** None found")

    result.append("")
    result.append("*Subcategory #1 matters more than overall rank for your use case.*")

    return "\n".join(result)


# ============================================================================
# MCP RESOURCES
# ============================================================================

@mcp.resource("sota://categories")
def get_categories() -> str:
    """List all available model categories."""
    categories = {
        "image_gen": "Text/image to image generation (Z-Image-Turbo, FLUX.2, Qwen-Image)",
        "image_edit": "Image editing and inpainting (Qwen-Image-Edit, Kontext)",
        "video": "Video generation (LTX-2, Wan 2.2, HunyuanVideo)",
        "llm_local": "Local LLMs for llama.cpp (Qwen3, Llama3.3, DeepSeek-V3)",
        "llm_api": "API-based LLMs (Claude, GPT, Gemini, Grok)",
        "llm_coding": "Coding-focused LLMs (Qwen3-Coder, DeepSeek-V3, Claude)",
        "tts": "Text-to-speech (ChatterboxTTS, F5-TTS)",
        "stt": "Speech-to-text (Whisper)",
        "music": "Music generation (Suno, Udio)",
        "3d": "3D model generation (Meshy, Tripo)",
        "embeddings": "Vector embeddings (BGE, E5)"
    }

    result = ["# Model Categories\n"]
    result.append("Default: open-source models. Use open_source_only=False for all.\n")
    for cat_id, desc in categories.items():
        result.append(f"- **{cat_id}**: {desc}")

    return "\n".join(result)


@mcp.resource("sota://forbidden")
def get_forbidden_resource() -> str:
    """Get the complete forbidden models list."""
    return _get_forbidden_impl()


# ============================================================================
# MAIN
# ============================================================================

def main():
    """Run the MCP server."""
    import sys

    if "--http" in sys.argv:
        # Run FastMCP with HTTP transport
        host = "0.0.0.0"
        port = 8000
        for i, arg in enumerate(sys.argv):
            if arg == "--host" and i + 1 < len(sys.argv):
                host = sys.argv[i + 1]
            if arg == "--port" and i + 1 < len(sys.argv):
                try:
                    port = int(sys.argv[i + 1])
                    if not (1 <= port <= 65535):
                        print(f"Error: Port must be between 1 and 65535, got {port}")
                        sys.exit(1)
                except ValueError:
                    print(f"Error: Invalid port '{sys.argv[i + 1]}' - must be an integer")
                    sys.exit(1)
        print(f"Starting HTTP server on {host}:{port}")
        mcp.run(transport="http", host=host, port=port)
    else:
        mcp.run()  # Default: stdio


if __name__ == "__main__":
    main()
