#!/usr/bin/env python3
"""
Update models with HuggingFace download URLs.

Run after migrations to add download URLs to existing models.
"""

import sqlite3
from pathlib import Path

PROJECT_DIR = Path(__file__).parent.parent
DB_PATH = PROJECT_DIR / "data" / "sota.db"

# Model ID -> Download URL mapping
DOWNLOAD_URLS = {
    # Image Generation
    "z-image-turbo": "https://huggingface.co/stabilityai/z-image-turbo",
    "flux2-dev": "https://huggingface.co/black-forest-labs/FLUX.2-dev",
    "qwen-image-2512": "https://huggingface.co/Qwen/Qwen-Image-2512",

    # Image Editing
    "qwen-image-edit-2511": "https://huggingface.co/Qwen/Qwen-Image-Edit-2511",
    "flux1-kontext": "https://huggingface.co/black-forest-labs/FLUX.1-Kontext-dev",
    "instructpix2pix": "https://huggingface.co/timbrooks/instruct-pix2pix",

    # Video Generation
    "ltx-2": "https://huggingface.co/Lightricks/LTX-2",
    "wan-2.2": "https://huggingface.co/alibaba-pai/Wan-2.2",
    "hunyuan-video-1.5": "https://huggingface.co/tencent/HunyuanVideo",
    "wan-2.1-i2v": "https://huggingface.co/alibaba-pai/Wan-2.1-I2V",

    # Local LLMs
    "qwen3-32b": "https://huggingface.co/Qwen/Qwen3-32B-GGUF",
    "qwen3-30b-a3b": "https://huggingface.co/Qwen/Qwen3-30B-A3B-GGUF",
    "qwen3-8b": "https://huggingface.co/Qwen/Qwen3-8B-GGUF",
    "deepseek-r1-32b": "https://huggingface.co/deepseek-ai/DeepSeek-R1-Distill-Qwen-32B-GGUF",
    "qwq-32b": "https://huggingface.co/Qwen/QwQ-32B-Preview-GGUF",
    "llama3.3-70b": "https://huggingface.co/meta-llama/Llama-3.3-70B-Instruct-GGUF",
    "deepseek-v3": "https://huggingface.co/deepseek-ai/DeepSeek-V3-GGUF",

    # Uncensored variants
    "josiefied-qwen3-8b": "https://huggingface.co/mradermacher/Josiefied-Qwen3-8B-abliterated-v1-GGUF",
    "qwen3-32b-uncensored": "https://huggingface.co/mradermacher/Qwen3-32B-Uncensored-GGUF",
    "qwq-32b-uncensored": "https://huggingface.co/mradermacher/QwQ-32B-Uncensored-GGUF",

    # Coding LLMs
    "qwen3-coder": "https://huggingface.co/Qwen/Qwen3-Coder-480B-A35B-Instruct-GGUF",
    "deepseek-v3-coder": "https://huggingface.co/deepseek-ai/DeepSeek-V3-GGUF",
    "starcoder2-15b": "https://huggingface.co/bigcode/starcoder2-15b-GGUF",

    # TTS
    "chatterbox-tts": "https://huggingface.co/ResembleAI/chatterbox",
    "f5-tts": "https://huggingface.co/SWivid/F5-TTS",
    "bark": "https://huggingface.co/suno/bark",
    "coqui-xtts-v2": "https://huggingface.co/coqui/XTTS-v2",

    # STT
    "whisper-large-v3": "https://huggingface.co/openai/whisper-large-v3",
    "canary-1b": "https://huggingface.co/nvidia/canary-1b",
    "faster-whisper-large-v3": "https://huggingface.co/Systran/faster-whisper-large-v3",

    # Music
    "musicgen-large": "https://huggingface.co/facebook/musicgen-large",
    "stable-audio-2": "https://huggingface.co/stabilityai/stable-audio-open-1.0",
    "audiocraft-musicgen-melody": "https://huggingface.co/facebook/musicgen-melody",

    # 3D
    "instantmesh": "https://huggingface.co/TencentARC/InstantMesh",
    "triposr": "https://huggingface.co/stabilityai/TripoSR",
    "openlrm": "https://huggingface.co/zxhezexin/openlrm-mix-base-1.1",

    # Embeddings
    "bge-m3": "https://huggingface.co/BAAI/bge-m3",
    "gte-qwen2-7b": "https://huggingface.co/Alibaba-NLP/gte-Qwen2-7B-instruct",
    "e5-mistral-7b": "https://huggingface.co/intfloat/e5-mistral-7b-instruct",
    "jina-embeddings-v3": "https://huggingface.co/jinaai/jina-embeddings-v3",
}


def update_urls():
    """Update download_url column for known models."""
    if not DB_PATH.exists():
        print(f"Database not found at {DB_PATH}")
        return False

    db = sqlite3.connect(str(DB_PATH))

    # Check if column exists
    cursor = db.execute("PRAGMA table_info(models)")
    columns = [row[1] for row in cursor.fetchall()]

    if "download_url" not in columns:
        print("download_url column not found. Run migrations first:")
        print("  python migrations/migrate.py")
        return False

    updated = 0
    for model_id, url in DOWNLOAD_URLS.items():
        cursor = db.execute(
            "UPDATE models SET download_url = ? WHERE id = ?",
            (url, model_id)
        )
        if cursor.rowcount > 0:
            updated += 1
            print(f"  Updated: {model_id}")

    db.commit()
    db.close()

    print(f"\nUpdated {updated} models with download URLs.")
    return True


if __name__ == "__main__":
    print("Updating models with HuggingFace download URLs...")
    update_urls()
