#!/usr/bin/env python3
"""
REST API for SOTA Tracker.

Provides traditional REST endpoints wrapping the MCP server's core functionality.
Use this for non-MCP clients that need simple HTTP access.

Usage:
    uvicorn rest_api:app --host 0.0.0.0 --port 8000

Endpoints:
    GET /api/v1/models?category=llm_api&open_source_only=true
    GET /api/v1/models/{model_name}/freshness
    GET /api/v1/forbidden
    GET /api/v1/recent?days=30&open_source_only=true
    GET /api/v1/compare?model_a=X&model_b=Y
    GET /health

Rate Limits:
    - 60 requests per minute per IP for most endpoints
    - 10 requests per minute for expensive operations (compare, hardware)
"""

from fastapi import FastAPI, Query, HTTPException, status, Request
from fastapi.middleware.cors import CORSMiddleware
from fastapi.responses import JSONResponse
from typing import Optional
from slowapi import Limiter, _rate_limit_exceeded_handler
from slowapi.util import get_remote_address
from slowapi.errors import RateLimitExceeded

# Rate limiter setup - uses IP address for identification
limiter = Limiter(key_func=get_remote_address)

# Import implementation functions from server
from server import (
    _query_sota_impl,
    _check_freshness_impl,
    _get_forbidden_impl,
    _compare_models_impl,
    _recent_releases_impl,
    _query_sota_for_hardware_impl,
    _get_model_recommendation_impl,
)

app = FastAPI(
    title="SOTA Tracker API",
    description="""
State-of-the-Art AI model rankings and recommendations.

Prevents suggesting outdated models by providing:
- Current SOTA rankings by category
- Model freshness checking
- Hardware-aware recommendations
- Forbidden/outdated model list

Rate Limits: 60 req/min for standard endpoints, 10 req/min for expensive operations.
    """,
    version="1.0.0",
    docs_url="/docs",
    redoc_url="/redoc"
)

# Register rate limiter with app
app.state.limiter = limiter
app.add_exception_handler(RateLimitExceeded, _rate_limit_exceeded_handler)

# Enable CORS for browser access
# Note: Using allow_origins=["*"] without credentials for public API access
# For production with auth, specify explicit origins and enable credentials
app.add_middleware(
    CORSMiddleware,
    allow_origins=["*"],
    allow_credentials=False,  # Cannot use credentials with wildcard origins
    allow_methods=["GET"],    # Read-only API
    allow_headers=["*"],
)


@app.get("/api/v1/models")
@limiter.limit("60/minute")
def list_models(
    request: Request,
    category: str = Query(..., description="Model category (image_gen, llm_api, llm_local, video, etc.)"),
    open_source_only: bool = Query(True, description="Filter to open-source models only")
):
    """
    Get SOTA models for a category.

    Categories: image_gen, image_edit, video, video2audio, llm_local, llm_api, llm_coding, tts, stt, music, 3d, embeddings
    """
    result = _query_sota_impl(category, open_source_only)
    if "Invalid category" in result:
        raise HTTPException(status_code=status.HTTP_400_BAD_REQUEST, detail=result)
    return {"category": category, "open_source_only": open_source_only, "result": result}


@app.get("/api/v1/models/{model_name}/freshness")
@limiter.limit("60/minute")
def check_model_freshness(request: Request, model_name: str):
    """
    Check if a specific model is current SOTA or outdated.

    Returns CURRENT with rank, or OUTDATED with replacement suggestion.
    """
    result = _check_freshness_impl(model_name)
    return {"model": model_name, "status": result}


@app.get("/api/v1/forbidden")
@limiter.limit("60/minute")
def get_forbidden_models(request: Request):
    """
    Get list of all forbidden/outdated models that should never be suggested.

    These models have been superseded by newer, better alternatives.
    """
    result = _get_forbidden_impl()
    return {"result": result}


@app.get("/api/v1/recent")
@limiter.limit("60/minute")
def recent_releases(
    request: Request,
    days: int = Query(30, ge=1, le=365, description="Number of days to look back"),
    open_source_only: bool = Query(True, description="Filter to open-source models only")
):
    """
    Get models released in the past N days.
    """
    result = _recent_releases_impl(days, open_source_only)
    return {"days": days, "open_source_only": open_source_only, "result": result}


@app.get("/api/v1/compare")
@limiter.limit("10/minute")
def compare_models(
    request: Request,
    model_a: str = Query(..., description="First model name"),
    model_b: str = Query(..., description="Second model name")
):
    """
    Compare two models side-by-side.

    Shows status, category, release date, and recommendation.
    """
    result = _compare_models_impl(model_a, model_b)
    return {"model_a": model_a, "model_b": model_b, "result": result}


@app.get("/api/v1/hardware/models")
@limiter.limit("10/minute")
def hardware_filtered_models(
    request: Request,
    category: str = Query(..., description="Model category"),
    concurrent_vram_gb: int = Query(0, ge=0, description="VRAM already in use"),
    concurrent_workload: Optional[str] = Query(None, description="Named workload (image_gen, video_gen, comfyui, gaming)")
):
    """
    Get SOTA models filtered by hardware capabilities.

    Requires hardware profile to be configured via MCP.
    """
    result = _query_sota_for_hardware_impl(category, concurrent_vram_gb, concurrent_workload)
    return {"category": category, "result": result}


@app.get("/api/v1/recommend")
@limiter.limit("60/minute")
def get_recommendation(
    request: Request,
    task: str = Query(..., description="Task type (chat, code, reasoning, creative, image, video)"),
    concurrent_workload: Optional[str] = Query(None, description="What else is running on GPU")
):
    """
    Get a single best model recommendation for a task.

    Tasks: chat, daily_chat, code, coding, reason, reasoning, creative, creative_writing, max_quality, image, image_gen, video
    """
    result = _get_model_recommendation_impl(task, concurrent_workload)
    return {"task": task, "result": result}


@app.get("/health")
def health_check():
    """Health check endpoint for monitoring."""
    return {"status": "ok", "service": "sota-tracker"}


@app.get("/")
def root():
    """Root endpoint with API info."""
    return {
        "service": "SOTA Tracker API",
        "version": "1.0.0",
        "docs": "/docs",
        "endpoints": {
            "models": "/api/v1/models?category=llm_api",
            "freshness": "/api/v1/models/{model_name}/freshness",
            "forbidden": "/api/v1/forbidden",
            "recent": "/api/v1/recent",
            "compare": "/api/v1/compare?model_a=X&model_b=Y",
            "hardware": "/api/v1/hardware/models?category=llm_local",
            "recommend": "/api/v1/recommend?task=chat"
        }
    }


if __name__ == "__main__":
    import uvicorn
    uvicorn.run(app, host="0.0.0.0", port=8000)
